/**
* \file
* \brief FPGA9001 Multi-Chip Synchronization (MCS) functions
*
* FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
*/
/**
 * Copyright 2019-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.pdf" file in this zip file.
 */

#ifndef _ADI_FPGA9001_MCS_H_
#define _ADI_FPGA9001_MCS_H_

#include <stdbool.h>

#include "adi_fpga9001_types.h"
#include "adi_fpga9001_mcs_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Configure MCS
 *
 * \param[in] fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 * \param[in] mcsCfg     The desired MCS configuration
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Mcs_Configure(adi_fpga9001_Device_t *fpga9001, 
                                   adi_fpga9001_McsCfg_t *mcsCfg);
/**
 * \brief Configure MCS for Tx
 *
 * \param[in] fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Mcs_TxSsiConfigure(adi_fpga9001_Device_t *fpga9001);

/**
* \brief Configure MCS for Rx
*
* \param[in] fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
*
* \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
*/
int32_t adi_fpga9001_Mcs_RxSsiConfigure(adi_fpga9001_Device_t *fpga9001);

/**
 * \brief Initiate a sequence of MCS pulses
 *
 * \param[in] fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Mcs_Start(adi_fpga9001_Device_t *fpga9001);

/**
 * \brief Get Rx MCS To strobe latency in samples
 *
 * \param[in]  fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 * \param[in]  channel    Channel to get MCS to strobe latency
 * \param[out] latency    MCS to strobe latency in samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Mcs_RxMcsToStrobeSampleLatency_Get (adi_fpga9001_Device_t *fpga9001, 
                                                         adi_common_ChannelNumber_e channel, 
                                                         uint32_t *latency);

/**
 * \brief Get MCS done status
 *
 * \param[in]  fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 * \param[out] mcsDone    MCS done status. Set to 1 if the MCS sequence is complete, 0 otherwise.
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Mcs_Done_Get (adi_fpga9001_Device_t *fpga9001, 
                                   bool *mcsDone);

/**
 * \brief Get Rx MCS status
 *
 * \param[in]  fpga9001   Context variable - Pointer to FPGA9001 fpga9001 data structure
 * \param[out] mcsStatus  MCS status
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or failure (non-zero)
 */
int32_t adi_fpga9001_Mcs_RxMcsStatus_Get (adi_fpga9001_Device_t *fpga9001,
                                          adi_fpga9001_McsStatus_t *mcsStatus);

#ifdef __cplusplus
}
#endif

#endif /* _ADI_FPGA9001_MCS_H_ */