/**
 * \file
 * \brief Hardware abstraction layer (HAL) for interacting with the FPGA9001
 *
 * Copyright 2020 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */
#ifndef _ADI_FPGA9001_HAL_H_
#define _ADI_FPGA9001_HAL_H_

#ifdef __KERNEL__
#include <linux/kernel.h>
#else
#include <stdint.h>
#endif

#include "adi_fpga9001_ssi_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \addtogroup fpga9001_hal "FPGA9001 HAL"
 * The FPGA9001 HAL consists of function pointers which must be implemented by the user, such that the FPGA9001 API can
 * call said functions to interact with the chip without having to worry about hardware details.
 * 
 * For an example implementation, see adi_fpga9001_hal_linux_uio.c.
 * @{
 */

/**
 * \brief Performs a read of a single 32-bit register
 *
 * \param[in]  devHalCfg        User-defined context variable
 * \param[in]  addr             Address of the register to be read. Must be on a word boundary
 * \param[out] data             The value read from the register
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
*/
extern int32_t(*adi_fpga9001_hal_Register_Read)(void *devHalCfg, const uint32_t addr, uint32_t *data);

/**
 * \brief Performs a write of a single 32-bit register
 *
 * \param[in] devHalCfg         User-defined context variable
 * \param[in] addr              Address of the register to be written. Must be on a word boundary
 * \param[in] data              The value to write to the register
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t(*adi_fpga9001_hal_Register_Write)(void *devHalCfg, uint32_t addr, uint32_t data);

/**
 * \brief Performs a RAM memory read
 *
 * \param[in]  devHalCfg        User-defined context variable
 * \param[in]  ramAddress       Address to begin the read from
 * \param[out] data             The array of values read from RAM. Must have enough space to store 'length' elements
 * \param[in]  length           The number of words to read starting from ramAddress
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t(*adi_fpga9001_hal_Ram_Read)(void *devhalCfg, 
                                           const uint32_t ramAddress,
                                           uint32_t data[],
                                           uint32_t length);

/**
 * \brief Performs a RAM memory write
 *
 * \param[in] devHalCfg         User-defined context variable
 * \param[in] ramAddress        Address to begin writing to
 * \param[in] data              Array of values to write to RAM
 * \param[in] length            The number of words to write starting at ramAddress
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t(*adi_fpga9001_hal_Ram_Write)(void *devHalCfg, 
                                            const uint32_t ramAddress,
                                            uint32_t data[],
                                            uint32_t length);

/**
 * \brief Set the FPGA SSI type to CMOS or LVDS
 *
 * \param[in] devHalCfg         User-defined context variable
 * \param[in] ssiType           The desired SSI type: CMOS or LVDS
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t(*adi_fpga9001_hal_SsiType_Set)(void *devHalCfg, adi_fpga9001_SsiType_e ssiType);

/**
 * \brief DMA channels within the FPGA9001
 */
typedef enum adi_fpga9001_hal_Dma
{
    ADI_FPGA9001_HAL_DMA_RX1,
    ADI_FPGA9001_HAL_DMA_RX2,
    ADI_FPGA9001_HAL_DMA_TX1,
    ADI_FPGA9001_HAL_DMA_TX2,
    ADI_FPGA9001_HAL_DMA_ORX1,
    ADI_FPGA9001_HAL_DMA_ORX2,
    ADI_FPGA9001_HAL_DMA_SPI
} adi_fpga9001_hal_Dma_e;

/**
 * \brief Supported hardware platforms
 */
typedef enum adi_fpga9001_Platform
{
    ADI_FPGA9001_PLATFORM_UNSUPPORTED,
    ADI_FPGA9001_PLATFORM_ZC706,
    ADI_FPGA9001_PLATFORM_ZCU102
} adi_fpga9001_Platform_e;

/**
 * \brief Get the relative memory address where data for the specified DMA will be stored
 * 
 * \param[in] devHalCfg         User-defined context variable
 * \param[in] dma               The DMA of interest
 * \param[out] addr             The returned address
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t(*adi_fpga9001_hal_DmaRelativeAddress_Get)(void *devHalCfg, adi_fpga9001_hal_Dma_e dma, uint32_t *addr);

/**
 * \brief Get the current FPGA platform. ZC706 and ZCU102 are the current supported platforms
 * 
 * \param[in]  devHalCfg         User-defined context variable
 * \param[out] platform          The curren FPGA platform
 *
 * \returns 0 to indicate success, negative values to indicate error. Specific error codes are defined by the user
 * implementation and are simply returned by FPGA9001 API functions.
 */
extern int32_t (*adi_fpga9001_hal_CurrentPlatform_Get)(void *devHalCfg, adi_fpga9001_Platform_e *platform);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* _ADI_FPGA9001_HAL_H_ */
