/**
 * \file
 * \brief Functions to configure and control the FPGA9001 GPIO pins
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */
/**
 * Copyright 2018 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#ifndef _ADI_FPGA9001_GPIO_H_
#define _ADI_FPGA9001_GPIO_H_

#include "adi_fpga9001_gpio_types.h"
#include "adi_fpga9001.h"
#include "adi_fpga9001_datachain_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Set the operationMode for the pin control
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] operationMode     The operation mode to set
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_OperationMode_Set(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e operationMode);

/**
 * \brief Get the current operationMode
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[out] operationMode    The current operation mode
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_OperationMode_Get(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e *operationMode);

/**
 * \brief Read the values on the GPIO input register
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  group            GPIO group to read: DGPIO, RX0,, RX1, TX0, TX1
 * \param[out] rdData           The current input values
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_Read(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_GpioGroup_e group, uint16_t *rdData);

/**
 * \brief Configure DGPIO for hop and/or RxGain/TxAtten index selection
 *
 * \note The index values of hop table and RxGain/TxAtten are packed together if both features are selected in FH configure.
 *       - hop table index and/or RxGain/TxAtten index selections are handled in board layer and are seamless to this function
 *       - RxGain and Txatten indices share the same GPIO pins.
 *       - If RxGain/TxAtten index selection by gpio feature is selected, then pins selected for RxGain/TxAtten must be contiguous.
 *       - If hop table index by gpio feature is selected, then pins selected for hop table index must be contiguous.
 *       - Pins don't have to be contiguous across hop table index and RxGain/TxAtten index.
 * 
 * \param[in] fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in] trigger          GPIO trigger source
 * \param[in] index            Array containing HOP and/or RxGain/TxAtten indices
 * \param[in] indexLength      Length of the index array       
 * \param[in] gpioPin          Selected GPIO pin
 * \param[in] gpioPinLength    Number of GPIO pins selected
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_HopIndex_GainIndex_Configure(adi_fpga9001_Device_t *fpga9001,
                                                       adi_fpga9001_DmaTrigger_e trigger,
                                                       uint32_t index[],
                                                       uint32_t indexLength,
                                                       adi_fpga9001_GpioPin_e gpioPin[],
                                                       uint32_t gpioPinLength);
/**
 * \brief Configure a single GPIO pin
 * 
 * \param[in] fpga9001       Context variable - Pointer to FPGA9001 device data structure
 * \param[in] gpioPinConfig  The desired GPIO pin configuration
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_pin_Configure(adi_fpga9001_Device_t *fpga9001,
                                        adi_fpga9001_GpioPinCfg_t *gpioPinConfig);

/**
 * \brief Inspect a single GPIO pin
 * 
 * \param[in]  fpga9001      Context variable - Pointer to FPGA9001 device data structure
 * \param[in,out] gpioPinConfig The current GPIO pin configuration
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_pin_Inspect(adi_fpga9001_Device_t *fpga9001,
                                      adi_fpga9001_GpioPinCfg_t *gpioPinConfig);
    
/**
 * \brief Configures the specified FPGA9001 GPIO pin with the desired source mode
 *
 * \param[in] fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in] gpioPinConfig    GPIO pin configurations to write
 * \param[in] length           Number of GPIO pin configurations to write
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_pin_SourceMode_Configure(adi_fpga9001_Device_t *fpga9001,
                                                   adi_fpga9001_GpioPinCfg_t gpioPinConfig[],
                                                   uint32_t length);

/**
 * \brief Inspect the FPGA9001 GPIO pins
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[out] gpioPinConfig    GPIO pin configurations to write
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_pin_SourceMode_Inspect(adi_fpga9001_Device_t *fpga9001,
                                                 adi_fpga9001_GpioPinCfg_t *gpioPinConfig);

/**
 * \brief Configure FPGA to assign GPIO pins as observation GPIO to capture gain index
 *
 * \param[in] fpga9001            Context variable - Pointer to FPGA9001 device data structure
 * \param[in] gainIndexConfig     The desired configuration to assign FPGA GPIO pins as observation GPIO
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_GainIndex_Configure(adi_fpga9001_Device_t *fpga9001,
                                              adi_fpga9001_GpioGainIndexCfg_t *gainIndexConfig);

/**
 * \brief Read observation GPIO data from FPGA capture memory
 *
 * \param[in]  fpga9001                 Context variable - Pointer to FPGA9001 device data structure
 * \param[out] gainIndex                Buffer to store the captured gain index
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_GainIndex_Get(adi_fpga9001_Device_t *fpga9001,
                                        uint32_t *gainIndex);

/**
 * \brief Configure FPGA to assign GPIO pins as trigger source
 *
 * \param[in] fpga9001            Context variable - Pointer to FPGA9001 device data structure
 * \param[in] gainIndexConfig     The desired configuration to assign FPGA GPIO pins as trigger source
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_TriggerSource_Configure(adi_fpga9001_Device_t *fpga9001,
                                                  adi_fpga9001_GpioTriggerCfg_t *triggerCfg);

/**
 * \brief Configure SMA pin 1 and 2
 * 
 * \param[in] fpga9001    Context variable - Pointer to FPGA9001 device data structure
 * \param[in] smaConfig   The desired SMA pin configuration
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_smapin_Configure(adi_fpga9001_Device_t *fpga9001,
                                           adi_fpga9001_GpioSmaCfg_t *smaConfig);

/**
 * \brief Inspect SMA pin 1 and 2
 * 
 * \param[in]  fpga9001     Context variable - Pointer to FPGA9001 device data structure
 * \param[out] smaConfig    The current direction of SMA pins 1 and 2
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_gpio_smapin_Inspect(adi_fpga9001_Device_t *fpga9001,
                                         adi_fpga9001_GpioSmaCfg_t *smaConfig);

#ifdef __cplusplus
}
#endif

#endif /* _ADI_FPGA9001_GPIO_H_ */