/**
 * \file
 *
 * \brief Functions to configure the FPGA9001 Data Chains
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2015 - 2018 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#ifndef _ADI_FPGA9001_DATACHAIN_H_
#define _ADI_FPGA9001_DATACHAIN_H_

#include "adi_common_types.h"

#include "adi_fpga9001_datachain_types.h"
#include "adi_fpga9001_types.h"
#include "adi_fpga9001_fh_types.h"
#include "adi_fpga9001_tdd_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * The adi_fpga9001_DataChain_Data_Get_XXX and adi_fpga9001_DataChain_Data_Set_XXX functions use local buffers to pack
 * or unpack data in the correct format. 
 * If ADI_FPGA9001_USE_DYNAMIC_MEMORY is defined, these buffers will be allocated dynamically on the heap using calloc
 * If undefined, the buffers will be allocated on the stack. The size of the buffers (and thus maximum capture size)
 * is controlled by ADI_FPGA9001_MAX_CAPTURE_BYTES. The default value is the maximum size allowed by the FPGA
 */
//#define ADI_FPGA9001_USE_DYNAMIC_MEMORY 1
#define ADI_FPGA9001_MAX_CAPTURE_BYTES 0x7FFFFF

/**
 * \brief Reads the DMA configuration for a single DMA module
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] dmaCfg           The DMA configuration that was read
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Dma_Inspect(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Reads the DMA status for a single DMA module
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] status           The DMA status that was read
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_DmaStatus_Get(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             adi_common_ChannelNumber_e channel,
                                             adi_fpga9001_DmaStatus_t *status);

/**
 * \brief Configure the data chains
 * If port is Rx: initiate capture, and wait for capture to complete
 * If port is Tx: start transmitting in continuous mode
 *
 * \note If port is Rx: This function configures the specified data chains, starts an Rx capture, and blocks until the captures
 * complete or the timeout is reached. Essentially, this function fills the RAM. Use the DataGet_X utility
 * functions to read data from RAM in the appropriate format.
 * 
 * \note If port is Tx: This function does NOT set the data to be transmitted. For that, use one of the DataSet_X utility functions
 * to write data to RAM in the appropriate format.
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] dmaCfg            The DMA configuration to configure the data chain
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_PerformTransfer(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Configure the data chains
 * If port is Rx: initiate capture and return without waiting for capture to complete
 * If port is Tx: start transmitting in continuous mode
 *
 * \note If port is Rx: This function configures the specified data chains, starts an Rx capture, and 
 * returns without waiting for capture to complete. In order to perform a successful capture, 
 * 'adi_fpga9001_DataChain_PerformTransfer_Finish' may be called after this call and will return when transfer is complete.
 * Essentially, this function fills the RAM. Use the DataGet_X utility functions to read data from RAM in the appropriate format.
 * 
 * \note If port is Tx: This function does NOT set the data to be transmitted. For that, use one of the DataSet_X utility functions
 * to write data to RAM in the appropriate format.
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] dmaCfg            The DMA configuration to configure the data chain
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_PerformTransfer_Start(adi_fpga9001_Device_t *fpga9001,
                                                     adi_common_Port_e port,
                                                     adi_common_ChannelNumber_e channel,
                                                     adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Wait for the initiated capture in 'adi_fpga9001_DataChain_PerformTransfer_Start()' function call to complete
 * If port is Rx: This function must be called after 'adi_fpga9001_DataChain_PerformTransfer_Start()' for capture to complete
 * If port is Tx: Since transmitting is in continuous mode, this function takes no action
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] dmaCfg            The DMA configuration for which to wait for the capture to complete
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_PerformTransfer_Finish(adi_fpga9001_Device_t *fpga9001,
                                                      adi_common_Port_e port,
                                                      adi_common_ChannelNumber_e channel,
                                                      adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Configure and start DMA for FH table transfer
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] dmaCfg            The DMA configuration to configure the SPI
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_HopTable_Transfer_Start(adi_fpga9001_Device_t *fpga9001,
                                                       adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Stop DMA for FH table transfer
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] timeout_ms        Stop DMA or until Timeout reached
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_HopTable_Transfer_Stop(adi_fpga9001_Device_t *fpga9001,
                                                      uint32_t timeout_ms);
    
/**
 * \brief Configure MSPI in DMA mode for FH table transfer to ARM memory
 *
 * \param[in] fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in] trigger          The trigger source to use
 * \param[in] bytesPerTable    Number of bytes to be transferred through SPI DMA
 * \param[in] framesPerTable   Number of frames to be transferred through SPI DMA
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Mspi_Dma_Configure(adi_fpga9001_Device_t *fpga9001,
                                                  adi_fpga9001_DmaTrigger_e trigger,
                                                  uint32_t bytesPerTable,
                                                  uint32_t framesPerTable);

/**
 * \brief Configure MSPI in FIFO mode
 *
 * \param[in] fpga9001   Context variable - Pointer to FPGA9001 device data structure
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Mspi_Fifo_Configure(adi_fpga9001_Device_t *fpga9001);

/**
 * \brief Configure the data chains for synchronous transfer between two channels
 * If port is Rx: Synchronously start a capture on two channels, and wait for both channels
 *                capture to complete.
 * If port is Tx: Synchronously starts transmission on two channels in continuous mode.
 *
 * \note If port is Rx: This function configures the data chains for channel 1 and channel 2, 
 *                      synchronously starts an Rx capture on each channel, and blocks until the captures complete 
 *                      or the timeout is reached. 
 *                      Essentially, this function fills the RAM for channel 1 and channel 2. 
 *                      Use the DataGet_X utility functions to read data from RAM in the appropriate format.
 * 
 * \note If port is Tx: This function does NOT set the data to be transmitted. For that, use one of the DataSet_X utility functions
 * to write data to RAM in the appropriate format.
 * 
 * \note The trigger in dmaCfg is overwritten with ADI_FPGA9001_DMA_TRIGGER_SYNC.
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port to initiate the transfer
 * \param[in] dmaCfg            The DMA configuration to configure the data chain
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_PerformSynchronousTransfer(adi_fpga9001_Device_t *fpga9001,
                                                          adi_common_Port_e port,
                                                          adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Configure dma without blocking, use status to wait subsequently
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port to initiate the transfer
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] dmaCfg            The DMA configuration to configure the data chain
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Dma_Configure(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             adi_common_ChannelNumber_e channel,
                                             adi_fpga9001_DmaCfg_t *dmaCfg);

/**
 * \brief Set the enabledness of DMA_ENABLE and/or DMA_TRIGGER of Channel1 as master for Channel2
 *                                      This ensures the DMA_ENABLE and/or DMA_TRIGGER signals have identical timing
 *                                      on both channels
 * \param[in] fpga9001                  Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port                      The port to set the enabledness
 * \param[in] enableMasterToEnable      Set to TRUE to enable Channel1 DMA_ENABLE as master for Channel2
 * \param[in] enableMasterToTrigger     Set to TRUE to enable Channel1 DMA_TRIGGER as master for Channel2
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Dma_Master_Set(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             bool enableMasterToEnable,
                                             bool enableMasterToTrigger);

/**
 * \brief Get the enabledness of DMA_ENABLE and DMA_TRIGGER of Channel1 as master for Channel2
 * \param[in] fpga9001                  Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port                      The port to get the enabledness 
 * \param[out] enableMasterToEnable      Returns as TRUE if Channel1 DMA_ENABLE is set as master for Channel 2
 * \param[out] enableMasterToTrigger     Returns as TRUE if Channel1 DMA_TRIGGER is set as master for Channel 2
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Dma_Master_Get(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             bool *enableMasterToEnable,
                                             bool *enableMasterToTrigger);

/**
 * \brief Status of dma with blocking, use status above for non-blocking (?)
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port to initiate the transfer
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] dmaCfg            The DMA configuration to configure the data chain
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Dma_Wait(adi_fpga9001_Device_t *fpga9001,
                                        adi_common_Port_e port,
                                        adi_common_ChannelNumber_e channel,
                                        adi_fpga9001_DmaCfg_t *dmaCfg);
    
/**
 * \brief Read data from FPGA capture memory for 22I22Q mode, gain change, and gain index
 *
 * In 22I22Q, gain change, gain index mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{22I0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x4 | 32'{22Q0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x8 | 32'{22I1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  0xC | 32'{22Q1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 22-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 22-bit I values
 * \param[out] qData            The returned 22-bit Q values
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC1
 *                        2 | GC2
 *                        3 | GC3
 *                      ... | ...
 * \param[out] gainIndexData   The returned 8-bit gain index values 
 *   gainIndexData[index] | Value
 *   -------------------------------
 *                        0 | GI0
 *                        1 | GI1
 *                        2 | GI2
 *                        3 | GI3
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, gainChangeData, and gainIndexData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_22I22Q_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    uint32_t iData[],
                                                                    uint32_t qData[],
                                                                    uint8_t gainChangeData[],
                                                                    uint8_t gainIndexData[],
                                                                    uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory for 22I22Q mode, gain change, and gain index
 *
 * In 22I22Q, gain change, gain index mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{22I0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x4 | 32'{22Q0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x8 | 32'{22I1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  0xC | 32'{22Q1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 22-bit arrays for convenient indexing and converts from two's
 * complement to signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 22-bit I values
 * \param[out] qData            The returned 22-bit Q values
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC1
 *                        2 | GC2
 *                        3 | GC3
 *                      ... | ...
 * \param[out] gainIndexData   The returned 8-bit gain index values 
 *   gainIndexData[index] | Value
 *   -------------------------------
 *                        0 | GI0
 *                        1 | GI1
 *                        2 | GI2
 *                        3 | GI3
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, gainChangeData, and gainIndexData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_22I22QSigned_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                          adi_common_Port_e port,
                                                                          adi_common_ChannelNumber_e channel,
                                                                          int32_t iData[],
                                                                          int32_t qData[],
                                                                          uint8_t gainChangeData[],
                                                                          uint8_t gainIndexData[],
                                                                          uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory for 22I22Q mode, gain change, and gain index
 *
 * In 22I22Q, gain change, gain index mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{22I0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x4 | 32'{22Q0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x8 | 32'{22I1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  0xC | 32'{22Q1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 22-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC0
 *                        2 | GC1
 *                        3 | GC1
 *                      ... | ...
 * \param[out] gainIndexData   The returned 8-bit gain index values 
 *   gainIndexData[index] | Value
 *   -------------------------------
 *                        0 | GI0
 *                        1 | GI0
 *                        2 | GI1
 *                        3 | GI1
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, gainChangeData, and gainIndexData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleaved_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                               adi_common_Port_e port,
                                                                               adi_common_ChannelNumber_e channel,
                                                                               uint32_t iqData[],
                                                                               uint8_t gainChangeData[],
                                                                               uint8_t gainIndexData[],
                                                                               uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory for 22I22Q mode, gain change, and gain index
 *
 * In 22I22Q, gain change, gain index mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{22I0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x4 | 32'{22Q0[21:0], 1'b0, GC0[0], GI0[7:0]}
 *                  0x8 | 32'{22I1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  0xC | 32'{22Q1[21:0], 1'b0, GC1[0], GI1[7:0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 22-bit arrays for convenient indexing and converts from two's
 * complement to signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC0
 *                        2 | GC1
 *                        3 | GC1
 *                      ... | ...
 * \param[out] gainIndexData   The returned 8-bit gain index values 
 *   gainIndexData[index] | Value
 *   -------------------------------
 *                        0 | GI0
 *                        1 | GI0
 *                        2 | GI1
 *                        3 | GI1
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, gainChangeData, and gainIndexData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleavedSigned_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                               adi_common_Port_e port,
                                                                               adi_common_ChannelNumber_e channel,
                                                                               int32_t iqData[],
                                                                               uint8_t gainChangeData[],
                                                                               uint8_t gainIndexData[],
                                                                               uint32_t length);

/**
 * \brief Read data from FPGA capture memory for 16I16Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 16-bit I values
 * \param[out] qData            The returned 16-bit Q values
 * \param[in]  length           The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16I16Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t iData[],
                                               uint16_t qData[],
                                               uint32_t length);

/**
 * \brief Read data from FPGA capture memory for 16I16Q mode, converting from two's complement to signed
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing and converts from two's
 * complement to signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 16-bit signed I values
 * \param[out] qData            The returned 16-bit signed Q values
 * \param[in]  length           The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16I16QSigned(adi_fpga9001_Device_t *fpga9001,
                                                     adi_common_Port_e port,
                                                     adi_common_ChannelNumber_e channel,
                                                     int16_t iData[],
                                                     int16_t qData[],
                                                     uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory in interleaved format for 16I16Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into a single interleaved 16-bit array
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t iqData[],
                                                        uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory in interleaved format for 16I16Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into a single interleaved 16-bit array and converts from two's complement to
 * signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleavedSigned(adi_fpga9001_Device_t *fpga9001,
                                                              adi_common_Port_e port,
                                                              adi_common_ChannelNumber_e channel,
                                                              int16_t iqData[],
                                                              uint32_t length);

/**
 * \brief Read data from FPGA capture memory in for 16I Only mode
 *
 * In 16I Only mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I1,16I0
 *                  0x4 | 16I3,16I2
 *                  ... | ...
 *
 * This function unpacks the I data into a 16-bit array for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData
 * \parblock
 * I data in the format:
 *     iData[index] | Value
 *     ---------------------
 *                0 | I0
 *                1 | I1
 *                2 | I2
 *                3 | I3
 *              ... | ...
 * \endparblock
 * \param[in]  length           The length of the iData array; i.e., the number of I samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16I(adi_fpga9001_Device_t *fpga9001,
                                            adi_common_Port_e port,
                                            adi_common_ChannelNumber_e channel,
                                            uint16_t iData[],
                                            uint32_t length);

/**
 * \brief Read data from FPGA capture memory in for 16I Only mode
 *
 * In 16I Only mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I1,16I0
 *                  0x4 | 16I3,16I2
 *                  ... | ...
 *
 * This function unpacks the I data into a 16-bit array for convenient indexing and converts from two's complement to
 * signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData
 * \parblock
 * I data in the format:
 *     iData[index] | Value
 *     ---------------------
 *                0 | I0
 *                1 | I1
 *                2 | I2
 *                3 | I3
 *              ... | ...
 * \endparblock
 * \param[in]  length           The length of the iData array; i.e., the number of I samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_16ISigned(adi_fpga9001_Device_t *fpga9001,
                                                  adi_common_Port_e port,
                                                  adi_common_ChannelNumber_e channel,
                                                  int16_t iData[],
                                                  uint32_t length);

/**
 * \brief Read data from FPGA capture memory in interleaved format for 12I12Q mode
 *
 * In 12I12Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 12I0,12Q0
 *                  0x4 | 12I1,12Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into a single interleaved 16-bit array
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_12IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t iqData[],
                                                        uint32_t length);

/**
 * \brief Read data from FPGA capture memory for 12I12Q mode
 *
 * In 12I12Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 12I0,12Q0
 *                  0x4 | 12I1,12Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 12-bit I values
 * \param[out] qData            The returned 12-bit Q values
 * \param[in]  length           The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_12I12Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t iData[],
                                               uint16_t qData[],
                                               uint32_t length);

/**
 * \brief Read data from FPGA capture memory in interleaved format for 12I12Q mode
 *
 * In 12I12Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 12I0,12Q0
 *                  0x4 | 12I1,12Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into a single interleaved 16-bit array and sign extends the values
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_12IQInterleavedSigned(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        int16_t iqData[],
                                                        uint32_t length);

/**
 * \brief Read data from FPGA capture memory for 12I12Q mode
 *
 * In 12I12Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 12I0,12Q0
 *                  0x4 | 12I1,12Q1
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing and sign extends the values
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 12-bit I values
 * \param[out] qData            The returned 12-bit Q values
 * \param[in]  length           The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_12I12QSigned(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               int16_t iData[],
                                               int16_t qData[],
                                               uint32_t length);

/**
 * \brief Read data from FPGA capture memory for 15I15Q mode and gain change
 *
 * In 15I15Q and gain change mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{15I0[14:0], GC0[0], 15Q0[14:0], GC0[0]}
 *                  0x4 | 32'{15I1[14:0], GC1[0], 15Q1[14:0], GC1[0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 15-bit I values
 * \param[out] qData            The returned 15-bit Q values
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC1
 *                        2 | GC2
 *                        3 | GC3
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, and gainChangeData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_15I15Q_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                            adi_common_Port_e port,
                                                            adi_common_ChannelNumber_e channel,
                                                            uint16_t iData[],
                                                            uint16_t qData[],
                                                            uint8_t gainChangeData[],
                                                            uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory for 15I15Q mode and gain change
 *
 * In 15I15Q and gain change mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{15I0[14:0], GC0[0], 15Q0[14:0], GC0[0]}
 *                  0x4 | 32'{15I1[14:0], GC1[0], 15Q1[14:0], GC1[0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing and converts from two's
 * complement to signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iData            The returned 15-bit I values
 * \param[out] qData            The returned 15-bit Q values
 * \param[out] gainChangeData   The returned 1-bit gain change values 
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC1
 *                        2 | GC2
 *                        3 | GC3
 *                      ... | ...
 * \param[in]  length           The length of the iData, qData, and gainChangeData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_15I15QSigned_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                adi_common_Port_e port,
                                                                adi_common_ChannelNumber_e channel,
                                                                int16_t iData[],
                                                                int16_t qData[],
                                                                uint8_t gainChangeData[],
                                                                uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory in interleaved format for 15I15Q mode and gain change
 *
 * In 15I15Q and gain change mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{15I0[14:0], GC0[0], 15Q0[14:0], GC0[0]}
 *                  0x4 | 32'{15I1[14:0], GC1[0], 15Q1[14:0], GC1[0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 *                
 * \param[out] gainChangeData   The returned 1-bit gain change values
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC0
 *                        2 | GC1
 *                        3 | GC1
 *                      ... | ...
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleaved_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                        adi_common_Port_e port,
                                                                        adi_common_ChannelNumber_e channel,
                                                                        uint16_t iqData[],
                                                                        uint8_t gainChangeData[],
                                                                        uint32_t length);
    
/**
 * \brief Read data from FPGA capture memory in interleaved format for 15I15Q mode and gain change
 *
 * In 15I15Q and gain change mode, data is packed into RAM in the format:
 *     Address (offset) | Data    
 *     -------------------------------------------
 *                  0x0 | 32'{15I0[14:0], GC0[0], 15Q0[14:0], GC0[0]}
 *                  0x4 | 32'{15I1[14:0], GC1[0], 15Q1[14:0], GC1[0]}
 *                  ... | ...
 *
 * This function unpacks the I and Q data into separate 16-bit arrays for convenient indexing and converts from two's
 * complement to signed
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 *                
 * \param[out] gainChangeData   The returned 1-bit gain change values
 * Gain change data format:
 *   gainControlData[index] | Value
 *   -------------------------------
 *                        0 | GC0
 *                        1 | GC0
 *                        2 | GC1
 *                        3 | GC1
 *                      ... | ...
 * \param[in]  length           The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleavedSigned_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                           adi_common_Port_e port,
                                                                           adi_common_ChannelNumber_e channel,
                                                                           int16_t iqData[],
                                                                           uint8_t gainChangeData[],
                                                                           uint32_t length);

/**
 * \brief Read data from FPGA capture memory in for 8-Bit Symbol mode
 *
 * In 8-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 8S3,8S2,8S1,8S0
 *                  0x4 | 8S7,8S6,8S5,8S4
 *                  ... | ...
 *
 * This function unpacks the data into an 8-bit array for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] symbols
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in]  length           The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_8S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t symbols[],
                                           uint32_t length);

/**
 * \brief Read gain index from FPGA capture memory in for 8-Bit Symbol mode
 *
 * In 8-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 8S3,8S2,8S1,8S0
 *                  0x4 | 8S7,8S6,8S5,8S4
 *                  ... | ...
 *
 * This function unpacks the data into an 8-bit array for convenient indexing with MSB set to '1' always for gain index and 
 * extracts gain index change bits in a separate array
 *
 * \param[in]  fpga9001        Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  channel         The channel of the specified port to configure
 * \param[out] gainChangeBit   Gain index change indicator; Toggles from 0 -> 1 or 1 -> 0 when there is a change in gain index
 * \param[out] symbols         Current gain index read
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in]  length           The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_GainIndex_Get_8S(adi_fpga9001_Device_t *fpga9001,
                                                adi_common_ChannelNumber_e channel,
                                                uint8_t gainChangeBit[],
                                                uint8_t symbols[],
                                                uint32_t length);
/**
 * \brief Read data from FPGA capture memory in for 2-Bit Symbol mode
 *
 * In 2-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 6X,2S3,6X,2S2,6X,2S1,6X,2S0
 *                  0x4 | 6X,2S7,6X,2S6,6X,2S5,6X,2S4
 *                  ... | ...
 *
 * This function unpacks the data into an 8-bit array containing 2 bit symbols for convenient indexing
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  port             The port that the channel refers to
 * \param[in]  channel          The channel of the specified port to configure
 * \param[out] symbols
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in]  length           The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Get_2S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t symbols[],
                                           uint32_t length);

/**
 * \brief Write data to FPGA transmit memory for 16I16Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function takes separate 16-bit arrays of I and Q data and packs them into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] iData             Array of 16-bit I values to be packed into RAM
 * \param[in] qData             Array of 16-bit Q values to be packed into RAM
 * \param[in] length            The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_16I16Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t iData[],
                                               uint16_t qData[],
                                               uint32_t length);

/**
 * \brief Write data to FPGA transmit memory from interleaved format for 16I16Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function takes a single 16-bit array of interleaved I and Q data and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in] length            The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_16IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t iqData[],
                                                        uint32_t length);

/**
 * \brief Write data to FPGA transmit memory for 16I Only mode
 *
 * In 16I Only mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I1,16I0
 *                  0x4 | 16I3,16I2
 *                  ... | ...
 *
 * This function takes a 16-bit array of I data and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] iData
 * \parblock
 * I data in the format:
 *     iData[index] | Value
 *     ---------------------
 *                0 | I0
 *                1 | I1
 *                2 | I2
 *                3 | I3
 *              ... | ...
 * \endparblock
 * \param[in] length            The length of the iData array; i.e., the number of I samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_16I(adi_fpga9001_Device_t *fpga9001,
                                            adi_common_Port_e port,
                                            adi_common_ChannelNumber_e channel,
                                            uint16_t iData[],
                                            uint32_t length);



/**
 * \brief Write data to FPGA transmit memory from interleaved format for 12I12Q mode
 *
 * In 12I12Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function takes a single 16-bit array of interleaved I and Q data and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] iqData
 * \parblock
 * Interleaved I and Q data in the format:
 *     iqData[index] | Value
 *     ---------------------
 *                 0 | I0
 *                 1 | Q0
 *                 2 | I1
 *                 3 | Q1
 *               ... | ...
 * \endparblock
 * \param[in] length            The length of the iqData array; i.e., twice the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_12IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t iqData[],
                                                        uint32_t length);


/**
 * \brief Write data to FPGA transmit memory for 12I12Q mode
 *
 * In 16I16Q mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 16I0,16Q0
 *                  0x4 | 16I1,16Q1
 *                  ... | ...
 *
 * This function takes separate 16-bit arrays of I and Q data and packs them into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] iData             Array of 16-bit I values to be packed into RAM
 * \param[in] qData             Array of 16-bit Q values to be packed into RAM
 * \param[in] length            The length of the iData and qData arrays; i.e., the number of IQ samples
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_12I12Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t iData[],
                                               uint16_t qData[],
                                               uint32_t length);

/**
 * \brief Write data to FPGA transmit memory for 8-Bit Symbol mode
 *
 * In 8-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 8S3,8S2,8S1,8S0
 *                  0x4 | 8S7,8S6,8S5,8S4
 *                  ... | ...
 *
 * This function takes an 8-bit array of symbols and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] symbols
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in] length            The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_8S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t symbols[],
                                           uint32_t length);

/**
 * \brief Write data to FPGA transmit memory for 2-Bit Symbol mode
 *
 * In 2-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 6X,2S3,6X,2S2,6X,2S1,6X,2S0
 *                  0x4 | 6X,2S7,6X,2S6,6X,2S5,6X,2S4
 *                  ... | ...
 *
 * This function takes an 8-bit array of 2-bit symbols and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] port              The port that the channel refers to
 * \param[in] channel           The channel of the specified port to configure
 * \param[in] symbols
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in] length            The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_Data_Set_2S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t symbols[],
                                           uint32_t length);

/**
 * \brief Write FH table to FPGA transmit memory in 8-bit symbol mode
 *
 * In 8-Bit Symbol mode, data is packed into RAM in the format:
 *     Address (offset) | Data
 *     ----------------------------
 *                  0x0 | 8S3,8S2,8S1,8S0
 *                  0x4 | 8S7,8S6,8S5,8S4
 *                  ... | ...
 *
 * This function takes an 8-bit array of symbols and packs it into RAM appropriately
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] spiPackedFhTable  SPI packed FH table
 * \parblock
 * Symbol data in the format:
 *     symbols[index] | Value
 *     ---------------------
 *                  0 | S0
 *                  1 | S1
 *                  2 | S2
 *                  3 | S3
 *                ... | ...
 * \endparblock
 * \param[in] byteCount         The length of the symbols array
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_DataChain_HopTable_Set_8S(adi_fpga9001_Device_t *fpga9001,
                                               uint8_t spiPackedFhTable[],
                                               uint32_t byteCount);

#ifdef __cplusplus
}
#endif

#endif /* _ADI_FPGA9001_DATACHAIN_H_ */
