/**
* \file
* \brief Contains internal use only Frequency Hopping engineering functions
*
* ADRV9001 API Version: $ADI_ADRV9001_API_VERSION$
*/

/**
* Copyright 2020 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/

#include "adi_adrv9001_internal_fh.h"
#include "adi_adrv9001_arm.h"
#include "adi_adrv9001_error.h"
#include "adi_adrv9001_radio.h"
#include "adrv9001_arm.h"
#include "adrv9001_arm_macros.h"
#include "adi_adrv9001_internal_utilities_types.h"

#define CALIBRATION_INFO_TABLE_MAX_NUM_BYTES 552

static __maybe_unused int32_t adi_adrv9001_internal_fh_Calibration_Config_Validate(adi_adrv9001_Device_t *adrv9001, 
                                                                    adi_adrv9001_FhCalibrationCfg_t *fhCalibrationCfg,
                                                                    uint32_t addr)
{
    int32_t frequencyIndex = 0;
    /* Check for NULL pointer */
    ADI_NULL_PTR_RETURN(&adrv9001->common, fhCalibrationCfg);

    if ((addr < ADRV9001_ADDR_ARM_START_DATA) || (addr > ADRV9001_ADDR_ARM_END_DATA))
    {
        ADI_ERROR_REPORT(&adrv9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_RESET_FULL,
            adrv9001,
            "Invalid FH Hop Table Address");

        ADI_API_RETURN(adrv9001);
    }

    /* Validate size */
    ADI_RANGE_CHECK(adrv9001, fhCalibrationCfg->numBands, 1, ADI_ADRV9001_FH_MAX_NUM_BANDS);

    /* Validate frequencies */
    for (frequencyIndex = 0; frequencyIndex < fhCalibrationCfg->numBands; frequencyIndex++)
    {
        ADI_RANGE_CHECK_X(adrv9001, fhCalibrationCfg->bandFreqList[frequencyIndex], 
                          ADI_ADRV9001_FH_MIN_CARRIER_FREQUENCY_HZ, 
                          ADI_ADRV9001_FH_MAX_CARRIER_FREQUENCY_HZ, "%llu");
    }
    /* TODO JP: Add validation for numGains */
    ADI_API_RETURN(adrv9001);
}        

static __maybe_unused int32_t adi_adrv9001_internal_fh_CalibrationInfo_Inspect_Validate(adi_adrv9001_Device_t *adrv9001, 
                                                                         adi_adrv9001_FhCalibrationCfg_t *fhCalibrationCfg,
                                                                         uint32_t addr)
{                                                                
    /* Check for NULL pointer */
    ADI_NULL_PTR_RETURN(&adrv9001->common, fhCalibrationCfg);

    if ((addr < ADRV9001_ADDR_ARM_START_DATA) || (addr > ADRV9001_ADDR_ARM_END_DATA))
    {
        ADI_ERROR_REPORT(&adrv9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_RESET_FULL,
            adrv9001,
            "Invalid FH Hop Table Address");
    }

    ADI_API_RETURN(adrv9001);
}                  

int32_t adi_adrv9001_internal_fh_Calibration_Config(adi_adrv9001_Device_t *adrv9001, 
                                                    adi_adrv9001_FhCalibrationCfg_t *fhCalibrationCfg)
{
    uint8_t  armData[CALIBRATION_INFO_TABLE_MAX_NUM_BYTES] = { 0 };
    uint32_t offset = 0;
    uint32_t index = 0;
    /* Table must be written directly to ARM memory, instead of SET buffer. Get pointer to hop table */
    uint8_t  _pFhPreCalTableBlock[4u];
    uint32_t _pFhPreCalTable;
    ADI_EXPECT(adi_adrv9001_arm_Memory_Read, adrv9001, ADRV9001_ADDR_P_FH_PRECAL_TABLE, _pFhPreCalTableBlock, sizeof(_pFhPreCalTableBlock), false);
    adrv9001_ParseFourBytes(&offset, _pFhPreCalTableBlock, &_pFhPreCalTable);

    /* Validate params */
    ADI_PERFORM_VALIDATION(adi_adrv9001_internal_fh_Calibration_Config_Validate, adrv9001, fhCalibrationCfg, _pFhPreCalTable);

    offset = 0; // Reset offset
    armData[offset++] = fhCalibrationCfg->numBands;
    armData[offset++] = fhCalibrationCfg->numGains;
    offset += 6; // Padding
    /* Load precalibration frequencies */
    for(index = 0 ; index < ADI_ADRV9001_FH_MAX_NUM_BANDS ; index++)
    {
        adrv9001_LoadEightBytes(&offset, armData, fhCalibrationCfg->bandFreqList[index]);
    }
    /* Load band mask list */
    for (index = 0; index < ADI_ADRV9001_FH_NUM_FREQ_DEPENDENT_INIT_CALS; index++)
    {
        adrv9001_LoadEightBytes(&offset, armData, fhCalibrationCfg->bandMaskList[index]);
    }

    /* Write to ARM memory */
    ADI_EXPECT(adi_adrv9001_arm_Memory_Write, adrv9001, _pFhPreCalTable, armData, sizeof(armData), ADI_ADRV9001_ARM_SINGLE_SPI_WRITE_MODE_STANDARD_BYTES_4);
    /* Don't need a SET command for this */
    ADI_API_RETURN(adrv9001);
}

int32_t adi_adrv9001_internal_fh_Calibration_Config_Inspect(adi_adrv9001_Device_t *adrv9001, 
                                                            adi_adrv9001_FhCalibrationCfg_t *fhCalibrationCfg)
{
    uint8_t armData[CALIBRATION_INFO_TABLE_MAX_NUM_BYTES]  = { 0 };
    uint32_t offset = 0;
    uint32_t index = 0;

    /* Read config data from ARM memory */
    uint8_t  _pFhPreCalTableBlock[4u];
    uint32_t _pFhPreCalTable;
    ADI_EXPECT(adi_adrv9001_arm_Memory_Read, adrv9001, ADRV9001_ADDR_P_FH_PRECAL_TABLE, _pFhPreCalTableBlock, sizeof(_pFhPreCalTableBlock), false);
    adrv9001_ParseFourBytes(&offset, _pFhPreCalTableBlock, &_pFhPreCalTable);

    /* Validate params */
    ADI_PERFORM_VALIDATION(adi_adrv9001_internal_fh_CalibrationInfo_Inspect_Validate, adrv9001, fhCalibrationCfg, _pFhPreCalTable);

    offset = 0;  // Reset offset
    ADI_EXPECT(adi_adrv9001_arm_Memory_Read, adrv9001, _pFhPreCalTable, armData, sizeof(armData), false);
    fhCalibrationCfg->numBands = armData[offset++];
    fhCalibrationCfg->numGains = armData[offset++];
    offset += 6u; // Padding 
    /* parse precalibration frequencies */
    for(index = 0 ; index < ADI_ADRV9001_FH_MAX_NUM_BANDS ; index++)
    {
        adrv9001_ParseEightBytes(&offset, armData, &fhCalibrationCfg->bandFreqList[index]);
    }
    /* Load band mask list */
    for (index = 0; index < ADI_ADRV9001_FH_NUM_FREQ_DEPENDENT_INIT_CALS; index++)
    {
        adrv9001_ParseEightBytes(&offset, armData, &fhCalibrationCfg->bandMaskList[index]);
    }

    ADI_API_RETURN(adrv9001);
}