/**
 * \file
 * \brief Contains Internal Frequency Hopping data types
 *
 * ADRV9001 API Version: $ADI_ADRV9001_API_VERSION$
 */

 /**
 * Copyright 2020 Analog Devices Inc.
 * Released under the ADRV9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#ifndef _ADI_ADRV9001_INTERNAL_FH_TYPES_H_
#define _ADI_ADRV9001_INTERNAL_FH_TYPES_H_

#ifdef __cplusplus
extern "C" {
#endif

#ifdef __KERNEL__
#include <linux/kernel.h>
#else
#include <stdint.h>
#endif

/**
 * \brief Max number of frequencies for precalibrations
 *
 * Results from each precalibration are used for a band of frequencies during hopping operation.
 */

#define ADI_ADRV9001_FH_MAX_NUM_BANDS 64u

/**
 * \brief Max number of frequency dependent initial calibrations
 */
#define ADI_ADRV9001_FH_NUM_FREQ_DEPENDENT_INIT_CALS 4u

/**
 * \brief Settings for frequency hopping precalibration
 *
 *  Frequency hopping calibration info.
 *  In Frequency Hopping, each algorithm calibrates over a set of
 *  frequency bands, which are determined through characterisation. During frequency 
 *  hopping operation, the carrier frequencies, which the device is hopping on, will
 *  be within these bands. 
 * 
 */
typedef struct {
    /*! Number of frequencies for precalibration, 1 to ADI_ADRV9001_FH_MAX_NUM_BANDS. */
    uint8_t     numBands;
    /*! Number of gain levels for precalibration. TODO JP: Determine range of gain levels*/
    uint8_t     numGains;
    /*! Array of the frequencies for precalibration in Hz. One for every frequency band, and must be in order. */
    uint64_t    bandFreqList [ADI_ADRV9001_FH_MAX_NUM_BANDS];
    /*! For each initcal, shows which bands to calibrate.  Each element gives the mask for a
        specific algorithm, and each bit of the mask indicates if the corresponding frequency
        shall be calibrated (1), or skipped (0).  If skipped, the algorithm will substitute
        a suitable value, usually from the previous frequency. */
    uint64_t    bandMaskList [ADI_ADRV9001_FH_NUM_FREQ_DEPENDENT_INIT_CALS];
} adi_adrv9001_FhCalibrationCfg_t;

#ifdef __cplusplus
}
#endif

#endif /* _ADI_ADRV9001_FH_TYPES_H_ */
