/**
* \file
* \brief API for interacting with the ADM1293 digital power monitor
*
* Copyright 2020 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/

#include "adi_adm1293.h"
#include "adi_adm1293_i2c.h"
#include "adi_common_error.h"

int32_t adi_adm1293_Configure(adi_adm1293_Device_t *adm1293, adi_adm1293_config_t *config)
{
    uint8_t wrData[3] = { 0xD4 };
    
    ADI_NULL_DEVICE_PTR_RETURN(adm1293);
    ADI_NULL_PTR_RETURN(adm1293, config);
    
    wrData[1] = (config->current_sense_range << 6) |
                (config->continuous << 4) |
                (config->vin_select << 2) |
                (config->vaux_enable << 1);
    wrData[2] = (config->simultaneous << 6) |
                (config->power_averaging << 3) |
                (config->voltage_current_averaging);
    
    
    ADI_EXPECT(adi_adm1293_i2c_write, adm1293, wrData, ADI_ARRAY_LEN(wrData));
    
    ADI_API_RETURN(adm1293);
}

int32_t adi_adm1293_Read(adi_adm1293_Device_t *adm1293,
                         uint32_t *measuredVoltage_uV,
                         uint32_t *measuredCurrent_uA)
{
	ADI_EXPECT(adi_adm1293_VoltageRead, adm1293, measuredVoltage_uV);
	ADI_EXPECT(adi_adm1293_CurrentRead, adm1293, measuredCurrent_uA);
    ADI_API_RETURN(adm1293);
}

int32_t adi_adm1293_VoltageRead(adi_adm1293_Device_t *adm1293,
								uint32_t *measuredVoltage_uV)
{

	uint8_t buff[2] = { 0 };
	uint32_t voltage = 0;
    uint16_t pmon_config = 0;
    adi_adm1293_vinSelect_e vin = ADI_ADM1293_VINSELECT_DISABLED;
    
	static const uint8_t VOLTAGE_ADDR = 0x88;
    static const uint8_t CONFIG_ADDR = 0xD4;
    
	ADI_NULL_DEVICE_PTR_RETURN(adm1293);
	ADI_NULL_PTR_RETURN(adm1293, measuredVoltage_uV);
    
	ADI_EXPECT(adi_adm1293_i2c_read, adm1293, &VOLTAGE_ADDR, 1, buff, 2);
	voltage = (((uint16_t)buff[1] << 8) | buff[0]);
    
    ADI_EXPECT(adi_adm1293_i2c_read, adm1293, &CONFIG_ADDR, 1, buff, 2);
    pmon_config = (((uint16_t)buff[1] << 8) | buff[0]);
    vin = (adi_adm1293_vinSelect_e)((pmon_config >> 2) & 0x3);
    
    switch (vin)
    {
    case ADI_ADM1293_VINSELECT_FS_1P2:
        *measuredVoltage_uV = 300 * voltage + 150;
        break;
    case ADI_ADM1293_VINSELECT_FS_7P4:
        *measuredVoltage_uV = 1801 * voltage + 901;
        break;
    case ADI_ADM1293_VINSELECT_FS_21:
        *measuredVoltage_uV = 5101 * voltage + 2551;
        break;
    case ADI_ADM1293_VINSELECT_DISABLED:    /* Falls through */
    default:
        *measuredVoltage_uV = 0;
    }
            
    ADI_API_RETURN(adm1293);
}

int32_t adi_adm1293_CurrentRead(adi_adm1293_Device_t *adm1293,
								uint32_t *measuredCurrent_uA)
{
	uint8_t buff[2] = { 0 };
	uint32_t current = 0;
    uint16_t pmon_config = 0;
    adi_adm1293_currentRange_e irange = ADI_ADM1293_CURRENTRANGE_PM25;
    
	static const uint8_t CURRENT_ADDR = 0x8c;
    static const uint8_t CONFIG_ADDR = 0xD4;
    
	ADI_NULL_DEVICE_PTR_RETURN(adm1293);
	ADI_NULL_PTR_RETURN(adm1293, measuredCurrent_uA);
    
	ADI_EXPECT(adi_adm1293_i2c_read, adm1293, &CURRENT_ADDR, 1, buff, 2);
	current = ((uint16_t)buff[1] << 8) | buff[0];
    
    ADI_EXPECT(adi_adm1293_i2c_read, adm1293, &CONFIG_ADDR, 1, buff, 2);
    pmon_config = (((uint16_t)buff[1] << 8) | buff[0]);
    irange = (adi_adm1293_currentRange_e)((pmon_config >> 6) & 0x3);
    
	if (current == 0xFFFF || current == 0xFFFE)
	{
		*measuredCurrent_uA = 0;
	}
	else
	{
    	switch (irange)
    	{
    	case ADI_ADM1293_CURRENTRANGE_PM25:
    	    *measuredCurrent_uA = 1250 * current + 1250;
        	break;
    	case ADI_ADM1293_CURRENTRANGE_PM50:
        	*measuredCurrent_uA = 2500 * current + 2500;
        	break;
    	case ADI_ADM1293_CURRENTRANGE_PM100:
        	*measuredCurrent_uA = 5000 * current + 5000;
        	break;
    	case ADI_ADM1293_CURRENTRANGE_PM200:
        	*measuredCurrent_uA = 10000 * current + 10000;
        	break;
    	default:
        	*measuredCurrent_uA = 0;
    	}
	}
    
    ADI_API_RETURN(adm1293);
}
