/**
* \file
* \brief Functions for interacting with the power monitoring chips on the CE board
*
* ADRV9001 API Version: $ADI_ADRV9001_API_VERSION$
*/

/**
* Copyright 2019 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/

#ifndef _ADI_ADM1293_H_
#define _ADI_ADM1293_H_

#ifdef __KERNEL__
#include <linux/kernel.h>
#else
#include <stdint.h>
#endif
#include "adi_adm1293_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Configure the power monitoring chips for operation
 * 
 * \note This function must be called once before calling adi_adrv9001Ce01_PowerMonitor_Read()
 *
 * \param[in] adm1293   Context variable
 * \param[in] config	The desired power monitor configuration
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_adm1293_Configure(adi_adm1293_Device_t *adm1293, adi_adm1293_config_t *config);

/**
 * \brief Read the voltage and current from the specified power monitor
 * 
 * \param[in]  adm1293					Context variable
 * \param[out] measuredVoltage_uV       The measured voltage, denoted in microvolts (uV)
 * \param[out] measuredCurrent_uA       The measured current, denoted in microamperes (uA)
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_adm1293_Read(adi_adm1293_Device_t *adm1293,
                         uint32_t *measuredVoltage_uV,
                         uint32_t *measuredCurrent_uA);

/**
 * \brief Read the voltage from the specified power monitor
 * 
 * \param[in]  adm1293					Context variable - Pointer to the ADRV9001 CE01 daughter card instance
 * \param[out] measuredVoltage_uV       The measured voltage, denoted in microvolts (uV)
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_adm1293_VoltageRead(adi_adm1293_Device_t *adm1293,
								uint32_t *measuredVoltage_uV); 

/**
 * \brief Read the current from the specified power monitor
 * 
 * \param[in]  adm1293					Context variable - Pointer to the ADRV9001 CE01 daughter card instance
 * \param[out] measuredCurrent_uA       The measured current, denoted in microamperes (uA)
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_adm1293_CurrentRead(adi_adm1293_Device_t *adm1293,
								uint32_t *measuredCurrent_uA); 

#ifdef __cplusplus
}
#endif

#endif
