// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       Platform Common
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

#include "common.h"
#include "xiicps.h"

#define AXI_DMA_RX0_OFFSET 0x0000000
#define AXI_DMA_RX1_OFFSET 0x1000000
#define AXI_DMA_TX0_OFFSET 0x2000000
#define AXI_DMA_TX1_OFFSET 0x3000000
#define AXI_DMA_LENGTH 0x10000

// **********************************************************************************
// **********************************************************************************

uint32_t axi_dma_data(uint32_t data) {

  int32_t i;
  uint32_t count;
  uint32_t data_32;

  data_32 = 0;
  count = data;

  for (i = 0; i < 4; i++) {
    count = (count + 1) & 0xff;
    data_32 = (data_32 << 8) | count;
  }

  return(data_32);
}

// **********************************************************************************
// **********************************************************************************

void axi_rxdma_setup(void) {

  struct axi_dma_params params;
  uint32_t mode[4];
  uint32_t base;
  uint32_t data_r;
  uint32_t data_e;
  uint32_t data_c;
  int32_t i;
  int32_t n;

  params.length = AXI_DMA_LENGTH;
  params.qthreshold = 2;
  params.cyclic = 0;

  mode[0] = 0;
  mode[1] = 0;
  mode[2] = 0;
  mode[3] = 0;

  for (i = 0; i < 2; i++) {
    base = (i == 0) ? AXI_DMA_RX0_OFFSET : AXI_DMA_RX1_OFFSET; 
    params.start_address = base + AXI_MEM_ID;
    base = (i == 0) ? AXI_DMA_RX0_ID : AXI_DMA_RX1_ID;
    axi_dma_run(0, base, &params, &mode[0], (uint32_t) -1);
    data_e = axi_reg_read(0, params.start_address, 0x0);
    data_c = 0;
    for (n = 1; n < (AXI_DMA_LENGTH/4); n++) {
      data_e = axi_dma_data(data_e);
      data_r = axi_reg_read(0, params.start_address, (n*4));
      if (data_r != data_e) {
        axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: RXDMA[%x]: "
          "index[%d], received[%x], expected[%x]!\n",
          __FILE__, __LINE__, base, n, data_r, data_e);
        data_c = data_c + 1;
        if (data_c >= 5) break;
      }
    }
  }
}

// **********************************************************************************
// **********************************************************************************

void axi_txdma_setup(void) {

  struct axi_dma_params params;
  uint32_t mode[4];
  uint32_t base;
  uint32_t data;
  int32_t i;
  int32_t n;

  params.length = AXI_DMA_LENGTH;
  params.qthreshold = 2;
  params.cyclic = 1;

  mode[0] = 0;
  mode[1] = 0;
  mode[2] = 0;
  mode[3] = 0;

  for (i = 0; i < 2; i++) {
    base = (i == 0) ? AXI_DMA_TX0_OFFSET : AXI_DMA_TX1_OFFSET; 
    params.start_address = base + AXI_MEM_ID;
    data = 0;
    for (n = 0; n < (AXI_DMA_LENGTH/4); n++) {
      data = axi_dma_data(data);
      axi_reg_write(0, params.start_address, (n*4), data);
    }
    base = (i == 0) ? AXI_DMA_TX0_ID : AXI_DMA_TX1_ID;
    axi_dma_trigger_mode_set(0, base, &mode[0]);
    axi_dma_config(0, base, &params);
    axi_dma_start(0, base);
  }
}

// **********************************************************************************
// **********************************************************************************

void axi_adrv9001_clk_setup(void) {

  int32_t status;
  uint32_t locked;
  struct axi_adrv9001_clk_params clk_param;

  axi_adrv9001_delay_us(0, AXI_ADRV9001_ID, 2000);
  axi_platform_log(0, AXI_PLATFORM_INFO, "reference clock [%d Hz]\n",
    axi_adrv9001_mmcm_ref_clk_freq_get(0, AXI_ADRV9001_ID));

  clk_param.ref_clk_freq_hz = 38400000;
  clk_param.dev_clk_freq_hz = 38400000;

  status = axi_adrv9001_clk_config(0, AXI_ADRV9001_ID, &clk_param, &locked);
  if ((status != 0) || (locked == 0)) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: function call "
      "`axi_adrv9001_clk_config()' has failed with return status [%d]!\n",
      __FILE__, __LINE__, status);
  }

  axi_adrv9001_delay_us(0, AXI_ADRV9001_ID, 4000);
  axi_platform_log(0, AXI_PLATFORM_INFO, "TDD clock [%d Hz]\n",
    axi_adrv9001_tdd_clk_freq_get(0, AXI_ADRV9001_ID));
  axi_platform_log(0, AXI_PLATFORM_INFO, "GPIO clock [%d Hz]\n",
    axi_adrv9001_gpio_clk_freq_get(0, AXI_ADRV9001_ID));
  axi_platform_log(0, AXI_PLATFORM_INFO, "MCS clock [%d Hz]\n",
    axi_adrv9001_mcs_clk_freq_get(0, AXI_ADRV9001_ID));
}

void axi_adrv9001_ssi_setup(uint32_t cmos1_lvds0) {

  int32_t i;
  int32_t status;
  uint32_t ssi_id;

  struct axi_adrv9001_ssi_params ssi_param;

  if (cmos1_lvds0 == 1) {
    axi_adrv9001_ssi_init_param(0, AXI_ADRV9001_ID, &ssi_param, CMOS_1L_PS_32X1);
    ssi_param.sdr1_ddr0 = 1;
  } else {
    axi_adrv9001_ssi_init_param(0, AXI_ADRV9001_ID, &ssi_param, LVDS_2L_PS_32X1);
    ssi_param.sdr1_ddr0 = 0;
  }

  for (i = 0; i < 4; i++) {
    ssi_id = AXI_ADRV9001_SSI_RX0_ID + (i * 0x1000);
    status = axi_adrv9001_ssi_config(0, AXI_ADRV9001_ID, ssi_id, &ssi_param);
    if (status != 0) {
      axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: function call "
        "`axi_adrv9001_ssi_config(%x)' has failed with return status [%d]!\n",
        __FILE__, __LINE__, ssi_id, status);
    }
    axi_adrv9001_delay_us(0, AXI_ADRV9001_ID, 4000);
    axi_platform_log(0, AXI_PLATFORM_INFO, "SSI[%x] clock [%d Hz], ratio [%d]\n", ssi_id,
      axi_adrv9001_ssi_clk_freq_get(0, AXI_ADRV9001_ID, ssi_id),
      axi_adrv9001_tdd_enable_clk_ratio_get(0, AXI_ADRV9001_ID,
      (AXI_ADRV9001_TDD_RX0_DMA_ID + (i * 0x100))));
  }
}

void axi_adrv9001_setup(uint32_t cmos1_lvds0) {

  axi_adrv9001_clk_setup();
  axi_adrv9001_ssi_setup(cmos1_lvds0);
}

// **********************************************************************************
// **********************************************************************************

int32_t axi_platform_ssi_calibrate(void) {

  int32_t i;
  int32_t status;
  int32_t status_all;
  uint32_t ssi_id;
  uint32_t ssi_delay;
  uint32_t pattern[2];

  status_all = 0;
  pattern[0] = 0xa956f143;
  pattern[1] = 0xffffffff;

  for (i = 0; i < 2; i++) {
    ssi_id = (i == 0) ? AXI_ADRV9001_SSI_RX0_ID : AXI_ADRV9001_SSI_RX1_ID;
    axi_adrv9001_ssi_data_sel_set(0, AXI_ADRV9001_ID, ssi_id, AXI_ADRV9001_SSI_DATA_SEL_RAMP);
    status = axi_adrv9001_ssi_delay_calibrate(0, AXI_ADRV9001_ID, ssi_id, 1, &ssi_delay);
    if (status != 0) {
      axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: function call "
        "`axi_adrv9001_ssi_delay_calibrate(%x)' has failed with return status [%d]!\n",
        __FILE__, __LINE__, ssi_id, status);
    } else {
      axi_platform_log(0, AXI_PLATFORM_INFO, "SSI[%x] Calibrated delay is "
        "%d (0x%x).\n", ssi_id, ssi_delay, ssi_delay);
    }
    axi_adrv9001_ssi_data_sel_set(0, AXI_ADRV9001_ID, ssi_id, AXI_ADRV9001_SSI_DATA_SEL_PATTERN);
    axi_adrv9001_ssi_data_pattern_set(0, AXI_ADRV9001_ID, ssi_id, &pattern[0]);
    status_all = status_all | status;
  }

  if (status_all != 0) {
    return(-1);
  }

  for (i = 0; i < 2; i++) {
    adrv9001_tx2rx_loopback(i);
    ssi_id = (i == 0) ? AXI_ADRV9001_SSI_TX0_ID : AXI_ADRV9001_SSI_TX1_ID;
    axi_adrv9001_ssi_data_sel_set(0, AXI_ADRV9001_ID, ssi_id, AXI_ADRV9001_SSI_DATA_SEL_PATTERN);
    axi_adrv9001_ssi_data_pattern_set(0, AXI_ADRV9001_ID, ssi_id, &pattern[0]);
    status = axi_adrv9001_ssi_delay_calibrate(0, AXI_ADRV9001_ID, ssi_id, 1, &ssi_delay);
    if (status != 0) {
      axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: function call "
        "`axi_adrv9001_ssi_delay_calibrate(%x)' has failed with return status [%d]!\n",
        __FILE__, __LINE__, ssi_id, status);
    } else {
      axi_platform_log(0, AXI_PLATFORM_INFO, "SSI[%x] Calibrated delay is "
        "%d (0x%x).\n", ssi_id, ssi_delay, ssi_delay);
    }
    axi_adrv9001_ssi_data_sel_set(0, AXI_ADRV9001_ID, ssi_id, AXI_ADRV9001_SSI_DATA_SEL_DMA);
    status_all = status_all | status;
  }

  return(status_all);
}

// **********************************************************************************
// **********************************************************************************

void axi_mspi_setup(void) {

  struct axi_mspi_params mspi_params;

  mspi_params.clk_period_count = 7;
  mspi_params.clk_high_count = 1;
  mspi_params.clk_low_count = 5;
  mspi_params.mosi_count = 0;
  mspi_params.miso_count = 6;
  mspi_params.sdio_count = 0;
  mspi_params.sdio_mode = 0;

  axi_mspi_config(0, AXI_MSPI_ID, &mspi_params);
  axi_mspi_chip_select_set(0, AXI_MSPI_ID, 0, 0x1);
  axi_mspi_enable_set(0, AXI_MSPI_ID, 0x1);
}

// **********************************************************************************
// **********************************************************************************

void axi_platform_setup(void) {

  uint32_t count;
  uint32_t data[9];

  data[8] = 0;

  count = 0;
  axi_reg_write(0, 0xf8000000, 0x240, 0x0);
  while ((axi_reg_read(0, 0xf8000000, 0x240) & 0xf) == 0x0) {
    if (count == 10) break;
    count = count + 1;
  }

  count = 0;
  axi_reg_write(0, 0xf8000000, 0x240, 0xf);
  while ((axi_reg_read(0, 0xf8000000, 0x240) & 0xf) == 0xf) {
    if (count == 10) break;
    count = count + 1;
  }

  axi_sysid_sys_info_0_get(0, AXI_SYSID_ID, &data[0]);
  axi_platform_log(0, AXI_PLATFORM_INFO, "%s\n", (char *) &data[0]);
  axi_sysid_sys_info_1_get(0, AXI_SYSID_ID, &data[0]);
  axi_platform_log(0, AXI_PLATFORM_INFO, "%s\n", (char *) &data[0]);
  axi_sysid_sys_version_get(0, AXI_SYSID_ID, &data[0]);
  axi_platform_log(0, AXI_PLATFORM_INFO, "version[%d.%d.%d]\n",
    ((data[0] >> 16) & 0xff), ((data[0] >> 8) & 0xff), (data[0] & 0xff));
  axi_sysid_sys_params_get(0, AXI_SYSID_ID, &data[0]);
  axi_platform_log(0, AXI_PLATFORM_INFO, "parameters[%x.%x]\n",
    data[0], data[1]);
  axi_sysid_sys_build_get(0, AXI_SYSID_ID, &data[0]);
  axi_platform_log(0, AXI_PLATFORM_INFO, "build[%x.%x]\n",
    data[0], data[1]);

  axi_reg_write(0, 0x43100000, 0x0, 0xa);
  axi_sysid_timer_set(0, AXI_SYSID_ID, (uint32_t) -1);
  while (axi_reg_read(0, 0x43100000, 0x4) == 0) {
    if (axi_sysid_timer_get(0, AXI_SYSID_ID) == 0) {
      axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: axi_platform_setup(), "
        "no reference clock (clock generator not locked)!\n", __FILE__, __LINE__);
    }
  }
}

// **********************************************************************************
// **********************************************************************************

int32_t axi_platform_clock_setup(void) {

  int32_t status;
  uint32_t locked;

  status = axi_clkgen_configure(0, AXI_CLKGEN_ID, 38400000, &locked);
  if ((status != 0) || (locked != 1)) {
    return(-1);
  }

  return(0);
}

// **********************************************************************************
// **********************************************************************************

int32_t axi_platform_power_setup(void) {

  int32_t status;
  uint32_t voltage;
  unsigned char data[3];

  XIicPs ps7_iic;
  XIicPs_Config *ps7_iic_config;

  status = 0;

  ps7_iic_config = XIicPs_LookupConfig(XPAR_PS7_I2C_0_DEVICE_ID);
  if (ps7_iic_config == NULL) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XIicPs_LookupConfig() FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  status = XIicPs_CfgInitialize(&ps7_iic, ps7_iic_config, ps7_iic_config->BaseAddress);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XIicPs_CfgInitialize() FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  XIicPs_SetSClk(&ps7_iic, 400000);

  data[0] = 0x80;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 1, 0x74);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(IIC_MUX) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterRecvPolled(&ps7_iic, &data[1], 1, 0x74);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Read(IIC_MUX) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  if (data[0] != data[1]) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "IIC_MUX_MISMATCH(%02x, %02x) FAILED!\n", __FILE__, __LINE__,
      data[0], data[1]);
    return(-1);
  }

  data[0] = 0x00;
  data[1] = 0x03;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 2, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_00) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0x44;
  data[1] = 0xf6;
  data[2] = 0x30;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 3, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_44) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0xfa;
  data[1] = 0x0f;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 2, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_FA) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0xfb;
  data[1] = 0x07;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 2, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_FB) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0x40;
  data[1] = 0x3e;
  data[2] = 0x42;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 3, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_40) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0x5e;
  data[1] = 0xd7;
  data[2] = 0x33;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 3, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_5E) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0x5f;
  data[1] = 0xf6;
  data[2] = 0x30;

  while (XIicPs_BusIsBusy(&ps7_iic) != 0);
  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 3, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(PMBUS_5F) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  status = XIicPs_SetOptions(&ps7_iic, XIICPS_REP_START_OPTION);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XIicPs_SetOptions() FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  data[0] = 0x8b;

  status = XIicPs_MasterSendPolled(&ps7_iic, &data[0], 1, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Write(CMD_READ_VOLTAGE) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  status = XIicPs_MasterRecvPolled(&ps7_iic, &data[0], 2, 0x65);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XicPs_Read(CMD_READ_VOLTAGE) FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  status = XIicPs_ClearOptions(&ps7_iic, XIICPS_REP_START_OPTION);
  if (status != XST_SUCCESS) {
    axi_platform_log(0, AXI_PLATFORM_ERROR, "%s:%d: "
      "XIicPs_ClearOptions() FAILED!\n", __FILE__, __LINE__);
    return(-1);
  }

  voltage = ((data[1] << 8) | data[0]) * 122;

  axi_platform_log(0, AXI_PLATFORM_INFO, "VADJ[%d uV].\n", voltage);
  return(status);
}

// **********************************************************************************
// **********************************************************************************

