// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       TDD reference frame
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module tdd_frame (

  // source

  input   wire            tdd_resetn,
  input   wire            tdd_clk,
  output  reg             tdd_state = 'd0,
  output  reg             tdd_frm_toggle = 'd0,
  output  reg   [ 31:0]   tdd_clk_cnt = 'd0,
  output  reg   [ 31:0]   tdd_frm_cnt = 'd0,

  // trigger inputs

  input   wire  [  3:0]   trig,

  // axilite

  input   wire            axilite_clk,
  input   wire            axilite_resetn,
  input   wire  [ 31:0]   axilite_frame_period,
  input   wire  [ 31:0]   axilite_num_of_frames,
  input   wire  [ 31:0]   axilite_frame_switch_period,
  input   wire  [ 31:0]   axilite_frame_switch_number,
  input   wire  [ 11:0]   axilite_trigger_mode,
  input   wire            axilite_frame_switch_enable,
  input   wire            axilite_sequence_repeat,
  input   wire            axilite_stop,
  input   wire            axilite_start,
  output  wire            axilite_request_clr);

  // internal registers

  reg                     tdd_start_d = 'd0;
  reg                     tdd_request_clr = 'd0;
  reg                     tdd_start_req = 'd0;
  reg           [  3:0]   tdd_trig_d = 'd0;
  reg           [ 11:0]   tdd_trig_mode_d = 'd0;
  reg           [  3:0]   tdd_trig_enable = 'd0;

  // internal signals

  wire          [  3:0]   tdd_trig;
  wire          [ 31:0]   tdd_frame_period;
  wire          [ 31:0]   tdd_num_of_frames;
  wire          [ 31:0]   tdd_frame_switch_period;
  wire          [ 31:0]   tdd_frame_switch_number;
  wire          [ 11:0]   tdd_trig_mode;
  wire                    tdd_frame_switch_enable;
  wire                    tdd_sequence_repeat;
  wire                    tdd_stop;
  wire                    tdd_start;

  // tdd timing reference

  always @(negedge tdd_resetn or posedge tdd_clk) begin
    if (tdd_resetn == 1'b0) begin
      tdd_start_d <= 1'd0;
      tdd_request_clr <= 1'd0;
      tdd_start_req <= 1'd0;
      tdd_state <= 1'd0;
      tdd_clk_cnt <= 32'd0;
      tdd_frm_toggle <= 1'd0;
      tdd_frm_cnt <= 32'd0;
    end else begin
      tdd_start_d <= tdd_start;
      tdd_request_clr <= ~tdd_state;
      if (tdd_start_req == 1'd1) begin
        tdd_start_req <= ~tdd_state;
      end else begin
        tdd_start_req <= ~tdd_start_d & tdd_start;
      end
      if (tdd_state == 1'd1) begin
        if (((tdd_clk_cnt >= tdd_frame_switch_period) &&
          (tdd_frm_cnt == tdd_frame_switch_number) &&
          (tdd_frame_switch_enable == 1'd1)) ||
          (tdd_clk_cnt >= tdd_frame_period)) begin
          tdd_clk_cnt <= 32'd0;
          tdd_frm_toggle <= ~tdd_frm_toggle;
          if (tdd_frm_cnt >= tdd_num_of_frames) begin
            tdd_frm_cnt <= 32'd0;
            tdd_state <= tdd_sequence_repeat & ~tdd_stop;
          end else begin
            tdd_frm_cnt <= tdd_frm_cnt + 1'd1;
            tdd_state <= tdd_state;
          end
        end else begin
          tdd_clk_cnt <= tdd_clk_cnt + 1'd1;
          tdd_frm_toggle <= tdd_frm_toggle;
          tdd_frm_cnt <= tdd_frm_cnt;
          tdd_state <= tdd_state;
        end
      end else begin
        tdd_clk_cnt <= 32'd0;
        tdd_frm_toggle <= tdd_frm_toggle;
        tdd_frm_cnt <= 32'd0;
        tdd_state <= tdd_start_req & (& tdd_trig_enable);
      end
    end
  end

  generate
  for (genvar n = 0; n < 4; n = n + 1) begin: g_tdd_trig

  always @(negedge tdd_resetn or posedge tdd_clk) begin
    if (tdd_resetn == 1'b0) begin
      tdd_trig_d[n] <= 1'd0;
      tdd_trig_mode_d[((n*3)+2):(n*3)] <= 3'd0;
      tdd_trig_enable[n] <= 1'd0;
    end else begin
      tdd_trig_d[n] <= tdd_trig[n];
      tdd_trig_mode_d[((n*3)+2):(n*3)] <= tdd_trig_mode[((n*3)+2):(n*3)];
      if ((tdd_trig_mode_d[((n*3)+2):(n*3)] != tdd_trig_mode[((n*3)+2):(n*3)]) ||
        (tdd_state == 1'd1)) begin
        tdd_trig_enable[n] <= 1'd0;
      end else if (tdd_trig_enable[n] == 1'd0) begin
        case (tdd_trig_mode[((n*3)+2):(n*3)])
          3'd6: tdd_trig_enable[n] <= tdd_trig[n] ^ tdd_trig_d[n];
          3'd5: tdd_trig_enable[n] <= tdd_trig[n] & ~tdd_trig_d[n];
          3'd4: tdd_trig_enable[n] <= ~tdd_trig[n] & tdd_trig_d[n];
          default: tdd_trig_enable[n] <= 1'd1;
        endcase
      end
    end
  end

  end
  endgenerate

  // instantiations

  cdc #(.DATA_WIDTH(4)) i_tdd_cdc (
    .src_data         (trig),
    .dest_resetn      (tdd_resetn),
    .dest_clk         (tdd_clk),
    .dest_data        (tdd_trig));

  cdc_cntrl #(.DATA_WIDTH(144)) i_tdd_cdc_cntrl (
    .src_resetn       (axilite_resetn),
    .src_clk          (axilite_clk),
    .src_data         ({axilite_frame_period,
                        axilite_num_of_frames,
                        axilite_frame_switch_period,
                        axilite_frame_switch_number,
                        axilite_trigger_mode,
                        axilite_frame_switch_enable,
                        axilite_sequence_repeat,
                        axilite_stop,
                        axilite_start}),
    .dest_resetn      (tdd_resetn),
    .dest_clk         (tdd_clk),
    .dest_data        ({tdd_frame_period,
                        tdd_num_of_frames,
                        tdd_frame_switch_period,
                        tdd_frame_switch_number,
                        tdd_trig_mode,
                        tdd_frame_switch_enable,
                        tdd_sequence_repeat,
                        tdd_stop,
                        tdd_start}));

  cdc #(.DATA_WIDTH(1)) i_axilite_cdc (
    .src_data         (tdd_request_clr),
    .dest_resetn      (axilite_resetn),
    .dest_clk         (axilite_clk),
    .dest_data        (axilite_request_clr));

endmodule

// **********************************************************************************
// **********************************************************************************
