// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       FIFO, simple dual port
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module mem_fifo #(

  parameter   DATA_WIDTH = 16) (

  input   wire                        wr_clk,
  input   wire                        wr_resetn,
  input   wire                        wr_valid,
  input   wire  [(DATA_WIDTH-1):0]    wr_data,
  output  reg                         wr_full = 'd0,

  input   wire                        rd_clk,
  input   wire                        rd_resetn,
  input   wire                        rd_read,
  output  reg                         rd_valid = 'd0,
  output  reg   [(DATA_WIDTH-1):0]    rd_data = 'd0,
  output  reg                         rd_empty = 'd0);

  // internal registers
 
  reg           [  3:0]               wr_wraddr = 'd0;
  reg           [  3:0]               wr_wraddr_g = 'd0;
  reg           [  3:0]               wr_rdaddr = 'd0;
  reg           [  3:0]               rd_rdaddr = 'd0;
  reg           [  3:0]               rd_rdaddr_g = 'd0;
  reg           [  3:0]               rd_wraddr = 'd0;
  reg                                 rd_valid_d = 'd0;
  reg                                 rd_valid_2d = 'd0;
  reg           [(DATA_WIDTH-1):0]    rd_data_2d = 'd0;

  // internal signals
 
  wire                                wr_full_s;
  wire          [  3:0]               wr_rdaddr_g;
  wire                                rd_empty_s;
  wire          [  3:0]               rd_wraddr_g;
  wire          [(DATA_WIDTH-1):0]    rd_data_d;

  // binary to gray

  function [3:0]  b2g;
    input [3:0] din;
    reg   [3:0] dout;
    begin
      dout[3] = din[3];
      dout[2] = din[3] ^ din[2];
      dout[1] = din[2] ^ din[1];
      dout[0] = din[1] ^ din[0];
      b2g = dout;
    end
  endfunction

  // gray to binary

  function [3:0]  g2b;
    input [3:0] din;
    reg   [3:0] dout;
    begin
      dout[3] = din[3];
      dout[2] = din[3] ^ din[2];
      dout[1] = din[3] ^ din[2] ^ din[1];
      dout[0] = din[3] ^ din[2] ^ din[1] ^ din[0];
      g2b = dout;
    end
  endfunction

  // write
 
  assign wr_full_s = (wr_rdaddr == (wr_wraddr + 1'd1)) ? 1'd1 : 1'd0;

  always @(negedge wr_resetn or posedge wr_clk) begin
    if (wr_resetn == 1'd0) begin
      wr_wraddr <= 4'h0;
      wr_wraddr_g <= 4'h0;
      wr_rdaddr <= 4'h0;
      wr_full <= 1'd0;
    end else begin
      if ((wr_valid == 1'd1) && (wr_full_s == 1'd0)) begin
        wr_wraddr <= wr_wraddr + 1'd1;
      end
      wr_wraddr_g <= b2g(wr_wraddr);
      wr_rdaddr <= g2b(wr_rdaddr_g);
      wr_full <= wr_full_s;
    end
  end

  // read

  assign rd_empty_s = (rd_wraddr == rd_rdaddr) ? 1'd1 : 1'd0;

  always @(negedge rd_resetn or posedge rd_clk) begin
    if (rd_resetn == 1'd0) begin
      rd_rdaddr <= 4'd0;
      rd_rdaddr_g <= 4'd0;
      rd_wraddr <= 4'd0;
      rd_empty <= 1'd1;
      rd_valid_d <= 1'd0;
      rd_valid_2d <= 1'd0;
      rd_data_2d <= 'd0;
      rd_valid <= 1'd0;
      rd_data <= 'd0;
    end else begin
      if ((rd_read == 1'd1) && (rd_empty_s == 1'd0)) begin
        rd_rdaddr <= rd_rdaddr + 1'd1;
      end
      rd_rdaddr_g <= b2g(rd_rdaddr);
      rd_wraddr <= g2b(rd_wraddr_g);
      if (rd_empty_s == 1'd0) begin
        rd_empty <= 1'd0;
      end else if (rd_read == 1'd1) begin
        rd_empty <= 1'd1;
      end
      rd_valid_d <= rd_read & ~rd_empty_s;
      rd_valid_2d <= rd_valid_d;
      rd_data_2d <= rd_data_d;
      rd_valid <= rd_valid_2d;
      if (rd_valid_2d == 1'd1) begin
        rd_data <= rd_data_2d;
      end
    end
  end


  // instantiations
 
  cdc #(.DATA_WIDTH(4)) i_rd_cdc (
    .src_data             (wr_wraddr_g),
    .dest_resetn          (rd_resetn),
    .dest_clk             (rd_clk),
    .dest_data            (rd_wraddr_g));

  cdc #(.DATA_WIDTH(4)) i_wr_cdc (
    .src_data             (rd_rdaddr_g),
    .dest_resetn          (wr_resetn),
    .dest_clk             (wr_clk),
    .dest_data            (wr_rdaddr_g));

  mem_sdp #(
    .DATA_WIDTH           (DATA_WIDTH),
    .ADDRESS_WIDTH        (4))
  i_lb_mem (
    .wr_clk               (wr_clk),
    .wr_enable            (wr_valid),
    .wr_addr              (wr_wraddr),
    .wr_data              (wr_data),
    .rd_clk               (rd_clk),
    .rd_enable            (rd_read),
    .rd_addr              (rd_rdaddr),
    .rd_data              (rd_data_d));

endmodule

// **********************************************************************************
// **********************************************************************************
