// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       AXILite slave interface
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axilite_slave_if #(parameter AXI_ADDRESS_WIDTH = 16) (

  // axilite interface
 
  input   wire                              axilite_clk,
  input   wire                              axilite_resetn,
  input   wire                              axilite_awvalid,
  input   wire  [(AXI_ADDRESS_WIDTH-1):0]   axilite_awaddr,
  output  reg                               axilite_awready = 'd0,
  input   wire                              axilite_wvalid,
  input   wire  [   31:0]                   axilite_wdata,
  output  reg                               axilite_wready = 'd0,
  output  reg                               axilite_bvalid = 'd0,
  output  wire  [    1:0]                   axilite_bresp,
  input   wire                              axilite_bready,
  input   wire                              axilite_arvalid,
  input   wire  [(AXI_ADDRESS_WIDTH-1):0]   axilite_araddr,
  output  reg                               axilite_arready = 'd0,
  output  reg                               axilite_rvalid = 'd0,
  output  wire  [    1:0]                   axilite_rresp,
  output  reg   [   31:0]                   axilite_rdata = 'd0,
  input   wire                              axilite_rready,

  // internal interface

  output  reg                               axilite_wrreq = 'd0,
  output  reg   [(AXI_ADDRESS_WIDTH-1):0]   axilite_wraddr = 'd0,
  output  reg   [   31:0]                   axilite_wrdata = 'd0,
  input   wire                              axilite_wrack,
  output  reg                               axilite_rdreq = 'd0,
  output  reg   [(AXI_ADDRESS_WIDTH-1):0]   axilite_rdaddr = 'd0,
  input   wire  [   31:0]                   axilite_rddata,
  input   wire                              axilite_rdack);

  // internal registers

  reg                                       axilite_wrack_d = 'd0;
  reg                                       axilite_wrsel = 'd0;
  reg           [    5:0]                   axilite_wrcount = 'd0;
  reg                                       axilite_rdack_d = 'd0;
  reg           [   31:0]                   axilite_rddata_d = 'd0;
  reg                                       axilite_rdsel = 'd0;
  reg           [    5:0]                   axilite_rdcount = 'd0;

  // internal signals

  wire                                      axilite_wrack_s;
  wire                                      axilite_rdack_s;
  wire          [   31:0]                   axilite_rddata_s;

  // write channel interface

  assign axilite_bresp = 2'd0;

  always @(negedge axilite_resetn or posedge axilite_clk) begin
    if (axilite_resetn == 1'b0) begin
      axilite_awready <= 'd0;
      axilite_wready <= 'd0;
      axilite_bvalid <= 'd0;
    end else begin
      if (axilite_awready == 1'b1) begin
        axilite_awready <= 1'b0;
      end else if (axilite_wrack_s == 1'b1) begin
        axilite_awready <= 1'b1;
      end
      if (axilite_wready == 1'b1) begin
        axilite_wready <= 1'b0;
      end else if (axilite_wrack_s == 1'b1) begin
        axilite_wready <= 1'b1;
      end
      if ((axilite_bready == 1'b1) && (axilite_bvalid == 1'b1)) begin
        axilite_bvalid <= 1'b0;
      end else if (axilite_wrack_d == 1'b1) begin
        axilite_bvalid <= 1'b1;
      end
    end
  end

  assign axilite_wrack_s = (axilite_wrcount == 6'h3f) ? 1'b1 :
    (axilite_wrcount[5] & axilite_wrack);

  always @(negedge axilite_resetn or posedge axilite_clk) begin
    if (axilite_resetn == 1'b0) begin
      axilite_wrack_d <= 'd0;
      axilite_wrsel <= 'd0;
      axilite_wrreq <= 'd0;
      axilite_wraddr <= 'd0;
      axilite_wrdata <= 'd0;
      axilite_wrcount <= 'd0;
    end else begin
      axilite_wrack_d <= axilite_wrack_s;
      if (axilite_wrsel == 1'b1) begin
        if ((axilite_bready == 1'b1) && (axilite_bvalid == 1'b1)) begin
          axilite_wrsel <= 1'b0;
        end
        axilite_wrreq <= 1'b0;
        axilite_wraddr <= axilite_wraddr;
        axilite_wrdata <= axilite_wrdata;
      end else begin
        axilite_wrsel <= axilite_awvalid & axilite_wvalid;
        axilite_wrreq <= axilite_awvalid & axilite_wvalid;
        axilite_wraddr <= axilite_awaddr;
        axilite_wrdata <= axilite_wdata;
      end
      if (axilite_wrack_s == 1'b1) begin
        axilite_wrcount <= 6'h00;
      end else if (axilite_wrcount[5] == 1'b1) begin
        axilite_wrcount <= axilite_wrcount + 1'b1;
      end else if (axilite_wrreq == 1'b1) begin
        axilite_wrcount <= 6'h20;
      end
    end
  end

  // read channel interface

  assign axilite_rresp = 2'd0;

  always @(negedge axilite_resetn or posedge axilite_clk) begin
    if (axilite_resetn == 1'b0) begin
      axilite_arready <= 'd0;
      axilite_rvalid <= 'd0;
      axilite_rdata <= 'd0;
    end else begin
      if (axilite_arready == 1'b1) begin
        axilite_arready <= 1'b0;
      end else if (axilite_rdack_s == 1'b1) begin
        axilite_arready <= 1'b1;
      end
      if ((axilite_rready == 1'b1) && (axilite_rvalid == 1'b1)) begin
        axilite_rvalid <= 1'b0;
        axilite_rdata <= 32'd0;
      end else if (axilite_rdack_d == 1'b1) begin
        axilite_rvalid <= 1'b1;
        axilite_rdata <= axilite_rddata_d;
      end
    end
  end

  assign axilite_rdack_s = (axilite_rdcount == 6'h3f) ? 1'b1 :
    (axilite_rdcount[5] & axilite_rdack);
  assign axilite_rddata_s = (axilite_rdcount == 6'h3f) ? {2{16'hdead}} : axilite_rddata;

  always @(negedge axilite_resetn or posedge axilite_clk) begin
    if (axilite_resetn == 1'b0) begin
      axilite_rdack_d <= 'd0;
      axilite_rddata_d <= 'd0;
      axilite_rdsel <= 'd0;
      axilite_rdreq <= 'd0;
      axilite_rdaddr <= 'd0;
      axilite_rdcount <= 'd0;
    end else begin
      axilite_rdack_d <= axilite_rdack_s;
      axilite_rddata_d <= axilite_rddata_s;
      if (axilite_rdsel == 1'b1) begin
        if ((axilite_rready == 1'b1) && (axilite_rvalid == 1'b1)) begin
          axilite_rdsel <= 1'b0;
        end
        axilite_rdreq <= 1'b0;
        axilite_rdaddr <= axilite_rdaddr;
      end else begin
        axilite_rdsel <= axilite_arvalid;
        axilite_rdreq <= axilite_arvalid;
        axilite_rdaddr <= axilite_araddr;
      end
      if (axilite_rdack_s == 1'b1) begin
        axilite_rdcount <= 6'h00;
      end else if (axilite_rdcount[5] == 1'b1) begin
        axilite_rdcount <= axilite_rdcount + 1'b1;
      end else if (axilite_rdreq == 1'b1) begin
        axilite_rdcount <= 6'h20;
      end
    end
  end

endmodule

// **********************************************************************************
// **********************************************************************************
