// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       Transmit DMA Core
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_txdma #(

  parameter integer INSTANCE_ID = 0) (

  // data interface, this is not truly a stream interface,
  // data is driven based on ready and valid has no effect.

  input   wire                  tx_clk,
  input   wire                  tx_resetn,
  input   wire                  tx_tready,
  output  reg                   tx_tvalid = 'd0,
  output  reg   [ 63:0]         tx_tdata = 'd0,

  // asynchronous events

  input   wire  [  3:0]         trig,

  // axi master interface

  input   wire                  aximm_clk,
  input   wire                  aximm_resetn,
  output  wire  [ 39:0]         aximm_araddr,
  output  wire  [  7:0]         aximm_arlen,
  output  wire  [  2:0]         aximm_arsize,
  output  wire  [  1:0]         aximm_arburst,
  output  wire  [  2:0]         aximm_arprot,
  output  wire  [  3:0]         aximm_arcache,
  output  wire  [  3:0]         aximm_aruser,
  output  wire                  aximm_arvalid,
  input   wire                  aximm_arready,
  input   wire  [511:0]         aximm_rdata,
  input   wire  [  1:0]         aximm_rresp,
  input   wire                  aximm_rlast,
  input   wire                  aximm_rvalid,
  output  wire                  aximm_rready,

  // axi lite interface

  input   wire                  axilite_clk,
  input   wire                  axilite_resetn,
  input   wire                  axilite_awvalid,
  input   wire  [ 11:0]         axilite_awaddr,
  output  wire                  axilite_awready,
  input   wire                  axilite_wvalid,
  input   wire  [ 31:0]         axilite_wdata,
  output  wire                  axilite_wready,
  output  wire                  axilite_bvalid,
  output  wire  [  1:0]         axilite_bresp,
  input   wire                  axilite_bready,
  input   wire                  axilite_arvalid,
  input   wire  [ 11:0]         axilite_araddr,
  output  wire                  axilite_arready,
  output  wire                  axilite_rvalid,
  output  wire  [  1:0]         axilite_rresp,
  output  wire  [ 31:0]         axilite_rdata,
  input   wire                  axilite_rready,
  output  wire                  interrupt);

  // constants

  localparam  VERSION = {8'd0, 8'd6, 8'd1, 8'd0};

  // internal registers
   
  reg           [  3:0]         tx_trig_d = 'd0;
  reg           [ 11:0]         tx_trig_mode_d = 'd0;
  reg           [  3:0]         tx_trig_enable = 'd0;
  reg                           tx_trig_ready = 'd0;
  reg                           tx_active = 'd0;
  reg                           aximm_swresetn = 'd0;
  reg                           aximm_swreset_d = 'd0;
  reg           [  5:0]         aximm_swreset_count = 'd0;
  reg                           aximm_start_req_d = 'd0;
  reg                           aximm_stop_req_d = 'd0;
  reg                           aximm_start = 'd0;
  reg                           aximm_stop = 'd0;
  reg                           aximm_active = 'd0;
  reg                           aximm_status = 'd0;
  reg                           aximm_slave_err = 'd0;
  reg                           aximm_addrdec_err = 'd0;
  reg                           aximm_tlast_err = 'd0;
  reg                           aximm_length_err = 'd0;
  reg           [  3:0]         aximm_tag = 'd0;
  reg           [  3:0]         aximm_req_tag = 'd0;
  reg                           aximm_req_valid = 'd0;
  reg                           aximm_req_active = 'd0;
  reg           [  3:0]         aximm_qcnt = 'd0;

  // internal signals
 
  wire          [  3:0]         tx_trig;
  wire          [ 11:0]         tx_trig_mode;
  wire                          tx_mm_active;
  wire                          tx_underflow;
  wire                          tx_ready;
  wire                          tx_valid;
  wire          [ 63:0]         tx_data;
  wire                          aximm_ack_ready;
  wire                          aximm_done;
  wire          [ 79:0]         aximm_req_data;
  wire                          aximm_swreset;
  wire                          aximm_start_req;
  wire                          aximm_stop_req;
  wire                          aximm_cyclic;
  wire          [  3:0]         aximm_qthreshold;
  wire          [ 39:0]         aximm_start_addr;
  wire          [ 22:0]         aximm_length;
  wire                          aximm_valid;
  wire          [ 63:0]         aximm_data;
  wire                          aximm_ready;
  wire                          aximm_err;
  wire                          aximm_req_ready;
  wire                          aximm_ack_valid;
  wire          [  7:0]         aximm_ack_data;
  wire                          axilite_underflow;
  wire                          axilite_swreset_clr;
  wire                          axilite_busy;
  wire                          axilite_status;
  wire                          axilite_length_err;
  wire                          axilite_tlast_err;
  wire                          axilite_addrdec_err;
  wire                          axilite_slave_err;
  wire                          axilite_swreset;
  wire                          axilite_start;
  wire                          axilite_stop;
  wire                          axilite_cyclic;
  wire          [  3:0]         axilite_qthreshold;
  wire          [ 39:0]         axilite_start_addr;
  wire          [ 22:0]         axilite_length;
  wire          [ 11:0]         axilite_trig_mode;
  wire                          axilite_wrreq;
  wire          [ 11:0]         axilite_wraddr;
  wire          [ 31:0]         axilite_wrdata;
  wire                          axilite_wrack;
  wire                          axilite_rdreq;
  wire          [ 11:0]         axilite_rdaddr;
  wire          [ 31:0]         axilite_rddata;
  wire                          axilite_rdack;

  // device interface

  assign interrupt = axilite_busy;
  assign tx_underflow = tx_active & tx_ready & ~tx_valid;
  assign tx_ready = tx_tready & tx_trig_ready;

  generate
  for (genvar n = 0; n < 4; n = n + 1) begin: g_tx_trig

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_trig_d[n] <= 1'd0;
      tx_trig_mode_d[((n*3)+2):(n*3)] <= 2'd0;
      tx_trig_enable[n] <= 1'd0;
    end else begin
      tx_trig_d[n] <= tx_trig[n];
      tx_trig_mode_d[((n*3)+2):(n*3)] <= tx_trig_mode[((n*3)+2):(n*3)];
      if (tx_trig_mode_d[((n*3)+2):(n*3)] == tx_trig_mode[((n*3)+2):(n*3)]) begin
        case (tx_trig_mode_d[((n*3)+2):(n*3)])
          3'd1: begin
            tx_trig_enable[n] <= tx_trig_d[n];
          end
          3'd4: begin
            if ((tx_trig[n] == 1'b1) && (tx_trig_d[n] == 1'b0)) begin
              tx_trig_enable[n] <= 1'd1;
            end
          end
          3'd5: begin
            if ((tx_trig[n] == 1'b0) && (tx_trig_d[n] == 1'b1)) begin
              tx_trig_enable[n] <= 1'd1;
            end
          end
          3'd6: begin
            if ((tx_trig[n] ^ tx_trig_d[n]) == 1'b1) begin
              tx_trig_enable[n] <= 1'd1;
            end
          end
          default: begin
            tx_trig_enable[n] <= 1'd1;
          end
        endcase
      end else begin
        tx_trig_enable[n] <= 1'd1;
      end
    end
  end

  end
  endgenerate

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_trig_ready <= 1'd0;
      tx_tvalid <= 1'd0;
      tx_tdata <= 64'd0;
      tx_active <= 1'd0;
    end else begin
      tx_trig_ready <= & tx_trig_enable;
      tx_tvalid <= tx_tready;
      if ((tx_valid == 1'd1) && (tx_tready == 1'd1)) begin
        tx_tdata <= tx_data;
      end else begin
        tx_tdata <= 64'd0;
      end
      if (tx_active == 1'd1) begin
        tx_active <= tx_mm_active;
      end else begin
        tx_active <= tx_mm_active & tx_valid;
      end
    end
  end

  // aximm interface

  assign aximm_ack_ready = 1'b1;
  assign aximm_done = (aximm_ack_data[3:0] == aximm_req_tag) ?
    (aximm_ack_valid & (~aximm_cyclic | aximm_stop)) : 1'd0;
  assign aximm_req_data[79:76] = 4'd0;
  assign aximm_req_data[75:72] = aximm_tag;
  assign aximm_req_data[71:32] = aximm_start_addr;
  assign aximm_req_data[31] = 1'b0;
  assign aximm_req_data[30] = 1'b1;
  assign aximm_req_data[29:24] = 6'd0;
  assign aximm_req_data[23] = 1'b1;
  assign aximm_req_data[22:0] = aximm_length;

  always @(negedge aximm_resetn or posedge aximm_clk) begin
    if (aximm_resetn == 1'b0) begin
      aximm_swresetn <= 1'd0;
      aximm_swreset_d <= 1'd0;
      aximm_swreset_count <= 6'd0;
    end else begin
      aximm_swresetn <= ~aximm_swreset_count[5];
      aximm_swreset_d <= aximm_swreset;
      if (aximm_swreset_count[5] == 1'd1) begin
        aximm_swreset_count <= aximm_swreset_count + 1'd1;
      end else if ((aximm_swreset == 1'd1) && (aximm_swreset_d == 1'd0)) begin
        aximm_swreset_count <= 6'h20;
      end
    end
  end

  always @(negedge aximm_swresetn or posedge aximm_clk) begin
    if (aximm_swresetn == 1'b0) begin
      aximm_start_req_d <= 1'd0;
      aximm_stop_req_d <= 1'd0;
      aximm_start <= 1'd0;
      aximm_stop <= 1'd0;
      aximm_active <= 1'd0;
      aximm_status <= 1'b1;
      aximm_slave_err <= 1'b0;
      aximm_addrdec_err <= 1'b0;
      aximm_tlast_err <= 1'b0;
      aximm_length_err <= 1'b0;
      aximm_tag <= 4'd0;
      aximm_req_tag <= 4'd0;
      aximm_req_valid <= 1'b0;
      aximm_req_active <= 1'b0;
      aximm_qcnt <= 4'd0;
    end else begin
      aximm_start_req_d <= aximm_start_req;
      aximm_stop_req_d <= aximm_stop_req;
      aximm_start <= aximm_start_req & ~aximm_start_req_d;
      if (aximm_stop == 1'b1) begin
        aximm_stop <= ~aximm_done;
      end else begin
        aximm_stop <= aximm_stop_req & ~aximm_stop_req_d;
      end
      if (aximm_active == 1'b1) begin
        aximm_active <= ~aximm_done;
      end else begin
        aximm_active <= aximm_start;
      end
      if (aximm_ack_valid == 1'b1) begin
        aximm_status <= aximm_status & aximm_ack_data[7];
        aximm_slave_err <= aximm_slave_err | aximm_ack_data[6];
        aximm_addrdec_err <= aximm_addrdec_err | aximm_ack_data[5];
        aximm_tlast_err <= aximm_tlast_err | aximm_ack_data[4];
        aximm_length_err <= aximm_length_err | aximm_err;
      end else if (aximm_start == 1'b1) begin
        aximm_status <= 1'b1;
        aximm_slave_err <= 1'b0;
        aximm_addrdec_err <= 1'b0;
        aximm_tlast_err <= 1'b0;
        aximm_length_err <= 1'b0;
      end
      if (aximm_req_active == 1'b1) begin
        if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
          aximm_tag <= aximm_tag + 1'b1;
          aximm_req_tag <= aximm_tag;
        end
        aximm_req_valid <= (aximm_qcnt > aximm_qthreshold) ? 1'b0 :
          (aximm_cyclic & ~aximm_stop);
        aximm_req_active <= aximm_cyclic & ~aximm_stop;
      end else if (aximm_start == 1'd1) begin
        aximm_tag <= 4'd0;
        aximm_req_tag <= 4'd0;
        aximm_req_valid <= 1'b1;
        aximm_req_active <= 1'b1;
      end
      if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
        aximm_qcnt <= aximm_qcnt + 1'b1;
      end else if ((aximm_ack_valid == 1'b1) && (aximm_ack_ready == 1'b1)) begin
        aximm_qcnt <= aximm_qcnt - 1'b1;
      end
    end
  end

  // instantiations

  cdc #(.DATA_WIDTH(4)) i_cdc_tx (
    .src_data                       (trig),
    .dest_resetn                    (tx_resetn),
    .dest_clk                       (tx_clk),
    .dest_data                      (tx_trig));

  cdc_cntrl #(.DATA_WIDTH(12)) i_cdc_cntrl_lite_tx (
    .src_resetn                     (axilite_resetn),
    .src_clk                        (axilite_clk),
    .src_data                       (axilite_trig_mode),
    .dest_resetn                    (tx_resetn),
    .dest_clk                       (tx_clk),
    .dest_data                      (tx_trig_mode));

  cdc_cntrl #(.DATA_WIDTH(1)) i_cdc_cntrl_mm_tx (
    .src_resetn                     (aximm_resetn),
    .src_clk                        (aximm_clk),
    .src_data                       (aximm_active),
    .dest_resetn                    (tx_resetn),
    .dest_clk                       (tx_clk),
    .dest_data                      (tx_mm_active));

  cdc_cntrl #(.DATA_WIDTH(71)) i_cdc_cntrl_aximm (
    .src_resetn                     (axilite_resetn),
    .src_clk                        (axilite_clk),
    .src_data                       ({axilite_swreset,
                                      axilite_start,
                                      axilite_stop,
                                      axilite_cyclic,
                                      axilite_qthreshold,
                                      axilite_start_addr,
                                      axilite_length}),
    .dest_resetn                    (aximm_resetn),
    .dest_clk                       (aximm_clk),
    .dest_data                      ({aximm_swreset,
                                      aximm_start_req,
                                      aximm_stop_req,
                                      aximm_cyclic,
                                      aximm_qthreshold,
                                      aximm_start_addr,
                                      aximm_length}));

  cdc_event #(.DATA_WIDTH(1)) i_cdc_event_axilite (
    .src_resetn                     (tx_resetn),
    .src_clk                        (tx_clk),
    .src_data                       (tx_underflow),
    .dest_resetn                    (axilite_resetn),
    .dest_clk                       (axilite_clk),
    .dest_data                      (axilite_underflow));

  cdc_cntrl #(.DATA_WIDTH(7)) i_cdc_cntrl_axilite (
    .src_resetn                     (aximm_resetn),
    .src_clk                        (aximm_clk),
    .src_data                       ({aximm_swresetn,
                                      aximm_active,
                                      aximm_status,
                                      aximm_length_err,
                                      aximm_tlast_err,
                                      aximm_addrdec_err,
                                      aximm_slave_err}),
    .dest_resetn                    (axilite_resetn),
    .dest_clk                       (axilite_clk),
    .dest_data                      ({axilite_swreset_clr,
                                      axilite_busy,
                                      axilite_status,
                                      axilite_length_err,
                                      axilite_tlast_err,
                                      axilite_addrdec_err,
                                      axilite_slave_err}));

  axi_txdma_es i_es (
    .s_axis_aresetn                 (aximm_resetn),
    .m_axis_aresetn                 (tx_resetn),
    .s_axis_aclk                    (aximm_clk),
    .s_axis_tvalid                  (aximm_valid),
    .s_axis_tready                  (aximm_ready),
    .s_axis_tdata                   (aximm_data),
    .m_axis_aclk                    (tx_clk),
    .m_axis_tvalid                  (tx_valid),
    .m_axis_tready                  (tx_ready),
    .m_axis_tdata                   (tx_data),
    .axis_data_count                (),
    .axis_wr_data_count             (),
    .axis_rd_data_count             ());

  axi_txdma_mm i_mm (
    .m_axi_mm2s_aclk                (aximm_clk),
    .m_axi_mm2s_aresetn             (aximm_swresetn),
    .mm2s_err                       (aximm_err),
    .m_axis_mm2s_cmdsts_aclk        (aximm_clk),
    .m_axis_mm2s_cmdsts_aresetn     (aximm_swresetn),
    .s_axis_mm2s_cmd_tvalid         (aximm_req_valid),
    .s_axis_mm2s_cmd_tready         (aximm_req_ready),
    .s_axis_mm2s_cmd_tdata          (aximm_req_data),
    .m_axis_mm2s_sts_tvalid         (aximm_ack_valid),
    .m_axis_mm2s_sts_tready         (aximm_ack_ready),
    .m_axis_mm2s_sts_tdata          (aximm_ack_data),
    .m_axis_mm2s_sts_tkeep          (),
    .m_axis_mm2s_sts_tlast          (),
    .m_axi_mm2s_araddr              (aximm_araddr),
    .m_axi_mm2s_arlen               (aximm_arlen),
    .m_axi_mm2s_arsize              (aximm_arsize),
    .m_axi_mm2s_arburst             (aximm_arburst),
    .m_axi_mm2s_arprot              (aximm_arprot),
    .m_axi_mm2s_arcache             (aximm_arcache),
    .m_axi_mm2s_aruser              (aximm_aruser),
    .m_axi_mm2s_arvalid             (aximm_arvalid),
    .m_axi_mm2s_arready             (aximm_arready),
    .m_axi_mm2s_rdata               (aximm_rdata),
    .m_axi_mm2s_rresp               (aximm_rresp),
    .m_axi_mm2s_rlast               (aximm_rlast),
    .m_axi_mm2s_rvalid              (aximm_rvalid),
    .m_axi_mm2s_rready              (aximm_rready),
    .m_axis_mm2s_tdata              (aximm_data),
    .m_axis_mm2s_tkeep              (),
    .m_axis_mm2s_tlast              (),
    .m_axis_mm2s_tvalid             (aximm_valid),
    .m_axis_mm2s_tready             (aximm_ready));

  axi_txdma_regs i_regs (
    .axilite_version                (VERSION),
    .axilite_instance_id            (INSTANCE_ID),
    .axilite_scratch                (),
    .axilite_timer                  (),
    .axilite_data_width             ('d512),
    .axilite_swreset                (axilite_swreset),
    .axilite_swreset_clr            (axilite_swreset_clr),
    .axilite_start                  (axilite_start),
    .axilite_start_clr              (~axilite_busy),
    .axilite_stop                   (axilite_stop),
    .axilite_stop_clr               (~axilite_busy),
    .axilite_cyclic                 (axilite_cyclic),
    .axilite_qthreshold             (axilite_qthreshold),
    .axilite_busy                   (axilite_busy),
    .axilite_status                 (axilite_status),
    .axilite_underflow              (axilite_underflow),
    .axilite_length_error           (axilite_length_err),
    .axilite_tlast_error            (axilite_tlast_err),
    .axilite_addrdec_error          (axilite_addrdec_err),
    .axilite_slave_error            (axilite_slave_err),
    .axilite_start_addr_low         (axilite_start_addr[31:0]),
    .axilite_start_addr_high        (axilite_start_addr[39:32]),
    .axilite_length                 (axilite_length),
    .axilite_trigger_mode_0         (axilite_trig_mode[2:0]),
    .axilite_trigger_mode_1         (axilite_trig_mode[5:3]),
    .axilite_trigger_mode_2         (axilite_trig_mode[8:6]),
    .axilite_trigger_mode_3         (axilite_trig_mode[11:9]),
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

  axilite_slave_if #(
    .AXI_ADDRESS_WIDTH              (12))
  i_axilite_if (
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_awvalid                (axilite_awvalid),
    .axilite_awaddr                 (axilite_awaddr),
    .axilite_awready                (axilite_awready),
    .axilite_wvalid                 (axilite_wvalid),
    .axilite_wdata                  (axilite_wdata),
    .axilite_wready                 (axilite_wready),
    .axilite_bvalid                 (axilite_bvalid),
    .axilite_bresp                  (axilite_bresp),
    .axilite_bready                 (axilite_bready),
    .axilite_arvalid                (axilite_arvalid),
    .axilite_araddr                 (axilite_araddr),
    .axilite_arready                (axilite_arready),
    .axilite_rvalid                 (axilite_rvalid),
    .axilite_rresp                  (axilite_rresp),
    .axilite_rdata                  (axilite_rdata),
    .axilite_rready                 (axilite_rready),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
