// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       System Id
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_sysid #(

  parameter integer INSTANCE_ID = 0,
  parameter integer STATUS_WIDTH = 8,
  parameter integer GPIO_IN_WIDTH = 8,
  parameter integer GPIO_OUT_WIDTH = 8,
  parameter [ 31:0] SYS_VERSION = 32'd0,
  parameter [ 63:0] SYS_PARAMS = 64'd0,
  parameter [255:0] SYS_INFO_0 = 256'd0,
  parameter [255:0] SYS_INFO_1 = 256'd0,
  parameter [ 63:0] SYS_BUILD = 64'd0) (

  // system interface

  input   wire  [(STATUS_WIDTH-1):0]      status,
  input   wire  [(GPIO_IN_WIDTH-1):0]     gpio_in,
  output  wire  [(GPIO_OUT_WIDTH-1):0]    gpio_out,

  // axilite interface

  input   wire                            axilite_clk,
  input   wire                            axilite_resetn,
  input   wire                            axilite_awvalid,
  input   wire  [ 11:0]                   axilite_awaddr,
  output  wire                            axilite_awready,
  input   wire                            axilite_wvalid,
  input   wire  [ 31:0]                   axilite_wdata,
  output  wire                            axilite_wready,
  output  wire                            axilite_bvalid,
  output  wire  [  1:0]                   axilite_bresp,
  input   wire                            axilite_bready,
  input   wire                            axilite_arvalid,
  input   wire  [ 11:0]                   axilite_araddr,
  output  wire                            axilite_arready,
  output  wire                            axilite_rvalid,
  output  wire  [  1:0]                   axilite_rresp,
  output  wire  [ 31:0]                   axilite_rdata,
  input   wire                            axilite_rready,
  output  wire                            interrupt);

  // constants

  localparam  VERSION = {8'd0, 8'd6, 8'd1, 8'd0};

  // internal signals
 
  wire          [  7:0]                   axilite_gpio_in_8;
  wire          [(GPIO_IN_WIDTH-1):0]     axilite_gpio_in;
  wire          [  7:0]                   axilite_gpio_out;
  wire          [  7:0]                   axilite_gpio_in_mask;
  wire          [(STATUS_WIDTH-1):0]      axilite_status;
  wire          [  7:0]                   axilite_status_state;
  wire          [  7:0]                   axilite_status_event;
  wire          [  7:0]                   axilite_status_event_mask;
  wire                                    axilite_wrreq;
  wire          [ 11:0]                   axilite_wraddr;
  wire          [ 31:0]                   axilite_wrdata;
  wire                                    axilite_wrack;
  wire                                    axilite_rdreq;
  wire          [ 11:0]                   axilite_rdaddr;
  wire          [ 31:0]                   axilite_rddata;
  wire                                    axilite_rdack;

  // interrupts

  assign interrupt = | ((axilite_status_event & ~axilite_status_event_mask) |
    (axilite_gpio_in_8 & ~axilite_gpio_in_mask));

  // status inputs

  generate
  if (STATUS_WIDTH >= 8) begin

  assign axilite_status_state = axilite_status[7:0];
  assign axilite_status_event = ~axilite_status[7:0];

  end else begin

  assign axilite_status_state[7:STATUS_WIDTH] = 'd0;
  assign axilite_status_state[(STATUS_WIDTH-1):0] = axilite_status;
  assign axilite_status_event[7:STATUS_WIDTH] = 'd0;
  assign axilite_status_event[(STATUS_WIDTH-1):0] = ~axilite_status;

  end
  endgenerate

  // gpio in (max width 8)

  generate
  if (GPIO_IN_WIDTH >= 8) begin

  assign axilite_gpio_in_8 = axilite_gpio_in[7:0];

  end else begin

  assign axilite_gpio_in_8[7:GPIO_IN_WIDTH] = 'd0;
  assign axilite_gpio_in_8[(GPIO_IN_WIDTH-1):0] = axilite_gpio_in;

  end
  endgenerate

  // gpio out (max width 8)

  generate
  if (GPIO_OUT_WIDTH > 8) begin

  assign gpio_out[(GPIO_OUT_WIDTH-1):8] = 'd0;
  assign gpio_out[7:0] = axilite_gpio_out;

  end else begin

  assign gpio_out = axilite_gpio_out[(GPIO_OUT_WIDTH-1):0];

  end
  endgenerate

  // instantiations

  cdc #(.DATA_WIDTH(STATUS_WIDTH + GPIO_IN_WIDTH)) i_cdc (
    .src_data                       ({status, gpio_in}),
    .dest_resetn                    (axilite_resetn),
    .dest_clk                       (axilite_clk),
    .dest_data                      ({axilite_status, axilite_gpio_in}));

  axi_sysid_regs i_regs (
    .axilite_version                (VERSION),
    .axilite_instance_id            (INSTANCE_ID),
    .axilite_scratch                (),
    .axilite_timer                  (),
    .axilite_gpio_in                (axilite_gpio_in_8),
    .axilite_gpio_out               (axilite_gpio_out),
    .axilite_gpio_in_mask           (axilite_gpio_in_mask),
    .axilite_status_state           (axilite_status_state),
    .axilite_status_event           (axilite_status_event),
    .axilite_status_event_mask      (axilite_status_event_mask),
    .axilite_sys_version            (SYS_VERSION),
    .axilite_sys_params_0           (SYS_PARAMS[63:32]),
    .axilite_sys_params_1           (SYS_PARAMS[31:0]),
    .axilite_sys_build_0            (SYS_BUILD[63:32]),
    .axilite_sys_build_1            (SYS_BUILD[31:0]),
    .axilite_sys_info_0_0           (SYS_INFO_0[((32*7)+31):(32*7)]),
    .axilite_sys_info_0_1           (SYS_INFO_0[((32*6)+31):(32*6)]),
    .axilite_sys_info_0_2           (SYS_INFO_0[((32*5)+31):(32*5)]),
    .axilite_sys_info_0_3           (SYS_INFO_0[((32*4)+31):(32*4)]),
    .axilite_sys_info_0_4           (SYS_INFO_0[((32*3)+31):(32*3)]),
    .axilite_sys_info_0_5           (SYS_INFO_0[((32*2)+31):(32*2)]),
    .axilite_sys_info_0_6           (SYS_INFO_0[((32*1)+31):(32*1)]),
    .axilite_sys_info_0_7           (SYS_INFO_0[((32*0)+31):(32*0)]),
    .axilite_sys_info_1_0           (SYS_INFO_1[((32*7)+31):(32*7)]),
    .axilite_sys_info_1_1           (SYS_INFO_1[((32*6)+31):(32*6)]),
    .axilite_sys_info_1_2           (SYS_INFO_1[((32*5)+31):(32*5)]),
    .axilite_sys_info_1_3           (SYS_INFO_1[((32*4)+31):(32*4)]),
    .axilite_sys_info_1_4           (SYS_INFO_1[((32*3)+31):(32*3)]),
    .axilite_sys_info_1_5           (SYS_INFO_1[((32*2)+31):(32*2)]),
    .axilite_sys_info_1_6           (SYS_INFO_1[((32*1)+31):(32*1)]),
    .axilite_sys_info_1_7           (SYS_INFO_1[((32*0)+31):(32*0)]),
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

  axilite_slave_if #(
    .AXI_ADDRESS_WIDTH              (12))
  i_axilite_if (
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_awvalid                (axilite_awvalid),
    .axilite_awaddr                 (axilite_awaddr),
    .axilite_awready                (axilite_awready),
    .axilite_wvalid                 (axilite_wvalid),
    .axilite_wdata                  (axilite_wdata),
    .axilite_wready                 (axilite_wready),
    .axilite_bvalid                 (axilite_bvalid),
    .axilite_bresp                  (axilite_bresp),
    .axilite_bready                 (axilite_bready),
    .axilite_arvalid                (axilite_arvalid),
    .axilite_araddr                 (axilite_araddr),
    .axilite_arready                (axilite_arready),
    .axilite_rvalid                 (axilite_rvalid),
    .axilite_rresp                  (axilite_rresp),
    .axilite_rdata                  (axilite_rdata),
    .axilite_rready                 (axilite_rready),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
