// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       Receive DMA Core
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_rxdma #(

  parameter integer INSTANCE_ID = 0,
  parameter integer AXIS_DATA_WIDTH = 64) (

  // ssi interface

  input   wire                            rx_clk,
  input   wire                            rx_resetn,
  input   wire                            rx_tvalid,
  input   wire  [(AXIS_DATA_WIDTH-1):0]   rx_tdata,

  // asynchronous events

  input   wire  [  3:0]                   trig,

  // axi master interface

  input   wire                            aximm_clk,
  input   wire                            aximm_resetn,
  output  wire  [ 39:0]                   aximm_awaddr,
  output  wire  [  7:0]                   aximm_awlen,
  output  wire  [  2:0]                   aximm_awsize,
  output  wire  [  1:0]                   aximm_awburst,
  output  wire  [  2:0]                   aximm_awprot,
  output  wire  [  3:0]                   aximm_awcache,
  output  wire                            aximm_awvalid,
  input   wire                            aximm_awready,
  output  wire  [511:0]                   aximm_wdata,
  output  wire  [ 63:0]                   aximm_wstrb,
  output  wire                            aximm_wlast,
  output  wire                            aximm_wvalid,
  input   wire                            aximm_wready,
  input   wire  [  1:0]                   aximm_bresp,
  input   wire                            aximm_bvalid,
  output  wire                            aximm_bready,

  // axilite interface

  input   wire                            axilite_clk,
  input   wire                            axilite_resetn,
  input   wire                            axilite_awvalid,
  input   wire  [ 11:0]                   axilite_awaddr,
  output  wire                            axilite_awready,
  input   wire                            axilite_wvalid,
  input   wire  [ 31:0]                   axilite_wdata,
  output  wire                            axilite_wready,
  output  wire                            axilite_bvalid,
  output  wire  [  1:0]                   axilite_bresp,
  input   wire                            axilite_bready,
  input   wire                            axilite_arvalid,
  input   wire  [ 11:0]                   axilite_araddr,
  output  wire                            axilite_arready,
  output  wire                            axilite_rvalid,
  output  wire  [  1:0]                   axilite_rresp,
  output  wire  [ 31:0]                   axilite_rdata,
  input   wire                            axilite_rready,
  output  wire                            interrupt);

  // constants

  localparam  VERSION = {8'd0, 8'd6, 8'd1, 8'd0};

  // internal registers
   
  reg           [  3:0]                   rx_trig_d = 'd0;
  reg           [ 11:0]                   rx_trig_mode_d = 'd0;
  reg           [  3:0]                   rx_trig_enable = 'd0;
  reg                                     rx_trig_valid = 'd0;
  reg                                     rx_valid = 'd0;
  reg           [(AXIS_DATA_WIDTH-1):0]   rx_data = 'd0;
  reg                                     aximm_swresetn = 'd0;
  reg                                     aximm_swreset_d = 'd0;
  reg           [  5:0]                   aximm_swreset_count = 'd0;
  reg                                     aximm_start_req_d = 'd0;
  reg                                     aximm_stop_req_d = 'd0;
  reg                                     aximm_start = 'd0;
  reg                                     aximm_stop = 'd0;
  reg                                     aximm_active = 'd0;
  reg                                     aximm_enable = 'd0;
  reg                                     aximm_status = 'd0;
  reg                                     aximm_slave_err = 'd0;
  reg                                     aximm_addrdec_err = 'd0;
  reg                                     aximm_tlast_err = 'd0;
  reg                                     aximm_length_err = 'd0;
  reg           [  3:0]                   aximm_tag = 'd0;
  reg           [  3:0]                   aximm_req_tag = 'd0;
  reg                                     aximm_req_valid = 'd0;
  reg                                     aximm_req_active = 'd0;
  reg           [  3:0]                   aximm_qcnt = 'd0;
  reg           [ 22:0]                   aximm_rdcnt = 'd0;

  // internal signals
 
  wire          [  3:0]                   rx_trig;
  wire                                    rx_enable;
  wire          [ 11:0]                   rx_trig_mode;
  wire                                    rx_overflow;
  wire                                    rx_ready;
  wire                                    rx_es_valid;
  wire                                    rx_es_ready;
  wire          [ 63:0]                   rx_es_data;
  wire                                    aximm_int_valid;
  wire                                    aximm_int_ready;
  wire                                    aximm_ack_ready;
  wire                                    aximm_done;
  wire                                    aximm_last;
  wire          [ 79:0]                   aximm_req_data;
  wire                                    aximm_swreset;
  wire                                    aximm_start_req;
  wire                                    aximm_stop_req;
  wire                                    aximm_cyclic;
  wire          [  3:0]                   aximm_qthreshold;
  wire          [ 39:0]                   aximm_start_addr;
  wire          [ 22:0]                   aximm_length;
  wire                                    aximm_valid;
  wire          [ 63:0]                   aximm_data;
  wire                                    aximm_err;
  wire                                    aximm_req_ready;
  wire                                    aximm_ack_valid;
  wire          [  7:0]                   aximm_ack_data;
  wire                                    aximm_ready;
  wire                                    axilite_overflow;
  wire                                    axilite_swreset_clr;
  wire                                    axilite_busy;
  wire                                    axilite_status;
  wire                                    axilite_length_err;
  wire                                    axilite_tlast_err;
  wire                                    axilite_addrdec_err;
  wire                                    axilite_slave_err;
  wire                                    axilite_swreset;
  wire                                    axilite_start;
  wire                                    axilite_stop;
  wire                                    axilite_cyclic;
  wire          [  3:0]                   axilite_qthreshold;
  wire          [ 39:0]                   axilite_start_addr;
  wire          [ 22:0]                   axilite_length;
  wire          [ 11:0]                   axilite_trig_mode;
  wire                                    axilite_wrreq;
  wire          [ 11:0]                   axilite_wraddr;
  wire          [ 31:0]                   axilite_wrdata;
  wire                                    axilite_wrack;
  wire                                    axilite_rdreq;
  wire          [ 11:0]                   axilite_rdaddr;
  wire          [ 31:0]                   axilite_rddata;
  wire                                    axilite_rdack;

  // device interface

  assign interrupt = axilite_busy;
  assign rx_overflow = rx_enable & rx_valid & ~rx_ready;

  generate
  for (genvar n = 0; n < 4; n = n + 1) begin: g_rx_trig

  always @(negedge rx_resetn or posedge rx_clk) begin
    if (rx_resetn == 1'b0) begin
      rx_trig_d[n] <= 1'd0;
      rx_trig_mode_d[((n*3)+2):(n*3)] <= 2'd0;
      rx_trig_enable[n] <= 1'd0;
    end else begin
      rx_trig_d[n] <= rx_trig[n];
      rx_trig_mode_d[((n*3)+2):(n*3)] <= rx_trig_mode[((n*3)+2):(n*3)];
      if (rx_trig_mode_d[((n*3)+2):(n*3)] == rx_trig_mode[((n*3)+2):(n*3)]) begin
        case (rx_trig_mode_d[((n*3)+2):(n*3)])
          3'd1: begin
            rx_trig_enable[n] <= rx_trig_d[n];
          end
          3'd4: begin
            if ((rx_trig[n] == 1'b1) && (rx_trig_d[n] == 1'b0)) begin
              rx_trig_enable[n] <= 1'd1;
            end
          end
          3'd5: begin
            if ((rx_trig[n] == 1'b0) && (rx_trig_d[n] == 1'b1)) begin
              rx_trig_enable[n] <= 1'd1;
            end
          end
          3'd6: begin
            if ((rx_trig[n] ^ rx_trig_d[n]) == 1'b1) begin
              rx_trig_enable[n] <= 1'd1;
            end
          end
          default: begin
            rx_trig_enable[n] <= 1'd1;
          end
        endcase
      end else begin
        rx_trig_enable[n] <= 1'd1;
      end
    end
  end

  end
  endgenerate

  always @(negedge rx_resetn or posedge rx_clk) begin
    if (rx_resetn == 1'b0) begin
      rx_trig_valid <= 1'd0;
      rx_valid <= 1'd0;
      rx_data <= 'd0;
    end else begin
      rx_trig_valid <= & rx_trig_enable;
      rx_valid <= rx_tvalid & rx_trig_valid;
      rx_data <= rx_tdata;
    end
  end

  // aximm interface

  assign aximm_valid = aximm_int_valid & aximm_enable;
  assign aximm_int_ready = aximm_ready | ~aximm_active;
  assign aximm_ack_ready = 1'b1;
  assign aximm_done = (aximm_ack_data[3:0] == aximm_req_tag) ?
    (aximm_ack_valid & (~aximm_cyclic | aximm_stop)) : 1'd0;
  assign aximm_last = (aximm_rdcnt >= aximm_length) ? aximm_active : 1'b0;
  assign aximm_req_data[79:76] = 4'd0;
  assign aximm_req_data[75:72] = aximm_tag;
  assign aximm_req_data[71:32] = aximm_start_addr;
  assign aximm_req_data[31] = 1'b0;
  assign aximm_req_data[30] = 1'b1;
  assign aximm_req_data[29:24] = 6'd0;
  assign aximm_req_data[23] = 1'b1;
  assign aximm_req_data[22:0] = aximm_length;

  always @(negedge aximm_resetn or posedge aximm_clk) begin
    if (aximm_resetn == 1'b0) begin
      aximm_swresetn <= 1'd0;
      aximm_swreset_d <= 1'd0;
      aximm_swreset_count <= 6'd0;
    end else begin
      aximm_swresetn <= ~aximm_swreset_count[5];
      aximm_swreset_d <= aximm_swreset;
      if (aximm_swreset_count[5] == 1'd1) begin
        aximm_swreset_count <= aximm_swreset_count + 1'd1;
      end else if ((aximm_swreset == 1'd1) && (aximm_swreset_d == 1'd0)) begin
        aximm_swreset_count <= 6'h20;
      end
    end
  end

  always @(negedge aximm_swresetn or posedge aximm_clk) begin
    if (aximm_swresetn == 1'b0) begin
      aximm_start_req_d <= 1'd0;
      aximm_stop_req_d <= 1'd0;
      aximm_start <= 1'd0;
      aximm_stop <= 1'd0;
      aximm_active <= 1'd0;
      aximm_enable <= 1'd0;
      aximm_status <= 1'b1;
      aximm_slave_err <= 1'b0;
      aximm_addrdec_err <= 1'b0;
      aximm_tlast_err <= 1'b0;
      aximm_length_err <= 1'b0;
      aximm_tag <= 4'd0;
      aximm_req_tag <= 4'd0;
      aximm_req_valid <= 1'b0;
      aximm_req_active <= 1'b0;
      aximm_qcnt <= 4'd0;
      aximm_rdcnt <= 'd0;
    end else begin
      aximm_start_req_d <= aximm_start_req;
      aximm_stop_req_d <= aximm_stop_req;
      aximm_start <= aximm_start_req & ~aximm_start_req_d;
      if (aximm_stop == 1'b1) begin
        aximm_stop <= ~aximm_done;
      end else begin
        aximm_stop <= aximm_stop_req & ~aximm_stop_req_d;
      end
      if (aximm_active == 1'b1) begin
        aximm_active <= ~aximm_done;
        if ((aximm_valid == 1'd1) && (aximm_ready == 1'd1) &&
          (aximm_last == 1'b1)) begin
          aximm_enable <= aximm_cyclic & ~aximm_stop;
        end
      end else begin
        aximm_active <= aximm_start;
        aximm_enable <= aximm_start;
      end
      if (aximm_ack_valid == 1'b1) begin
        aximm_status <= aximm_status & aximm_ack_data[7];
        aximm_slave_err <= aximm_slave_err | aximm_ack_data[6];
        aximm_addrdec_err <= aximm_addrdec_err | aximm_ack_data[5];
        aximm_tlast_err <= aximm_tlast_err | aximm_ack_data[4];
        aximm_length_err <= aximm_length_err | aximm_err;
      end else if (aximm_start == 1'b1) begin
        aximm_status <= 1'b1;
        aximm_slave_err <= 1'b0;
        aximm_addrdec_err <= 1'b0;
        aximm_tlast_err <= 1'b0;
        aximm_length_err <= 1'b0;
      end
      if (aximm_req_active == 1'b1) begin
        if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
          aximm_tag <= aximm_tag + 1'b1;
          aximm_req_tag <= aximm_tag;
        end
        aximm_req_valid <= (aximm_qcnt > aximm_qthreshold) ? 1'b0 :
          (aximm_cyclic & ~aximm_stop);
        aximm_req_active <= aximm_cyclic & ~aximm_stop;
      end else if (aximm_start == 1'd1) begin
        aximm_tag <= 4'd0;
        aximm_req_tag <= 4'd0;
        aximm_req_valid <= 1'b1;
        aximm_req_active <= 1'b1;
      end
      if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
        aximm_qcnt <= aximm_qcnt + 1'b1;
      end else if ((aximm_ack_valid == 1'b1) && (aximm_ack_ready == 1'b1)) begin
        aximm_qcnt <= aximm_qcnt - 1'b1;
      end
      if (aximm_active == 1'b0) begin
        aximm_rdcnt <= 23'd8;
      end else if ((aximm_valid == 1'd1) && (aximm_ready == 1'd1)) begin
        if (aximm_rdcnt >= aximm_length) begin
          aximm_rdcnt <= 23'd8;
        end else begin
          aximm_rdcnt <= aximm_rdcnt + 4'd8;
        end
      end
    end
  end

  // instantiations

  cdc #(.DATA_WIDTH(5)) i_cdc_rx (
    .src_data                       ({trig,
                                      aximm_enable}),
    .dest_resetn                    (rx_resetn),
    .dest_clk                       (rx_clk),
    .dest_data                      ({rx_trig,
                                      rx_enable}));

  cdc_cntrl #(.DATA_WIDTH(12)) i_cdc_cntrl_rx (
    .src_resetn                     (axilite_resetn),
    .src_clk                        (axilite_clk),
    .src_data                       (axilite_trig_mode),
    .dest_resetn                    (rx_resetn),
    .dest_clk                       (rx_clk),
    .dest_data                      (rx_trig_mode));

  cdc_cntrl #(.DATA_WIDTH(71)) i_cdc_cntrl_aximm (
    .src_resetn                     (axilite_resetn),
    .src_clk                        (axilite_clk),
    .src_data                       ({axilite_swreset,
                                      axilite_start,
                                      axilite_stop,
                                      axilite_cyclic,
                                      axilite_qthreshold,
                                      axilite_start_addr,
                                      axilite_length}),
    .dest_resetn                    (aximm_resetn),
    .dest_clk                       (aximm_clk),
    .dest_data                      ({aximm_swreset,
                                      aximm_start_req,
                                      aximm_stop_req,
                                      aximm_cyclic,
                                      aximm_qthreshold,
                                      aximm_start_addr,
                                      aximm_length}));

  cdc_event #(.DATA_WIDTH(1)) i_cdc_event_axilite (
    .src_resetn                     (rx_resetn),
    .src_clk                        (rx_clk),
    .src_data                       (rx_overflow),
    .dest_resetn                    (axilite_resetn),
    .dest_clk                       (axilite_clk),
    .dest_data                      (axilite_overflow));

  cdc_cntrl #(.DATA_WIDTH(7)) i_cdc_cntrl_axilite (
    .src_resetn                     (aximm_resetn),
    .src_clk                        (aximm_clk),
    .src_data                       ({aximm_swresetn,
                                      aximm_active,
                                      aximm_status,
                                      aximm_length_err,
                                      aximm_tlast_err,
                                      aximm_addrdec_err,
                                      aximm_slave_err}),
    .dest_resetn                    (axilite_resetn),
    .dest_clk                       (axilite_clk),
    .dest_data                      ({axilite_swreset_clr,
                                      axilite_busy,
                                      axilite_status,
                                      axilite_length_err,
                                      axilite_tlast_err,
                                      axilite_addrdec_err,
                                      axilite_slave_err}));

  generate
  if (AXIS_DATA_WIDTH == 8) begin

  axi_rxdma_dw i_dw (
    .aresetn                        (rx_resetn),
    .aclk                           (rx_clk),
    .s_axis_tvalid                  (rx_valid),
    .s_axis_tready                  (rx_ready),
    .s_axis_tdata                   (rx_data),
    .m_axis_tvalid                  (rx_es_valid),
    .m_axis_tready                  (rx_es_ready),
    .m_axis_tdata                   (rx_es_data));

  end else begin

  assign rx_es_valid = rx_valid;
  assign rx_es_data = rx_data;
  assign rx_ready = rx_es_ready;

  end
  endgenerate

  axi_rxdma_es i_es (
    .s_axis_aresetn                 (rx_resetn),
    .m_axis_aresetn                 (aximm_swresetn),
    .s_axis_aclk                    (rx_clk),
    .s_axis_tvalid                  (rx_es_valid),
    .s_axis_tready                  (rx_es_ready),
    .s_axis_tdata                   (rx_es_data),
    .m_axis_aclk                    (aximm_clk),
    .m_axis_tvalid                  (aximm_int_valid),
    .m_axis_tready                  (aximm_int_ready),
    .m_axis_tdata                   (aximm_data),
    .axis_data_count                (),
    .axis_wr_data_count             (),
    .axis_rd_data_count             ());

  axi_rxdma_mm i_mm (
    .m_axi_s2mm_aclk                (aximm_clk),
    .m_axi_s2mm_aresetn             (aximm_swresetn),
    .s2mm_err                       (aximm_err),
    .m_axis_s2mm_cmdsts_awclk       (aximm_clk),
    .m_axis_s2mm_cmdsts_aresetn     (aximm_swresetn),
    .s_axis_s2mm_cmd_tvalid         (aximm_req_valid),
    .s_axis_s2mm_cmd_tready         (aximm_req_ready),
    .s_axis_s2mm_cmd_tdata          (aximm_req_data),
    .m_axis_s2mm_sts_tvalid         (aximm_ack_valid),
    .m_axis_s2mm_sts_tready         (aximm_ack_ready),
    .m_axis_s2mm_sts_tdata          (aximm_ack_data),
    .m_axis_s2mm_sts_tkeep          (),
    .m_axis_s2mm_sts_tlast          (),
    .m_axi_s2mm_awaddr              (aximm_awaddr),
    .m_axi_s2mm_awlen               (aximm_awlen),
    .m_axi_s2mm_awsize              (aximm_awsize),
    .m_axi_s2mm_awburst             (aximm_awburst),
    .m_axi_s2mm_awprot              (aximm_awprot),
    .m_axi_s2mm_awcache             (aximm_awcache),
    .m_axi_s2mm_awuser              (),
    .m_axi_s2mm_awvalid             (aximm_awvalid),
    .m_axi_s2mm_awready             (aximm_awready),
    .m_axi_s2mm_wdata               (aximm_wdata),
    .m_axi_s2mm_wstrb               (aximm_wstrb),
    .m_axi_s2mm_wlast               (aximm_wlast),
    .m_axi_s2mm_wvalid              (aximm_wvalid),
    .m_axi_s2mm_wready              (aximm_wready),
    .m_axi_s2mm_bresp               (aximm_bresp),
    .m_axi_s2mm_bvalid              (aximm_bvalid),
    .m_axi_s2mm_bready              (aximm_bready),
    .s_axis_s2mm_tdata              (aximm_data),
    .s_axis_s2mm_tkeep              (8'hff),
    .s_axis_s2mm_tlast              (aximm_last),
    .s_axis_s2mm_tvalid             (aximm_valid),
    .s_axis_s2mm_tready             (aximm_ready));

  axi_rxdma_regs i_regs (
    .axilite_version                (VERSION),
    .axilite_instance_id            (INSTANCE_ID),
    .axilite_scratch                (),
    .axilite_timer                  (),
    .axilite_data_width             ('d512),
    .axilite_swreset                (axilite_swreset),
    .axilite_swreset_clr            (axilite_swreset_clr),
    .axilite_start                  (axilite_start),
    .axilite_start_clr              (~axilite_busy),
    .axilite_stop                   (axilite_stop),
    .axilite_stop_clr               (~axilite_busy),
    .axilite_cyclic                 (axilite_cyclic),
    .axilite_qthreshold             (axilite_qthreshold),
    .axilite_busy                   (axilite_busy),
    .axilite_status                 (axilite_status),
    .axilite_overflow               (axilite_overflow),
    .axilite_length_error           (axilite_length_err),
    .axilite_tlast_error            (axilite_tlast_err),
    .axilite_addrdec_error          (axilite_addrdec_err),
    .axilite_slave_error            (axilite_slave_err),
    .axilite_start_addr_low         (axilite_start_addr[31:0]),
    .axilite_start_addr_high        (axilite_start_addr[39:32]),
    .axilite_length                 (axilite_length),
    .axilite_trigger_mode_0         (axilite_trig_mode[2:0]),
    .axilite_trigger_mode_1         (axilite_trig_mode[5:3]),
    .axilite_trigger_mode_2         (axilite_trig_mode[8:6]),
    .axilite_trigger_mode_3         (axilite_trig_mode[11:9]),
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

  axilite_slave_if #(
    .AXI_ADDRESS_WIDTH              (12))
  i_axilite_if (
    .axilite_clk                    (axilite_clk),
    .axilite_resetn                 (axilite_resetn),
    .axilite_awvalid                (axilite_awvalid),
    .axilite_awaddr                 (axilite_awaddr),
    .axilite_awready                (axilite_awready),
    .axilite_wvalid                 (axilite_wvalid),
    .axilite_wdata                  (axilite_wdata),
    .axilite_wready                 (axilite_wready),
    .axilite_bvalid                 (axilite_bvalid),
    .axilite_bresp                  (axilite_bresp),
    .axilite_bready                 (axilite_bready),
    .axilite_arvalid                (axilite_arvalid),
    .axilite_araddr                 (axilite_araddr),
    .axilite_arready                (axilite_arready),
    .axilite_rvalid                 (axilite_rvalid),
    .axilite_rresp                  (axilite_rresp),
    .axilite_rdata                  (axilite_rdata),
    .axilite_rready                 (axilite_rready),
    .axilite_wrreq                  (axilite_wrreq),
    .axilite_wraddr                 (axilite_wraddr),
    .axilite_wrdata                 (axilite_wrdata),
    .axilite_wrack                  (axilite_wrack),
    .axilite_rdreq                  (axilite_rdreq),
    .axilite_rdaddr                 (axilite_rdaddr),
    .axilite_rddata                 (axilite_rddata),
    .axilite_rdack                  (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
