`timescale 1ns / 1ps
//-----------------------------------------------------------------------------------
// ################  Company:        Analog Devices, Inc.
// ##   ###########  Engineer:       JCE
// ##      ########
// ##         #####  Create Date:    
// ##            ##  Design Name:    
// ##         #####  Module Name:    
// ##      ########  Project Name:   
// ##   ###########  Target Devices: 
// ################  Tool versions:  
//-----------------------------------------------------------------------------------
// Description:
//
//
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
//
// Additional Comments: 
//
//-----------------------------------------------------------------------------------
// The following code is provided as-is and is intended to be used as a reference 
// only.  ADI is not responsible for supporting or providing updates.  The code may 
// contain manufacturer specific primitive elements.  You are solely responsible for 
// obtaining the proper license to use the manufacturer's design tools.
//
// All rights reserved.  
//-----------------------------------------------------------------------------------


module axi_mspi_engine
  #(parameter NUM_SLAVES = 1)
   (
    input                        clk,
    input                        reset,

    // Control 
    input                        enable,
    input                        loop,
    input                        cpol,
    input                        cpha,
    input                        man_ss_en,
    input                        mstr_inhibit,
    input                        lsb_first,

    input                        miso_slip,
    input                        four_wire,
    input [7:0]                  tri_point,
    input [15:0]                 clock_div,

    input [NUM_SLAVES-1:0]       slave_select,

    // TX FIFO
    output reg                   tx_fifo_rd,
    input [7:0]                  tx_fifo_rd_data,
    input                        tx_fifo_empty,

    // RX FIFO
    output reg                   rx_fifo_wr,
    output reg [7:0]             rx_fifo_wr_data,

    // SPI Interface
    input                        spi_clk_i,
    output wire                  spi_clk_o,
    output wire                  spi_clk_t,
    output wire [NUM_SLAVES-1:0] spi_csb_o,
    output wire [NUM_SLAVES-1:0] spi_csb_t,
    output wire                  spi_mosi_o,
    output wire                  spi_mosi_t,
    input                        spi_mosi_i,
    input                        spi_miso_i
    );



   // internal signals
   // state machine
   parameter                     S_IDLE  = 3'b001;
   parameter                     S_LOAD  = 3'b010;
   parameter                     S_SHIFT = 3'b100;
   
   reg [2:0]                     state,         next_state;
   reg [15:0] 			 clock_divider, next_clock_divider;
   reg [4:0] 			 cycle_counter, next_cycle_counter;
   reg [8:0] 			 tri_counter,   next_tri_counter;
   reg [16:0] 			 sck_sr,        next_sck_sr;
   reg 				 csb,           next_csb;
   reg [16:0] 			 mosi_sr,       next_mosi_sr;
   reg [15:0] 			 miso_sr,       next_miso_sr;
   
   reg 				 next_tx_fifo_rd;
   reg 				 next_rx_fifo_wr;
   reg [7:0] 			 next_rx_fifo_wr_data;

   reg 				 miso_i;
   reg 				 mosi_i;
   reg 				 loop_bit;

   wire [16:0]                   default_sck_sr = {cpol,~cpol,cpol,~cpol,cpol,~cpol,
                                                   cpol,~cpol,cpol,~cpol,cpol,~cpol,
                                                   cpol,~cpol,cpol,~cpol,cpol};
   
   wire [16:0]                   mosi_sr_cpha1_msb = {tx_fifo_rd_data[0],tx_fifo_rd_data[0],
                                                      tx_fifo_rd_data[1],tx_fifo_rd_data[1],
                                                      tx_fifo_rd_data[2],tx_fifo_rd_data[2],
                                                      tx_fifo_rd_data[3],tx_fifo_rd_data[3],
                                                      tx_fifo_rd_data[4],tx_fifo_rd_data[4],
                                                      tx_fifo_rd_data[5],tx_fifo_rd_data[5],
                                                      tx_fifo_rd_data[6],tx_fifo_rd_data[6],
                                                      tx_fifo_rd_data[7],tx_fifo_rd_data[7],
                                                      1'b0};
   wire [16:0]                   mosi_sr_cpha1_lsb = {tx_fifo_rd_data[7],tx_fifo_rd_data[7],
                                                      tx_fifo_rd_data[6],tx_fifo_rd_data[6],
                                                      tx_fifo_rd_data[5],tx_fifo_rd_data[5],
                                                      tx_fifo_rd_data[4],tx_fifo_rd_data[4],
                                                      tx_fifo_rd_data[3],tx_fifo_rd_data[3],
                                                      tx_fifo_rd_data[2],tx_fifo_rd_data[2],
                                                      tx_fifo_rd_data[1],tx_fifo_rd_data[1],
                                                      tx_fifo_rd_data[0],tx_fifo_rd_data[0],
                                                      1'b0};
   wire [16:0]                   mosi_sr_cpha0_msb = {1'b0,
                                                      tx_fifo_rd_data[0],tx_fifo_rd_data[0],
                                                      tx_fifo_rd_data[1],tx_fifo_rd_data[1],
                                                      tx_fifo_rd_data[2],tx_fifo_rd_data[2],
                                                      tx_fifo_rd_data[3],tx_fifo_rd_data[3],
                                                      tx_fifo_rd_data[4],tx_fifo_rd_data[4],
                                                      tx_fifo_rd_data[5],tx_fifo_rd_data[5],
                                                      tx_fifo_rd_data[6],tx_fifo_rd_data[6],
                                                      tx_fifo_rd_data[7],tx_fifo_rd_data[7]};
   wire [16:0]                   mosi_sr_cpha0_lsb = {1'b0,
                                                      tx_fifo_rd_data[7],tx_fifo_rd_data[7],
                                                      tx_fifo_rd_data[6],tx_fifo_rd_data[6],
                                                      tx_fifo_rd_data[5],tx_fifo_rd_data[5],
                                                      tx_fifo_rd_data[4],tx_fifo_rd_data[4],
                                                      tx_fifo_rd_data[3],tx_fifo_rd_data[3],
                                                      tx_fifo_rd_data[2],tx_fifo_rd_data[2],
                                                      tx_fifo_rd_data[1],tx_fifo_rd_data[1],
                                                      tx_fifo_rd_data[0],tx_fifo_rd_data[0]};
   
   wire                          miso;
   wire [7:0]                    miso_sr_slipped_msb  = {miso_sr[13],miso_sr[11],
                                                         miso_sr[9], miso_sr[7],
                                                         miso_sr[5], miso_sr[3],
                                                         miso_sr[1], miso};
   wire [7:0]                    miso_sr_slipped_lsb  = {miso,       miso_sr[1],
                                                         miso_sr[3], miso_sr[5],
                                                         miso_sr[7], miso_sr[9],
                                                         miso_sr[11],miso_sr[13]};
   wire [7:0]                    miso_sr_straight_msb = {miso_sr[14],miso_sr[12],
                                                         miso_sr[10],miso_sr[8],
                                                         miso_sr[6], miso_sr[4],
                                                         miso_sr[2], miso_sr[0]};
   wire [7:0]                    miso_sr_straight_lsb = {miso_sr[0], miso_sr[2],
                                                         miso_sr[4], miso_sr[6],
                                                         miso_sr[8], miso_sr[10],
                                                         miso_sr[12],miso_sr[14]};
   
   

   

   // ---------------------------------------------------------------------------
   // Output Signals
   // ---------------------------------------------------------------------------
   assign spi_clk_o  = sck_sr[0];
   assign spi_clk_t  = (enable & ~mstr_inhibit ) ? 1'b0 : 1'b1;
   
   assign spi_csb_o  = man_ss_en ? slave_select : ({NUM_SLAVES{csb}} | slave_select);
   assign spi_csb_t  = (enable & ~mstr_inhibit ) ? {NUM_SLAVES{1'b0}} : {NUM_SLAVES{1'b1}};

   assign spi_mosi_o = mosi_sr[0];
   assign spi_mosi_t = four_wire ? 
		       ((enable & ~mstr_inhibit) ? 1'b0 : 1'b1) : 
		       ((enable & ~mstr_inhibit & tri_counter != 8'b0 ) ? 1'b0 : 1'b1);

   

   // ---------------------------------------------------------------------------
   // Internal Signals
   // ---------------------------------------------------------------------------
   assign miso = loop ? loop_bit : four_wire ? miso_i : mosi_i;

   always @( posedge clk or posedge reset )
     if ( reset )
       begin
	  miso_i <= 1'b0;
	  mosi_i <= 1'b0;
	  loop_bit <= 1'b0;
       end
     else if ( clock_divider == 16'b0 )
       begin
	  miso_i <= spi_miso_i;
	  mosi_i <= spi_mosi_i;
	  loop_bit <= mosi_sr[0];
       end

   
   
   
   // ---------------------------------------------------------------------------
   // State Machine
   // ---------------------------------------------------------------------------
   always @( posedge clk or posedge reset )
     if ( reset )
       begin
	  state                                   <= S_IDLE;

	  cycle_counter                           <= 5'b0;
	  clock_divider                           <= 16'b0;
	  tri_counter                             <= 9'b0;
	  sck_sr                                  <= 17'b0;
	  csb                                     <= 1'b1;
	  mosi_sr                                 <= 17'b0;
	  miso_sr                                 <= 16'b0;

	  tx_fifo_rd                              <= 1'b0;
	  rx_fifo_wr                              <= 1'b0;
	  rx_fifo_wr_data                         <= 8'b0;
       end
     else
       begin
	  state                                   <= next_state;

	  cycle_counter                           <= next_cycle_counter;
	  clock_divider                           <= next_clock_divider;
	  tri_counter                             <= next_tri_counter;
	  sck_sr                                  <= next_sck_sr;
	  csb                                     <= next_csb;
	  mosi_sr                                 <= next_mosi_sr;
	  miso_sr                                 <= next_miso_sr;

	  tx_fifo_rd                              <= next_tx_fifo_rd;
	  rx_fifo_wr                              <= next_rx_fifo_wr;
	  rx_fifo_wr_data                         <= next_rx_fifo_wr_data;
       end

   always @(*)
     begin
	// assign defaults - blocking assignment intentional
	next_state                                = state;
	
	next_cycle_counter                        = cycle_counter;
	next_clock_divider                        = clock_divider;
	next_tri_counter                          = tri_counter;
	next_sck_sr                               = sck_sr;
	next_csb                                  = 1'b1;
	next_mosi_sr                              = mosi_sr;
	next_miso_sr                              = miso_sr;

	next_tx_fifo_rd                           = 1'b0;
	next_rx_fifo_wr                           = 1'b0;
	next_rx_fifo_wr_data                      = rx_fifo_wr_data;

	  
	case ( state )
	  // --------------------------------------------------------------------
	  // Idle State
	  //   Stay in Idle State until enabled and TX FIFO is not empty.  CSB is
	  //   high.  Clock is idle (based on CPOL) and MOSI is low.
	  // --------------------------------------------------------------------
	  S_IDLE:
	    begin
	       if ( cpha )
		 next_tri_counter                 = {tri_point,1'b1};
	       else
		 next_tri_counter                 = {tri_point,1'b0};
	       
	       next_sck_sr                        = default_sck_sr;
	       if ( enable && ~tx_fifo_empty && ~mstr_inhibit )
		 next_state                       = S_LOAD;
	    end
	  
	  // --------------------------------------------------------------------
	  // Load State
	  //   The Load State loads the MOSI and Clock shift registers and 
	  //   immediately transitions to Shift.
	  // --------------------------------------------------------------------
	  S_LOAD:
	    begin
	       case ({cpha,lsb_first})
		 2'b00:
		   begin
		      next_cycle_counter          = 5'd16;
		      next_mosi_sr                = mosi_sr_cpha0_msb;
		   end
		 2'b01:
		   begin
		      next_cycle_counter          = 5'd16;
		      next_mosi_sr                = mosi_sr_cpha0_lsb;
		   end
		 2'b10:
		   begin
		      next_cycle_counter          = 5'd17;
		      next_mosi_sr                = mosi_sr_cpha1_msb;
		   end
		 2'b11:
		   begin
		      next_cycle_counter          = 5'd17;
		      next_mosi_sr                = mosi_sr_cpha1_lsb;
		   end
	       endcase

	       next_sck_sr                        = default_sck_sr;
	       next_csb                           = 1'b0;
	       next_clock_divider                 = clock_div;
	       next_tx_fifo_rd                    = 1'b1;

	       next_state                         = S_SHIFT;
	    end
	  
	  // --------------------------------------------------------------------
	  // Shift State
	  //   Shifts the MOSI and Clock shift registers and and loads up the 
	  //   MISO shift register.  When the cycle counter indicates that the 
	  //   cycle is done, the MISO shift register is dumped to the RX FIFO.  
	  //   If TX FIFO is not empty, go to the Load State.  If the TX FIFO 
	  //   is empty, got to the IDLE state.
	  // --------------------------------------------------------------------
	  S_SHIFT:
	    begin
	       next_csb                           = 1'b0;
	       
	       if ( clock_divider == 16'b0 )
		 begin
		    next_clock_divider            = clock_div;
		    
		    if ( tri_counter != 9'b0 )
		      next_tri_counter            = tri_counter - 1'b1;
		    
		    if ( cycle_counter == 5'b0 )
		      begin
			 next_rx_fifo_wr          = 1'b1;
			 case ({miso_slip, lsb_first})
			   2'b00: next_rx_fifo_wr_data = miso_sr_straight_msb;
			   2'b01: next_rx_fifo_wr_data = miso_sr_straight_lsb;
			   2'b10: next_rx_fifo_wr_data = miso_sr_slipped_msb;
			   2'b11: next_rx_fifo_wr_data = miso_sr_slipped_lsb;
			 endcase
			     
			 if ( tx_fifo_empty )
			   next_state             = S_IDLE;
			 else
			   next_state             = S_LOAD;
		      end
		    else
		      begin
			 next_cycle_counter       = cycle_counter - 1'b1;
			 next_sck_sr              = {cpol,sck_sr[16:1]};
			 next_mosi_sr             = {1'b0,mosi_sr[16:1]};
			 next_miso_sr             = {miso_sr[14:0],miso};
		      end
		 end
	       else
		 begin
		    next_clock_divider            = clock_divider - 1'b1;
		 end
	    end
	  
	  default: next_state                     = S_IDLE;
	endcase
     end
   

   
   
endmodule

