// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty, Jim Marvill
// Description:       AXI_ADRV9001, transmit data generator
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx_dgen (

  // transmit interface

  input   wire            tx_clk,
  input   wire            tx_resetn,
  input   wire            tx_read,
  output  reg             tx_valid = 'd0,
  output  reg   [ 63:0]   tx_data = 'd0,
  input   wire            tx_mcs_latency_start,
  input   wire  [ 31:0]   tx_pattern,
  input   wire            tx_pattern_pulse,
  input   wire  [  7:0]   tx_pattern_delay,
  input   wire            tx_pattern_ins,
  output  reg             tx_pattern_done = 'd0,
  input   wire  [  2:0]   tx_num_of_bits,
  input   wire  [  3:0]   tx_data_sel);

  // internal registers
 
  reg           [  1:0]   tx_tone32_cnt = 'd0;
  reg           [ 63:0]   tx_patt32 = 'd0;
  reg           [ 63:0]   tx_nibb32 = 'd0;
  reg           [ 63:0]   tx_ramp32 = 'd0;
  reg           [ 63:0]   tx_pn1532 = 'd0;
  reg           [ 63:0]   tx_pn0732 = 'd0;
  reg           [ 63:0]   tx_patt24 = 'd0;
  reg           [ 63:0]   tx_ramp24 = 'd0;
  reg           [ 63:0]   tx_patt16 = 'd0;
  reg           [ 63:0]   tx_ramp16 = 'd0;
  reg           [ 63:0]   tx_ramp08 = 'd0;
  reg           [ 63:0]   tx_patt08 = 'd0;
  reg           [ 63:0]   tx_patt02 = 'd0;
  reg                     tx_pattern_ins_d = 'd0;
  reg                     tx_pattern_insert = 'd0;
  reg                     tx_pattern_state = 'd0;
  reg           [  7:0]   tx_pattern_count = 'd0;

  // internal signals

  wire          [ 63:0]   tx_tone32;
  wire          [ 63:0]   tx_pn1532_fn;
  wire          [ 63:0]   tx_pn0732_fn;
  wire          [ 63:0]   tx_ramp02;
  wire                    tx_pattern_ins_p;

  // pn7 function

  function [31:0] pn7;
    input [31:0] din;
    reg   [31:0] dout;
    begin
      dout[31] = din[ 6] ^ din[ 5];
      dout[30] = din[ 5] ^ din[ 4];
      dout[29] = din[ 4] ^ din[ 3];
      dout[28] = din[ 3] ^ din[ 2];
      dout[27] = din[ 2] ^ din[ 1];
      dout[26] = din[ 1] ^ din[ 0];
      dout[25] = din[ 0] ^ din[ 6] ^ din[ 5];
      dout[24] = din[ 6] ^ din[ 4];
      dout[23] = din[ 5] ^ din[ 3];
      dout[22] = din[ 4] ^ din[ 2];
      dout[21] = din[ 3] ^ din[ 1];
      dout[20] = din[ 2] ^ din[ 0];
      dout[19] = din[ 1] ^ din[ 6] ^ din[ 5];
      dout[18] = din[ 0] ^ din[ 5] ^ din[ 4];
      dout[17] = din[ 6] ^ din[ 4] ^ din[ 5] ^ din[ 3];
      dout[16] = din[ 5] ^ din[ 3] ^ din[ 4] ^ din[ 2];
      dout[15] = din[ 4] ^ din[ 2] ^ din[ 3] ^ din[ 1];
      dout[14] = din[ 3] ^ din[ 1] ^ din[ 2] ^ din[ 0];
      dout[13] = din[ 2] ^ din[ 0] ^ din[ 1] ^ din[ 6] ^ din[ 5];
      dout[12] = din[ 1] ^ din[ 6] ^ din[ 0] ^ din[ 4];
      dout[11] = din[ 0] ^ din[ 6] ^ din[ 3];
      dout[10] = din[ 6] ^ din[ 2];
      dout[ 9] = din[ 5] ^ din[ 1];
      dout[ 8] = din[ 4] ^ din[ 0];
      dout[ 7] = din[ 3] ^ din[ 6] ^ din[ 5];
      dout[ 6] = din[ 2] ^ din[ 5] ^ din[ 4];
      dout[ 5] = din[ 1] ^ din[ 4] ^ din[ 3];
      dout[ 4] = din[ 0] ^ din[ 3] ^ din[ 2];
      dout[ 3] = din[ 6] ^ din[ 2] ^ din[ 5] ^ din[ 1];
      dout[ 2] = din[ 5] ^ din[ 1] ^ din[ 4] ^ din[ 0];
      dout[ 1] = din[ 4] ^ din[ 0] ^ din[ 3] ^ din[ 6] ^ din[ 5];
      dout[ 0] = din[ 3] ^ din[ 6] ^ din[ 2] ^ din[ 4];
      pn7 = dout;
    end
  endfunction

  // pn15 function

  function [31:0] pn15;
    input [31:0] din;
    reg   [31:0] dout;
    begin
      dout[31] = din[14] ^ din[13];
      dout[30] = din[13] ^ din[12];
      dout[29] = din[12] ^ din[11];
      dout[28] = din[11] ^ din[10];
      dout[27] = din[10] ^ din[ 9];
      dout[26] = din[ 9] ^ din[ 8];
      dout[25] = din[ 8] ^ din[ 7];
      dout[24] = din[ 7] ^ din[ 6];
      dout[23] = din[ 6] ^ din[ 5];
      dout[22] = din[ 5] ^ din[ 4];
      dout[21] = din[ 4] ^ din[ 3];
      dout[20] = din[ 3] ^ din[ 2];
      dout[19] = din[ 2] ^ din[ 1];
      dout[18] = din[ 1] ^ din[ 0];
      dout[17] = din[ 0] ^ din[14] ^ din[13];
      dout[16] = din[14] ^ din[12];
      dout[15] = din[13] ^ din[11];
      dout[14] = din[12] ^ din[10];
      dout[13] = din[11] ^ din[ 9];
      dout[12] = din[10] ^ din[ 8];
      dout[11] = din[ 9] ^ din[ 7];
      dout[10] = din[ 8] ^ din[ 6];
      dout[ 9] = din[ 7] ^ din[ 5];
      dout[ 8] = din[ 6] ^ din[ 4];
      dout[ 7] = din[ 5] ^ din[ 3];
      dout[ 6] = din[ 4] ^ din[ 2];
      dout[ 5] = din[ 3] ^ din[ 1];
      dout[ 4] = din[ 2] ^ din[ 0];
      dout[ 3] = din[ 1] ^ din[14] ^ din[13];
      dout[ 2] = din[ 0] ^ din[13] ^ din[12];
      dout[ 1] = din[14] ^ din[12] ^ din[13] ^ din[11];
      dout[ 0] = din[13] ^ din[11] ^ din[12] ^ din[10];
      pn15 = dout;
    end
  endfunction

  // transmit data sources
 
  always @(posedge tx_clk) begin
    tx_valid <= tx_read;
    if (tx_read == 1'b1) begin
      case ({tx_num_of_bits, tx_data_sel})
        7'b1000011: tx_data <= tx_patt32;
        7'b1000100: tx_data <= tx_nibb32;
        7'b1000101: tx_data <= tx_ramp32;
        7'b1000110: tx_data <= tx_pn1532;
        7'b1000111: tx_data <= tx_pn0732;
        7'b1001000: tx_data <= tx_tone32;
        7'b0110011: tx_data <= tx_patt24;
        7'b0110101: tx_data <= tx_ramp24;
        7'b0100011: tx_data <= tx_patt16;
        7'b0100101: tx_data <= tx_ramp16;
        7'b0010011: tx_data <= tx_patt08;
        7'b0010101: tx_data <= tx_ramp08;
        7'b0000011: tx_data <= tx_patt02;
        7'b0000101: tx_data <= tx_ramp02;
        default: tx_data <= 64'd0;
      endcase
    end
  end

  // 32bit (16I, 16Q) mode
 
  assign tx_tone32[63:32] = (tx_tone32_cnt == 2'd0) ? 32'h16a116a1 :
    ((tx_tone32_cnt == 2'd1) ? 32'he9f516a1 : ((tx_tone32_cnt == 2'd2) ?
    32'he9f5e9f5 : 32'h16a1e9f5));

  assign tx_tone32[31: 0] = (tx_tone32_cnt == 2'd0) ? 32'h20000000 :
    ((tx_tone32_cnt == 2'd1) ? 32'h00002000 : ((tx_tone32_cnt == 2'd2) ?
    32'he0000000 : 32'h0000e000));

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_tone32_cnt <= 2'd0;
    end else if (tx_read == 1'b1) begin
      tx_tone32_cnt <= tx_tone32_cnt + 1'b1;
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_patt32 <= 64'd0;
    end else if (tx_read == 1'b1) begin
      if (tx_pattern_pulse == 1'd0) begin
        tx_patt32[63:32] <= tx_pattern[31:0];
        tx_patt32[31: 0] <= tx_pattern[31:0];
      end else if (tx_pattern_insert == 1'd1) begin
        tx_patt32[63:32] <= 32'd0;
        tx_patt32[31: 0] <= tx_pattern[31:0];
      end else begin
        tx_patt32[63:32] <= 32'd0;
        tx_patt32[31: 0] <= 32'd0;
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_nibb32[63:60] <= 4'h1;
      tx_nibb32[59:56] <= 4'h1;
      tx_nibb32[55:52] <= 4'h1;
      tx_nibb32[51:48] <= 4'h1;
      tx_nibb32[47:44] <= 4'h1;
      tx_nibb32[43:40] <= 4'h1;
      tx_nibb32[39:36] <= 4'h1;
      tx_nibb32[35:32] <= 4'h1;
      tx_nibb32[31:28] <= 4'h0;
      tx_nibb32[27:24] <= 4'h0;
      tx_nibb32[23:20] <= 4'h0;
      tx_nibb32[19:16] <= 4'h0;
      tx_nibb32[15:12] <= 4'h0;
      tx_nibb32[11: 8] <= 4'h0;
      tx_nibb32[ 7: 4] <= 4'h0;
      tx_nibb32[ 3: 0] <= 4'h0;
    end else if (tx_read == 1'b1) begin
      tx_nibb32[63:60] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[59:56] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[55:52] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[51:48] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[47:44] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[43:40] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[39:36] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[35:32] <= tx_nibb32[3:0] + 2'h3;
      tx_nibb32[31:28] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[27:24] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[23:20] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[19:16] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[15:12] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[11: 8] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[ 7: 4] <= tx_nibb32[3:0] + 2'h2;
      tx_nibb32[ 3: 0] <= tx_nibb32[3:0] + 2'h2;
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_ramp32[63:48] <= 16'h1;
      tx_ramp32[47:32] <= 16'h1;
      tx_ramp32[31:16] <= 16'h0;
      tx_ramp32[15: 0] <= 16'h0;
    end else if (tx_read == 1'b1) begin
      tx_ramp32[63:48] <= tx_ramp32[15:0] + 2'h3;
      tx_ramp32[47:32] <= tx_ramp32[15:0] + 2'h3;
      tx_ramp32[31:16] <= tx_ramp32[15:0] + 2'h2;
      tx_ramp32[15: 0] <= tx_ramp32[15:0] + 2'h2;
    end
  end

  assign tx_pn1532_fn = pn15({tx_pn1532[47:32], tx_pn1532[15:0]});

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_pn1532[63:48] <= 16'hffff;
      tx_pn1532[47:32] <= 16'hffff;
      tx_pn1532[31:16] <= 16'hffff;
      tx_pn1532[15: 0] <= 16'hffff;
    end else if (tx_read == 1'b1) begin
      tx_pn1532[63:48] <= tx_pn1532_fn[31:16];
      tx_pn1532[47:32] <= tx_pn1532_fn[31:16];
      tx_pn1532[31:16] <= tx_pn1532_fn[15: 0];
      tx_pn1532[15: 0] <= tx_pn1532_fn[15: 0];
    end
  end

  assign tx_pn0732_fn = pn7({tx_pn0732[47:32], tx_pn0732[15:0]});

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_pn0732[63:48] <= 16'hffff;
      tx_pn0732[47:32] <= 16'hffff;
      tx_pn0732[31:16] <= 16'hffff;
      tx_pn0732[15: 0] <= 16'hffff;
    end else if (tx_read == 1'b1) begin
      tx_pn0732[63:48] <= tx_pn0732_fn[31:16];
      tx_pn0732[47:32] <= tx_pn0732_fn[31:16];
      tx_pn0732[31:16] <= tx_pn0732_fn[15: 0];
      tx_pn0732[15: 0] <= tx_pn0732_fn[15: 0];
    end
  end

  // 24bit (12I, 12Q) mode
 
  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_patt24 <= 64'd0;
    end else if (tx_read == 1'b1) begin
      if (tx_pattern_pulse == 1'd0) begin
        tx_patt24[63:48] <= {4'd0, tx_pattern[27:16]};
        tx_patt24[47:32] <= {4'd0, tx_pattern[11: 0]};
        tx_patt24[31:16] <= {4'd0, tx_pattern[27:16]};
        tx_patt24[15: 0] <= {4'd0, tx_pattern[11: 0]};
      end else if (tx_pattern_insert == 1'd1) begin
        tx_patt24[63:48] <= 16'd0;
        tx_patt24[47:32] <= 16'd0;
        tx_patt24[31:16] <= {4'd0, tx_pattern[27:16]};
        tx_patt24[15: 0] <= {4'd0, tx_pattern[11: 0]};
      end else begin
        tx_patt24[63:48] <= 16'd0;
        tx_patt24[47:32] <= 16'd0;
        tx_patt24[31:16] <= 16'd0;
        tx_patt24[15: 0] <= 16'd0;
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_ramp24[63:48] <= 16'h1;
      tx_ramp24[47:32] <= 16'h1;
      tx_ramp24[31:16] <= 16'h0;
      tx_ramp24[15: 0] <= 16'h0;
    end else if (tx_read == 1'b1) begin
      tx_ramp24[63:60] <= 4'd0;
      tx_ramp24[59:48] <= tx_ramp24[11:0] + 2'h3;
      tx_ramp24[47:44] <= 4'd0;
      tx_ramp24[43:32] <= tx_ramp24[11:0] + 2'h3;
      tx_ramp24[31:28] <= 4'd0;
      tx_ramp24[27:16] <= tx_ramp24[11:0] + 2'h2;
      tx_ramp24[15:12] <= 4'd0;
      tx_ramp24[11: 0] <= tx_ramp24[11:0] + 2'h2;
    end
  end

  // 16bit (16I) mode
 
  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_patt16 <= 64'd0;
    end else if (tx_read == 1'b1) begin
      if (tx_pattern_pulse == 1'd0) begin
        tx_patt16[63:48] <= tx_pattern[15:0];
        tx_patt16[47:32] <= tx_pattern[15:0];
        tx_patt16[31:16] <= tx_pattern[15:0];
        tx_patt16[15: 0] <= tx_pattern[15:0];
      end else if (tx_pattern_insert == 1'd1) begin
        tx_patt16[63:48] <= 16'd0;
        tx_patt16[47:32] <= 16'd0;
        tx_patt16[31:16] <= 16'd0;
        tx_patt16[15: 0] <= tx_pattern[15:0];
      end else begin
        tx_patt16[63:48] <= 16'd0;
        tx_patt16[47:32] <= 16'd0;
        tx_patt16[31:16] <= 16'd0;
        tx_patt16[15: 0] <= 16'd0;
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_ramp16[63:48] <= 16'h3;
      tx_ramp16[47:32] <= 16'h2;
      tx_ramp16[31:16] <= 16'h1;
      tx_ramp16[15: 0] <= 16'h0;
    end else if (tx_read == 1'b1) begin
      tx_ramp16[63:48] <= tx_ramp16[15:0] + 3'h7;
      tx_ramp16[47:32] <= tx_ramp16[15:0] + 3'h6;
      tx_ramp16[31:16] <= tx_ramp16[15:0] + 3'h5;
      tx_ramp16[15: 0] <= tx_ramp16[15:0] + 3'h4;
    end
  end

  // 8bit (8I) mode
 
  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_patt08 <= 64'd0;
    end else if (tx_read == 1'b1) begin
      if (tx_pattern_pulse == 1'd0) begin
        tx_patt08[63:56] <= tx_pattern[7:0];
        tx_patt08[55:48] <= tx_pattern[7:0];
        tx_patt08[47:40] <= tx_pattern[7:0];
        tx_patt08[39:32] <= tx_pattern[7:0];
        tx_patt08[31:24] <= tx_pattern[7:0];
        tx_patt08[23:16] <= tx_pattern[7:0];
        tx_patt08[15: 8] <= tx_pattern[7:0];
        tx_patt08[ 7: 0] <= tx_pattern[7:0];
      end else if (tx_pattern_insert == 1'd1) begin
        tx_patt08[63:56] <= 8'd0;
        tx_patt08[55:48] <= 8'd0;
        tx_patt08[47:40] <= 8'd0;
        tx_patt08[39:32] <= 8'd0;
        tx_patt08[31:24] <= 8'd0;
        tx_patt08[23:16] <= 8'd0;
        tx_patt08[15: 8] <= 8'd0;
        tx_patt08[ 7: 0] <= tx_pattern[7:0];
      end else begin
        tx_patt08[63:56] <= 8'd0;
        tx_patt08[55:48] <= 8'd0;
        tx_patt08[47:40] <= 8'd0;
        tx_patt08[39:32] <= 8'd0;
        tx_patt08[31:24] <= 8'd0;
        tx_patt08[23:16] <= 8'd0;
        tx_patt08[15: 8] <= 8'd0;
        tx_patt08[ 7: 0] <= 8'd0;
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_ramp08[63:56] <= 8'h7;
      tx_ramp08[55:48] <= 8'h6;
      tx_ramp08[47:40] <= 8'h5;
      tx_ramp08[39:32] <= 8'h4;
      tx_ramp08[31:24] <= 8'h3;
      tx_ramp08[23:16] <= 8'h2;
      tx_ramp08[15: 8] <= 8'h1;
      tx_ramp08[ 7: 0] <= 8'h0;
    end else if (tx_read == 1'b1) begin
      tx_ramp08[63:56] <= tx_ramp08[7:0] + 4'hf;
      tx_ramp08[55:48] <= tx_ramp08[7:0] + 4'he;
      tx_ramp08[47:40] <= tx_ramp08[7:0] + 4'hd;
      tx_ramp08[39:32] <= tx_ramp08[7:0] + 4'hc;
      tx_ramp08[31:24] <= tx_ramp08[7:0] + 4'hb;
      tx_ramp08[23:16] <= tx_ramp08[7:0] + 4'ha;
      tx_ramp08[15: 8] <= tx_ramp08[7:0] + 4'h9;
      tx_ramp08[ 7: 0] <= tx_ramp08[7:0] + 4'h8;
    end
  end

  // 2bit (2I) mode
 
  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_patt02 <= 64'd0;
    end else if (tx_read == 1'b1) begin
      if (tx_pattern_pulse == 1'd0) begin
        tx_patt02[63:56] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[55:48] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[47:40] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[39:32] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[31:24] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[23:16] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[15: 8] <= {6'd0, tx_pattern[1:0]};
        tx_patt02[ 7: 0] <= {6'd0, tx_pattern[1:0]};
      end else if (tx_pattern_insert == 1'd1) begin
        tx_patt02[63:56] <= 8'd0;
        tx_patt02[55:48] <= 8'd0;
        tx_patt02[47:40] <= 8'd0;
        tx_patt02[39:32] <= 8'd0;
        tx_patt02[31:24] <= 8'd0;
        tx_patt02[23:16] <= 8'd0;
        tx_patt02[15: 8] <= 8'd0;
        tx_patt02[ 7: 0] <= {6'd0, tx_pattern[1:0]};
      end else begin
        tx_patt02[63:56] <= 8'd0;
        tx_patt02[55:48] <= 8'd0;
        tx_patt02[47:40] <= 8'd0;
        tx_patt02[39:32] <= 8'd0;
        tx_patt02[31:24] <= 8'd0;
        tx_patt02[23:16] <= 8'd0;
        tx_patt02[15: 8] <= 8'd0;
        tx_patt02[ 7: 0] <= 8'd0;
      end
    end
  end

  assign tx_ramp02[63:56] = 8'h3;
  assign tx_ramp02[55:48] = 8'h2;
  assign tx_ramp02[47:40] = 8'h1;
  assign tx_ramp02[39:32] = 8'h0;
  assign tx_ramp02[31:24] = 8'h3;
  assign tx_ramp02[23:16] = 8'h2;
  assign tx_ramp02[15: 8] = 8'h1;
  assign tx_ramp02[ 7: 0] = 8'h0;

  // pattern insert control
 
  assign tx_pattern_ins_p = (~tx_pattern_ins_d & tx_pattern_ins) | tx_mcs_latency_start;

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_pattern_ins_d <= 1'd0;
      tx_pattern_done <= 1'd0;
      tx_pattern_insert <= 1'd0;
      tx_pattern_state <= 1'd0;
      tx_pattern_count <= 8'd0;
    end else begin
      tx_pattern_ins_d <= tx_pattern_ins;
      tx_pattern_done <= ~tx_pattern_state;
      if (tx_read == 1'd1) begin
        if (tx_pattern_count == 8'd0) begin
          tx_pattern_insert <= tx_pattern_state;
        end else begin
          tx_pattern_insert <= 1'd0;
        end
      end
      if ((tx_pattern_state == 1'd1) && (tx_read == 1'd1)) begin
        if (tx_pattern_count == 8'd0) begin
          tx_pattern_state <= 1'd0;
        end
        if (tx_pattern_count > 8'd0) begin
          tx_pattern_count <= tx_pattern_count - 1'd1;
        end
      end else if (tx_pattern_ins_p == 1'd1) begin
        tx_pattern_state <= 1'd1;
        tx_pattern_count <= tx_pattern_delay;
      end
    end
  end

endmodule

// ***************************************************************************
// ***************************************************************************
