// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       AXI_ADRV9001, transmit interface
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx #(

  parameter ID = 0,
  parameter GPIO2_CMOS1_LVDS0 = 0,
  parameter IODELAY_GROUP = "IODELAY_GROUP",
  parameter IODELAY_CTRL_ENABLE = 1) (

  // transmit interface

  input   wire            tx_cssi_refclk_lssi_refclk_p,
  input   wire            tx_cssi_nc_lssi_refclk_n,
  output  wire            tx_cssi_clk_lssi_clk_p,
  output  wire            tx_cssi_nc_lssi_clk_n,
  output  wire            tx_cssi_strobe_lssi_strobe_p,
  output  wire            tx_cssi_nc_lssi_strobe_n,
  output  wire            tx_cssi_data1_lssi_data0_p,
  output  wire            tx_cssi_data0_lssi_data0_n,
  output  wire            tx_cssi_data3_lssi_data1_p,
  output  wire            tx_cssi_data2_lssi_data1_n,
  input   wire  [  9:0]   tx_cssi_gpio_in,
  output  wire  [  9:0]   tx_cssi_gpio_out,
  output  wire  [  9:0]   tx_cssi_gpio_enb,

  // internal interface

  input   wire            rx_buf_clk,
  input   wire            rx_clk,
  input   wire            rx_resetn,
  input   wire            rx_valid,
  input   wire  [ 63:0]   rx_data,
  output  wire            tx_clk,
  output  reg             tx_resetn = 'd0,
  output  reg             tx_read = 'd0,
  input   wire            tx_valid,
  input   wire  [ 63:0]   tx_data,
  input   wire            tx_enable,
  output  wire  [  9:0]   tx_gpio_in,
  input   wire  [  9:0]   tx_gpio_out,
  input   wire  [  9:0]   tx_gpio_enb,

  // mcs & latency interface

  input   wire            mcs_toggle,
  output  wire            tx_pattern_done,

  // delay interface

  input   wire            sys_200m_clk,
  input   wire            sys_200m_resetn,

  // axi interface

  input   wire            axilite_clk,
  input   wire            axilite_resetn,
  input   wire            axilite_wrreq,
  input   wire  [ 15:0]   axilite_wraddr,
  input   wire  [ 31:0]   axilite_wrdata,
  output  wire            axilite_wrack,
  input   wire            axilite_rdreq,
  input   wire  [ 15:0]   axilite_rdaddr,
  output  wire  [ 31:0]   axilite_rddata,
  output  wire            axilite_rdack);

  // cmos & lvds data lines

  localparam              GPIO2_CMOS1 = (GPIO2_CMOS1_LVDS0 == 1) ? 1 : 2;

  // supported interface modes
 
  localparam    [  7:0]   CMOS_1L_LS_32X1 = {1'b1, 1'b0, 1'b1, 3'b100, 2'b00};
  localparam    [  7:0]   CMOS_1L_PS_16X1 = {1'b1, 1'b0, 1'b0, 3'b010, 2'b00};
  localparam    [  7:0]   CMOS_1L_PS_08X1 = {1'b1, 1'b0, 1'b0, 3'b001, 2'b00};
  localparam    [  7:0]   CMOS_1L_PS_02X1 = {1'b1, 1'b0, 1'b0, 3'b000, 2'b00};
  localparam    [  7:0]   CMOS_1L_PS_32X1 = {1'b1, 1'b0, 1'b0, 3'b100, 2'b00};
  localparam    [  7:0]   CMOS_4L_PS_32X1 = {1'b1, 1'b1, 1'b0, 3'b100, 2'b00};
  localparam    [  7:0]   LVDS_1L_LS_32X1 = {1'b0, 1'b0, 1'b1, 3'b100, 2'b00};
  localparam    [  7:0]   LVDS_1L_LS_24X1 = {1'b0, 1'b0, 1'b1, 3'b011, 2'b00};
  localparam    [  7:0]   LVDS_1L_PS_32X1 = {1'b0, 1'b0, 1'b0, 3'b100, 2'b00};
  localparam    [  7:0]   LVDS_1L_PS_24X1 = {1'b0, 1'b0, 1'b0, 3'b011, 2'b00};
  localparam    [  7:0]   LVDS_2L_PS_32X1 = {1'b0, 1'b1, 1'b0, 3'b100, 2'b00};
  localparam    [  7:0]   LVDS_2L_PS_24X1 = {1'b0, 1'b1, 1'b0, 3'b011, 2'b00};

  // internal registers
 
  reg                     tx_valid_iq = 'd0;
  reg           [ 63:0]   tx_data_iq = 'd0;
  reg                     tx_valid_64 = 'd0;
  reg           [ 63:0]   tx_data_64 = 'd0;
  reg           [  1:0]   tx_bcnt_d = 'd0;
  reg           [  1:0]   tx_bcnt_2d = 'd0;
  reg                     tx_wrcnt = 'd0;
  reg           [ 31:0]   tx_wrdata_0 = 'd0;
  reg           [ 31:0]   tx_wrdata_1 = 'd0;
  reg           [ 31:0]   tx_wrdata_2 = 'd0;
  reg           [ 31:0]   tx_wrdata_3 = 'd0;
  reg           [  1:0]   tx_rdcnt = 'd0;
  reg           [ 31:0]   tx_rddata = 'd0;
  reg           [  7:0]   tx_rddata_h = 'd0;
  reg                     tx_read_toggle = 'd0;
  reg                     tx_read_32 = 'd0;
  reg                     tx_bvalid = 'd0;
  reg                     tx_rdvalid = 'd0;
  reg           [  1:0]   tx_bcnt = 'd0;
  reg           [  7:0]   tx_bstrobe = 'd0;
  reg           [  7:0]   tx_bdata_0 = 'd0;
  reg           [  7:0]   tx_bdata_1 = 'd0;
  reg           [  7:0]   tx_bdata_2 = 'd0;
  reg           [  7:0]   tx_bdata_3 = 'd0;
  reg           [  7:0]   tx_pstrobe = 'd0;
  reg           [  7:0]   tx_pdata_0 = 'd0;
  reg           [  7:0]   tx_pdata_1 = 'd0;
  reg           [  7:0]   tx_pdata_2 = 'd0;
  reg           [  7:0]   tx_pdata_3 = 'd0;
  reg                     tx_mcs_toggle_d = 'd0;
  reg                     tx_mcs_latency_start = 'd0;
  reg                     tx_mcs_latency_stop = 'd0;
  reg           [  3:0]   tx_mcs_latency_incr = 'd0;
  reg                     tx_mcs_latency_enable = 'd0;
  reg           [ 15:0]   tx_mcs_latency_count = 'd0;
  reg                     tx_init_req_d = 'd0;
  reg           [  3:0]   tx_init_cnt = 'd0;
  reg                     axilite_delay_unlocked = 'd1;
  reg                     axilite_clk_enable = 'd0;
  reg                     axilite_cpu_resetn = 'd0;
  reg                     axilite_cpu_reset = 'd0;
  reg                     axilite_mcs_sync_d = 1'd0;
  reg                     axilite_mcs_toggle_d = 1'd0;
  reg                     axilite_mcs_resetn = 1'd0;
  reg                     axilite_mcs_clk_enable = 'd0;
  reg           [  3:0]   axilite_mcs_count = 4'd0;
  reg                     axilite_sw_resetn = 'd0;
  reg                     axilite_sw_clk_enable = 'd0;
  reg                     axilite_sw_reset_in_d = 'd0;
  reg           [ 11:0]   axilite_sw_reset_count = 'd0;

  // internal signals
 
  wire                    tx_cmos1_lvds0;
  wire          [  7:0]   tx_mode;
  wire                    tx_init_ack;
  wire                    tx_mcs_toggle;
  wire                    tx_valid_mux;
  wire          [ 63:0]   tx_data_mux;
  wire                    tx_swap_iq;
  wire          [  2:0]   tx_num_of_bits;
  wire                    tx_lstrb1_pstrb0;
  wire                    tx_mlane1_slane0;
  wire                    tx_sdr1_ddr0;
  wire                    tx_lsb1_msb0;
  wire                    tx_q1_i0;
  wire                    tx_buf_enable;
  wire                    tx_init_req;
  wire                    tx_mcs_sync;
  wire          [  3:0]   tx_data_sel;
  wire          [ 31:0]   tx_pattern;
  wire                    tx_pattern_pulse;
  wire          [  7:0]   tx_pattern_delay;
  wire                    tx_pattern_ins;
  wire                    tx_lb_valid;
  wire          [ 63:0]   tx_lb_data;
  wire                    tx_dgen_valid;
  wire          [ 63:0]   tx_dgen_data;
  wire          [  3:0]   tx_data_p;
  wire          [  3:0]   tx_data_n;
  wire                    tx_hs_clk;
  wire                    tx_cpu_resetn;
  wire                    delay_locked;
  wire          [ 31:0]   axilite_cmos_lvds_id;
  wire          [  2:0]   axilite_num_of_bits;
  wire                    axilite_lstrb1_pstrb0;
  wire                    axilite_mlane1_slane0;
  wire                    axilite_sdr1_ddr0;
  wire                    axilite_lsb1_msb0;
  wire                    axilite_q1_i0;
  wire                    axilite_cmos1_lvds0;
  wire                    axilite_buf_enable;
  wire                    axilite_clk_sel;
  wire                    axilite_clk_msb;
  wire                    axilite_init_req;
  wire                    axilite_init_ack;
  wire                    axilite_lb_ovf;
  wire                    axilite_lb_unf;
  wire          [  3:0]   axilite_data_sel;
  wire          [ 31:0]   axilite_pattern;
  wire                    axilite_pattern_pulse;
  wire          [  7:0]   axilite_pattern_delay;
  wire                    axilite_pattern_ins;
  wire                    axilite_pattern_done;
  wire                    axilite_delay_locked;
  wire          [  4:0]   axilite_wrdelay_0;
  wire          [  4:0]   axilite_rddelay_0;
  wire          [  4:0]   axilite_wrdelay_1;
  wire          [  4:0]   axilite_rddelay_1;
  wire          [  4:0]   axilite_wrdelay_2;
  wire          [  4:0]   axilite_rddelay_2;
  wire          [  4:0]   axilite_wrdelay_3;
  wire          [  4:0]   axilite_rddelay_3;
  wire          [  4:0]   axilite_wrdelay_s;
  wire          [  4:0]   axilite_rddelay_s;
  wire          [  4:0]   axilite_wrdelay_c;
  wire          [  4:0]   axilite_rddelay_c;
  wire          [  4:0]   axilite_wrdelay_int_c;
  wire          [  4:0]   axilite_rddelay_int_c;
  wire          [  4:0]   axilite_wrdelay_ext_c;
  wire          [  4:0]   axilite_rddelay_ext_c;
  wire          [ 31:0]   axilite_clk_mon_count;
  wire          [ 15:0]   axilite_mcs_latency_count;
  wire                    axilite_mcs_sync;
  wire          [  3:0]   axilite_mcs_active;
  wire                    axilite_mcs_toggle;
  wire                    axilite_mcs_toggle_p;
  wire                    axilite_sw_reset_in;

  // configuration options

  assign axilite_cmos_lvds_id = (GPIO2_CMOS1_LVDS0 == 0) ? 32'h4c564453 :
    ((GPIO2_CMOS1_LVDS0 == 1) ? 32'h434d4f53 : 32'h4750494f);
  assign axilite_cmos1_lvds0 = (GPIO2_CMOS1_LVDS0 == 0) ? 1'b0 : 1'b1;
  assign tx_cmos1_lvds0 = (GPIO2_CMOS1_LVDS0 == 0) ? 1'b0 : 1'b1;

  assign tx_mode[7:7] = tx_cmos1_lvds0;
  assign tx_mode[6:6] = tx_mlane1_slane0;
  assign tx_mode[5:5] = tx_lstrb1_pstrb0;
  assign tx_mode[4:2] = tx_num_of_bits;
  assign tx_mode[1:0] = 2'd0;

  // transmit data sources

  assign tx_valid_mux = (tx_data_sel == 4'd0) ? tx_valid :
    ((tx_data_sel == 4'd1) ? tx_lb_valid : tx_dgen_valid);
  assign tx_data_mux = (tx_data_sel == 4'd0) ? tx_data :
    ((tx_data_sel == 4'd1) ? tx_lb_data : tx_dgen_data);

  // iq swap

  assign tx_swap_iq = ((tx_cmos1_lvds0 == 1'b0) || ((tx_cmos1_lvds0 == 1'b1) &&
    (tx_num_of_bits == 3'd4))) ? tx_q1_i0 : 1'b0;

  always @(posedge tx_clk) begin
    tx_valid_iq <= tx_valid_mux;
    if (tx_swap_iq == 1'b1) begin
      tx_data_iq <= {tx_data_mux[47:32], tx_data_mux[63:48],
        tx_data_mux[15:0], tx_data_mux[31:16]};
    end else begin
      tx_data_iq <= tx_data_mux;
    end 
  end

  // lsb-msb swap

  always @(posedge tx_clk) begin
    tx_valid_64 <= tx_valid_iq;
    case ({tx_lsb1_msb0, tx_num_of_bits})
      4'b1000: begin
        tx_data_64[63] <= tx_data_iq[63];
        tx_data_64[62] <= tx_data_iq[62];
        tx_data_64[61] <= tx_data_iq[61];
        tx_data_64[60] <= tx_data_iq[60];
        tx_data_64[59] <= tx_data_iq[59];
        tx_data_64[58] <= tx_data_iq[58];
        tx_data_64[57] <= tx_data_iq[56];
        tx_data_64[56] <= tx_data_iq[57];
        tx_data_64[55] <= tx_data_iq[55];
        tx_data_64[54] <= tx_data_iq[54];
        tx_data_64[53] <= tx_data_iq[53];
        tx_data_64[52] <= tx_data_iq[52];
        tx_data_64[51] <= tx_data_iq[51];
        tx_data_64[50] <= tx_data_iq[50];
        tx_data_64[49] <= tx_data_iq[48];
        tx_data_64[48] <= tx_data_iq[49];
        tx_data_64[47] <= tx_data_iq[47];
        tx_data_64[46] <= tx_data_iq[46];
        tx_data_64[45] <= tx_data_iq[45];
        tx_data_64[44] <= tx_data_iq[44];
        tx_data_64[43] <= tx_data_iq[43];
        tx_data_64[42] <= tx_data_iq[42];
        tx_data_64[41] <= tx_data_iq[40];
        tx_data_64[40] <= tx_data_iq[41];
        tx_data_64[39] <= tx_data_iq[39];
        tx_data_64[38] <= tx_data_iq[38];
        tx_data_64[37] <= tx_data_iq[37];
        tx_data_64[36] <= tx_data_iq[36];
        tx_data_64[35] <= tx_data_iq[35];
        tx_data_64[34] <= tx_data_iq[34];
        tx_data_64[33] <= tx_data_iq[32];
        tx_data_64[32] <= tx_data_iq[33];
        tx_data_64[31] <= tx_data_iq[31];
        tx_data_64[30] <= tx_data_iq[30];
        tx_data_64[29] <= tx_data_iq[29];
        tx_data_64[28] <= tx_data_iq[28];
        tx_data_64[27] <= tx_data_iq[27];
        tx_data_64[26] <= tx_data_iq[26];
        tx_data_64[25] <= tx_data_iq[24];
        tx_data_64[24] <= tx_data_iq[25];
        tx_data_64[23] <= tx_data_iq[23];
        tx_data_64[22] <= tx_data_iq[22];
        tx_data_64[21] <= tx_data_iq[21];
        tx_data_64[20] <= tx_data_iq[20];
        tx_data_64[19] <= tx_data_iq[19];
        tx_data_64[18] <= tx_data_iq[18];
        tx_data_64[17] <= tx_data_iq[16];
        tx_data_64[16] <= tx_data_iq[17];
        tx_data_64[15] <= tx_data_iq[15];
        tx_data_64[14] <= tx_data_iq[14];
        tx_data_64[13] <= tx_data_iq[13];
        tx_data_64[12] <= tx_data_iq[12];
        tx_data_64[11] <= tx_data_iq[11];
        tx_data_64[10] <= tx_data_iq[10];
        tx_data_64[ 9] <= tx_data_iq[ 8];
        tx_data_64[ 8] <= tx_data_iq[ 9];
        tx_data_64[ 7] <= tx_data_iq[ 7];
        tx_data_64[ 6] <= tx_data_iq[ 6];
        tx_data_64[ 5] <= tx_data_iq[ 5];
        tx_data_64[ 4] <= tx_data_iq[ 4];
        tx_data_64[ 3] <= tx_data_iq[ 3];
        tx_data_64[ 2] <= tx_data_iq[ 2];
        tx_data_64[ 1] <= tx_data_iq[ 0];
        tx_data_64[ 0] <= tx_data_iq[ 1];
      end
      4'b1001: begin
        tx_data_64[63] <= tx_data_iq[56];
        tx_data_64[62] <= tx_data_iq[57];
        tx_data_64[61] <= tx_data_iq[58];
        tx_data_64[60] <= tx_data_iq[59];
        tx_data_64[59] <= tx_data_iq[60];
        tx_data_64[58] <= tx_data_iq[61];
        tx_data_64[57] <= tx_data_iq[62];
        tx_data_64[56] <= tx_data_iq[63];
        tx_data_64[55] <= tx_data_iq[48];
        tx_data_64[54] <= tx_data_iq[49];
        tx_data_64[53] <= tx_data_iq[50];
        tx_data_64[52] <= tx_data_iq[51];
        tx_data_64[51] <= tx_data_iq[52];
        tx_data_64[50] <= tx_data_iq[53];
        tx_data_64[49] <= tx_data_iq[54];
        tx_data_64[48] <= tx_data_iq[55];
        tx_data_64[47] <= tx_data_iq[40];
        tx_data_64[46] <= tx_data_iq[41];
        tx_data_64[45] <= tx_data_iq[42];
        tx_data_64[44] <= tx_data_iq[43];
        tx_data_64[43] <= tx_data_iq[44];
        tx_data_64[42] <= tx_data_iq[45];
        tx_data_64[41] <= tx_data_iq[46];
        tx_data_64[40] <= tx_data_iq[47];
        tx_data_64[39] <= tx_data_iq[32];
        tx_data_64[38] <= tx_data_iq[33];
        tx_data_64[37] <= tx_data_iq[34];
        tx_data_64[36] <= tx_data_iq[35];
        tx_data_64[35] <= tx_data_iq[36];
        tx_data_64[34] <= tx_data_iq[37];
        tx_data_64[33] <= tx_data_iq[38];
        tx_data_64[32] <= tx_data_iq[39];
        tx_data_64[31] <= tx_data_iq[24];
        tx_data_64[30] <= tx_data_iq[25];
        tx_data_64[29] <= tx_data_iq[26];
        tx_data_64[28] <= tx_data_iq[27];
        tx_data_64[27] <= tx_data_iq[28];
        tx_data_64[26] <= tx_data_iq[29];
        tx_data_64[25] <= tx_data_iq[30];
        tx_data_64[24] <= tx_data_iq[31];
        tx_data_64[23] <= tx_data_iq[16];
        tx_data_64[22] <= tx_data_iq[17];
        tx_data_64[21] <= tx_data_iq[18];
        tx_data_64[20] <= tx_data_iq[19];
        tx_data_64[19] <= tx_data_iq[20];
        tx_data_64[18] <= tx_data_iq[21];
        tx_data_64[17] <= tx_data_iq[22];
        tx_data_64[16] <= tx_data_iq[23];
        tx_data_64[15] <= tx_data_iq[ 8];
        tx_data_64[14] <= tx_data_iq[ 9];
        tx_data_64[13] <= tx_data_iq[10];
        tx_data_64[12] <= tx_data_iq[11];
        tx_data_64[11] <= tx_data_iq[12];
        tx_data_64[10] <= tx_data_iq[13];
        tx_data_64[ 9] <= tx_data_iq[14];
        tx_data_64[ 8] <= tx_data_iq[15];
        tx_data_64[ 7] <= tx_data_iq[ 0];
        tx_data_64[ 6] <= tx_data_iq[ 1];
        tx_data_64[ 5] <= tx_data_iq[ 2];
        tx_data_64[ 4] <= tx_data_iq[ 3];
        tx_data_64[ 3] <= tx_data_iq[ 4];
        tx_data_64[ 2] <= tx_data_iq[ 5];
        tx_data_64[ 1] <= tx_data_iq[ 6];
        tx_data_64[ 0] <= tx_data_iq[ 7];
      end
      4'b1010: begin
        tx_data_64[63] <= tx_data_iq[48];
        tx_data_64[62] <= tx_data_iq[49];
        tx_data_64[61] <= tx_data_iq[50];
        tx_data_64[60] <= tx_data_iq[51];
        tx_data_64[59] <= tx_data_iq[52];
        tx_data_64[58] <= tx_data_iq[53];
        tx_data_64[57] <= tx_data_iq[54];
        tx_data_64[56] <= tx_data_iq[55];
        tx_data_64[55] <= tx_data_iq[56];
        tx_data_64[54] <= tx_data_iq[57];
        tx_data_64[53] <= tx_data_iq[58];
        tx_data_64[52] <= tx_data_iq[59];
        tx_data_64[51] <= tx_data_iq[60];
        tx_data_64[50] <= tx_data_iq[61];
        tx_data_64[49] <= tx_data_iq[62];
        tx_data_64[48] <= tx_data_iq[63];
        tx_data_64[47] <= tx_data_iq[32];
        tx_data_64[46] <= tx_data_iq[33];
        tx_data_64[45] <= tx_data_iq[34];
        tx_data_64[44] <= tx_data_iq[35];
        tx_data_64[43] <= tx_data_iq[36];
        tx_data_64[42] <= tx_data_iq[37];
        tx_data_64[41] <= tx_data_iq[38];
        tx_data_64[40] <= tx_data_iq[39];
        tx_data_64[39] <= tx_data_iq[40];
        tx_data_64[38] <= tx_data_iq[41];
        tx_data_64[37] <= tx_data_iq[42];
        tx_data_64[36] <= tx_data_iq[43];
        tx_data_64[35] <= tx_data_iq[44];
        tx_data_64[34] <= tx_data_iq[45];
        tx_data_64[33] <= tx_data_iq[46];
        tx_data_64[32] <= tx_data_iq[47];
        tx_data_64[31] <= tx_data_iq[16];
        tx_data_64[30] <= tx_data_iq[17];
        tx_data_64[29] <= tx_data_iq[18];
        tx_data_64[28] <= tx_data_iq[19];
        tx_data_64[27] <= tx_data_iq[20];
        tx_data_64[26] <= tx_data_iq[21];
        tx_data_64[25] <= tx_data_iq[22];
        tx_data_64[24] <= tx_data_iq[23];
        tx_data_64[23] <= tx_data_iq[24];
        tx_data_64[22] <= tx_data_iq[25];
        tx_data_64[21] <= tx_data_iq[26];
        tx_data_64[20] <= tx_data_iq[27];
        tx_data_64[19] <= tx_data_iq[28];
        tx_data_64[18] <= tx_data_iq[29];
        tx_data_64[17] <= tx_data_iq[30];
        tx_data_64[16] <= tx_data_iq[31];
        tx_data_64[15] <= tx_data_iq[ 0];
        tx_data_64[14] <= tx_data_iq[ 1];
        tx_data_64[13] <= tx_data_iq[ 2];
        tx_data_64[12] <= tx_data_iq[ 3];
        tx_data_64[11] <= tx_data_iq[ 4];
        tx_data_64[10] <= tx_data_iq[ 5];
        tx_data_64[ 9] <= tx_data_iq[ 6];
        tx_data_64[ 8] <= tx_data_iq[ 7];
        tx_data_64[ 7] <= tx_data_iq[ 8];
        tx_data_64[ 6] <= tx_data_iq[ 9];
        tx_data_64[ 5] <= tx_data_iq[10];
        tx_data_64[ 4] <= tx_data_iq[11];
        tx_data_64[ 3] <= tx_data_iq[12];
        tx_data_64[ 2] <= tx_data_iq[13];
        tx_data_64[ 1] <= tx_data_iq[14];
        tx_data_64[ 0] <= tx_data_iq[15];
      end
      4'b1011: begin
        tx_data_64[63] <= tx_data_iq[63];
        tx_data_64[62] <= tx_data_iq[62];
        tx_data_64[61] <= tx_data_iq[61];
        tx_data_64[60] <= tx_data_iq[60];
        tx_data_64[59] <= tx_data_iq[48];
        tx_data_64[58] <= tx_data_iq[49];
        tx_data_64[57] <= tx_data_iq[50];
        tx_data_64[56] <= tx_data_iq[51];
        tx_data_64[55] <= tx_data_iq[52];
        tx_data_64[54] <= tx_data_iq[53];
        tx_data_64[53] <= tx_data_iq[54];
        tx_data_64[52] <= tx_data_iq[55];
        tx_data_64[51] <= tx_data_iq[56];
        tx_data_64[50] <= tx_data_iq[57];
        tx_data_64[49] <= tx_data_iq[58];
        tx_data_64[48] <= tx_data_iq[59];
        tx_data_64[47] <= tx_data_iq[47];
        tx_data_64[46] <= tx_data_iq[46];
        tx_data_64[45] <= tx_data_iq[45];
        tx_data_64[44] <= tx_data_iq[44];
        tx_data_64[43] <= tx_data_iq[32];
        tx_data_64[42] <= tx_data_iq[33];
        tx_data_64[41] <= tx_data_iq[34];
        tx_data_64[40] <= tx_data_iq[35];
        tx_data_64[39] <= tx_data_iq[36];
        tx_data_64[38] <= tx_data_iq[37];
        tx_data_64[37] <= tx_data_iq[38];
        tx_data_64[36] <= tx_data_iq[39];
        tx_data_64[35] <= tx_data_iq[40];
        tx_data_64[34] <= tx_data_iq[41];
        tx_data_64[33] <= tx_data_iq[42];
        tx_data_64[32] <= tx_data_iq[43];
        tx_data_64[31] <= tx_data_iq[31];
        tx_data_64[30] <= tx_data_iq[30];
        tx_data_64[29] <= tx_data_iq[29];
        tx_data_64[28] <= tx_data_iq[28];
        tx_data_64[27] <= tx_data_iq[16];
        tx_data_64[26] <= tx_data_iq[17];
        tx_data_64[25] <= tx_data_iq[18];
        tx_data_64[24] <= tx_data_iq[19];
        tx_data_64[23] <= tx_data_iq[20];
        tx_data_64[22] <= tx_data_iq[21];
        tx_data_64[21] <= tx_data_iq[22];
        tx_data_64[20] <= tx_data_iq[23];
        tx_data_64[19] <= tx_data_iq[24];
        tx_data_64[18] <= tx_data_iq[25];
        tx_data_64[17] <= tx_data_iq[26];
        tx_data_64[16] <= tx_data_iq[27];
        tx_data_64[15] <= tx_data_iq[15];
        tx_data_64[14] <= tx_data_iq[14];
        tx_data_64[13] <= tx_data_iq[13];
        tx_data_64[12] <= tx_data_iq[12];
        tx_data_64[11] <= tx_data_iq[ 0];
        tx_data_64[10] <= tx_data_iq[ 1];
        tx_data_64[ 9] <= tx_data_iq[ 2];
        tx_data_64[ 8] <= tx_data_iq[ 3];
        tx_data_64[ 7] <= tx_data_iq[ 4];
        tx_data_64[ 6] <= tx_data_iq[ 5];
        tx_data_64[ 5] <= tx_data_iq[ 6];
        tx_data_64[ 4] <= tx_data_iq[ 7];
        tx_data_64[ 3] <= tx_data_iq[ 8];
        tx_data_64[ 2] <= tx_data_iq[ 9];
        tx_data_64[ 1] <= tx_data_iq[10];
        tx_data_64[ 0] <= tx_data_iq[11];
      end
      4'b1100: begin
        tx_data_64[63] <= tx_data_iq[48];
        tx_data_64[62] <= tx_data_iq[49];
        tx_data_64[61] <= tx_data_iq[50];
        tx_data_64[60] <= tx_data_iq[51];
        tx_data_64[59] <= tx_data_iq[52];
        tx_data_64[58] <= tx_data_iq[53];
        tx_data_64[57] <= tx_data_iq[54];
        tx_data_64[56] <= tx_data_iq[55];
        tx_data_64[55] <= tx_data_iq[56];
        tx_data_64[54] <= tx_data_iq[57];
        tx_data_64[53] <= tx_data_iq[58];
        tx_data_64[52] <= tx_data_iq[59];
        tx_data_64[51] <= tx_data_iq[60];
        tx_data_64[50] <= tx_data_iq[61];
        tx_data_64[49] <= tx_data_iq[62];
        tx_data_64[48] <= tx_data_iq[63];
        tx_data_64[47] <= tx_data_iq[32];
        tx_data_64[46] <= tx_data_iq[33];
        tx_data_64[45] <= tx_data_iq[34];
        tx_data_64[44] <= tx_data_iq[35];
        tx_data_64[43] <= tx_data_iq[36];
        tx_data_64[42] <= tx_data_iq[37];
        tx_data_64[41] <= tx_data_iq[38];
        tx_data_64[40] <= tx_data_iq[39];
        tx_data_64[39] <= tx_data_iq[40];
        tx_data_64[38] <= tx_data_iq[41];
        tx_data_64[37] <= tx_data_iq[42];
        tx_data_64[36] <= tx_data_iq[43];
        tx_data_64[35] <= tx_data_iq[44];
        tx_data_64[34] <= tx_data_iq[45];
        tx_data_64[33] <= tx_data_iq[46];
        tx_data_64[32] <= tx_data_iq[47];
        tx_data_64[31] <= tx_data_iq[16];
        tx_data_64[30] <= tx_data_iq[17];
        tx_data_64[29] <= tx_data_iq[18];
        tx_data_64[28] <= tx_data_iq[19];
        tx_data_64[27] <= tx_data_iq[20];
        tx_data_64[26] <= tx_data_iq[21];
        tx_data_64[25] <= tx_data_iq[22];
        tx_data_64[24] <= tx_data_iq[23];
        tx_data_64[23] <= tx_data_iq[24];
        tx_data_64[22] <= tx_data_iq[25];
        tx_data_64[21] <= tx_data_iq[26];
        tx_data_64[20] <= tx_data_iq[27];
        tx_data_64[19] <= tx_data_iq[28];
        tx_data_64[18] <= tx_data_iq[29];
        tx_data_64[17] <= tx_data_iq[30];
        tx_data_64[16] <= tx_data_iq[31];
        tx_data_64[15] <= tx_data_iq[ 0];
        tx_data_64[14] <= tx_data_iq[ 1];
        tx_data_64[13] <= tx_data_iq[ 2];
        tx_data_64[12] <= tx_data_iq[ 3];
        tx_data_64[11] <= tx_data_iq[ 4];
        tx_data_64[10] <= tx_data_iq[ 5];
        tx_data_64[ 9] <= tx_data_iq[ 6];
        tx_data_64[ 8] <= tx_data_iq[ 7];
        tx_data_64[ 7] <= tx_data_iq[ 8];
        tx_data_64[ 6] <= tx_data_iq[ 9];
        tx_data_64[ 5] <= tx_data_iq[10];
        tx_data_64[ 4] <= tx_data_iq[11];
        tx_data_64[ 3] <= tx_data_iq[12];
        tx_data_64[ 2] <= tx_data_iq[13];
        tx_data_64[ 1] <= tx_data_iq[14];
        tx_data_64[ 0] <= tx_data_iq[15];
      end
      default: begin
        tx_data_64 <= tx_data_iq;
      end
    endcase
  end

  // transmit data to bytes
 
  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_bcnt_d <= 2'd0;
      tx_bcnt_2d <= 2'd0;
      tx_wrcnt <= 1'd0;
      tx_wrdata_0 <= 32'd0;
      tx_wrdata_1 <= 32'd0;
      tx_wrdata_2 <= 32'd0;
      tx_wrdata_3 <= 32'd0;
      tx_rdcnt <= 2'd3;
      tx_rddata <= 32'd0;
      tx_rddata_h <= 8'd0;
    end else begin
      tx_bcnt_d <= tx_bcnt;
      tx_bcnt_2d <= tx_bcnt_d;
      if ((tx_valid_64 == 1'b1) && (tx_valid_iq == 1'b0)) begin
        tx_wrcnt <= tx_rdcnt[1];
      end
      if ((tx_valid_64 == 1'b1) && (tx_wrcnt == 1'd0)) begin
        tx_wrdata_0 <= tx_data_64[31:0];
      end else if ((tx_rdvalid == 1'b1) && (tx_rdcnt == 2'd0)) begin
        tx_wrdata_0 <= 32'd0;
      end
      if ((tx_valid_64 == 1'b1) && (tx_wrcnt == 1'd0)) begin
        tx_wrdata_1 <= tx_data_64[63:32];
      end else if ((tx_rdvalid == 1'b1) && (tx_rdcnt == 2'd1)) begin
        tx_wrdata_1 <= 32'd0;
      end
      if ((tx_valid_64 == 1'b1) && (tx_wrcnt == 1'd1)) begin
        tx_wrdata_2 <= tx_data_64[31:0];
      end else if ((tx_rdvalid == 1'b1) && (tx_rdcnt == 2'd2)) begin
        tx_wrdata_2 <= 32'd0;
      end
      if ((tx_valid_64 == 1'b1) && (tx_wrcnt == 1'd1)) begin
        tx_wrdata_3 <= tx_data_64[63:32];
      end else if ((tx_rdvalid == 1'b1) && (tx_rdcnt == 2'd3)) begin
        tx_wrdata_3 <= 32'd0;
      end
      if (tx_rdvalid == 1'b1) begin
        tx_rdcnt <= tx_rdcnt + 1'b1;
        case (tx_rdcnt)
          2'd0: tx_rddata <= tx_wrdata_0;
          2'd1: tx_rddata <= tx_wrdata_1;
          2'd2: tx_rddata <= tx_wrdata_2;
          default: tx_rddata <= tx_wrdata_3;
        endcase
        tx_rddata_h <= {tx_rddata[19:16], tx_rddata[3:0]};
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_read <= 1'd0;
      tx_read_toggle <= 1'd0;
      tx_read_32 <= 1'd0;
      tx_bvalid <= 1'd0;
      tx_rdvalid <= 1'd0;
      tx_bcnt <= 2'd0;
    end else begin
      tx_read <= tx_read_32 & ~tx_read_toggle;
      if (tx_read_32 == 1'd1) begin
        tx_read_toggle <= ~tx_read_toggle;
      end
      tx_read_32 <= tx_rdvalid & tx_enable;
      if (tx_sdr1_ddr0 == 1'b1) begin
        tx_bvalid <= ~tx_bvalid;
      end else begin
        tx_bvalid <= 1'b1;
      end
      if (tx_bvalid == 1'b1) begin
        case (tx_mode)
          CMOS_1L_LS_32X1, CMOS_1L_PS_32X1, LVDS_1L_LS_32X1, LVDS_1L_PS_32X1,
          CMOS_1L_PS_16X1, CMOS_1L_PS_08X1: begin
            tx_rdvalid <= ~(|tx_bcnt);
            if (tx_bcnt >= 2'd3) begin
              tx_bcnt <= 2'd0;
            end else begin
              tx_bcnt <= tx_bcnt + 1'b1;
            end
          end
          LVDS_1L_LS_24X1, LVDS_1L_PS_24X1: begin
            tx_rdvalid <= ~(|tx_bcnt);
            if (tx_bcnt >= 2'd2) begin
              tx_bcnt <= 2'd0;
            end else begin
              tx_bcnt <= tx_bcnt + 1'b1;
            end
          end
          LVDS_2L_PS_24X1: begin
            tx_rdvalid <= ~tx_bcnt[1];
            if (tx_bcnt >= 2'd2) begin
              tx_bcnt <= 2'd0;
            end else begin
              tx_bcnt <= tx_bcnt + 1'b1;
            end
          end
          LVDS_2L_PS_32X1: begin
            tx_rdvalid <= ~(|tx_bcnt);
            if (tx_bcnt >= 2'd1) begin
              tx_bcnt <= 2'd0;
            end else begin
              tx_bcnt <= tx_bcnt + 1'b1;
            end
          end
          CMOS_1L_PS_02X1, CMOS_4L_PS_32X1: begin
            tx_rdvalid <= 1'b1;
            tx_bcnt <= 2'd0;
          end
          default: begin
            tx_rdvalid <= 1'b0;
            tx_bcnt <= 2'd0;
          end
        endcase
      end else begin
        tx_rdvalid <= 1'b0;
        tx_bcnt <= tx_bcnt;
      end
    end
  end

  always @(negedge tx_resetn or posedge tx_clk) begin
    if (tx_resetn == 1'b0) begin
      tx_bstrobe <= 8'h00; 
      tx_bdata_0 <= 8'h00; 
      tx_bdata_1 <= 8'h00; 
      tx_bdata_2 <= 8'h00; 
      tx_bdata_3 <= 8'h00;
    end else begin
      if (tx_bvalid == 1'b1) begin
        case (tx_mode)
          CMOS_1L_LS_32X1, CMOS_1L_PS_32X1, LVDS_1L_LS_32X1,
          LVDS_1L_PS_32X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= {1'b1, {7{tx_lstrb1_pstrb0}}};
                tx_bdata_0 <= tx_rddata[31:24];
              end 
              2'd1: begin
                tx_bstrobe <= {8{tx_lstrb1_pstrb0}};
                tx_bdata_0 <= tx_rddata[23:16];
              end 
              2'd2: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[15:8];
              end 
              default: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[7:0];
              end 
            endcase
            tx_bdata_1 <= 8'h00; 
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          CMOS_1L_PS_16X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[15:8];
              end 
              2'd1: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[7:0];
              end 
              2'd2: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[31:24];
              end 
              default: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[23:16];
              end 
            endcase
            tx_bdata_1 <= 8'h00; 
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          CMOS_1L_PS_08X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[7:0];
              end 
              2'd1: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[15:8];
              end 
              2'd2: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[23:16];
              end 
              default: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[31:24];
              end 
            endcase
            tx_bdata_1 <= 8'h00; 
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          LVDS_1L_LS_24X1, LVDS_1L_PS_24X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= {1'b1, {7{tx_lstrb1_pstrb0}}};
                tx_bdata_0 <= tx_rddata[27:20];
              end 
              2'd1: begin
                tx_bstrobe <= {{4{tx_lstrb1_pstrb0}}, 4'd0};
                tx_bdata_0 <= {tx_rddata[19:16], tx_rddata[11:8]};
              end 
              default: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[7:0];
              end 
            endcase
            tx_bdata_1 <= 8'h00; 
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          LVDS_2L_PS_24X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[27:20];
                tx_bdata_1 <= tx_rddata[11:4];
              end 
              2'd1: begin
                tx_bstrobe <= 8'h08;
                tx_bdata_0 <= {tx_rddata_h[3:0], tx_rddata[27:24]};
                tx_bdata_1 <= {tx_rddata_h[7:4], tx_rddata[11:8]};
              end 
              default: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[23:16];
                tx_bdata_1 <= tx_rddata[7:0];
              end 
            endcase
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          LVDS_2L_PS_32X1: begin
            case (tx_bcnt_2d)
              2'd0: begin
                tx_bstrobe <= 8'h80;
                tx_bdata_0 <= tx_rddata[31:24];
                tx_bdata_1 <= tx_rddata[15:8];
              end 
              default: begin
                tx_bstrobe <= 8'h00;
                tx_bdata_0 <= tx_rddata[23:16];
                tx_bdata_1 <= tx_rddata[7:0];
              end 
            endcase
            tx_bdata_2 <= 8'h00;
            tx_bdata_3 <= 8'h00;
          end
          CMOS_1L_PS_02X1: begin
            tx_bstrobe <= 8'haa;
            tx_bdata_0 <= {tx_rddata[1:0], tx_rddata[9:8],
              tx_rddata[17:16], tx_rddata[25:24]};
            tx_bdata_1 <= 8'h00;
            tx_bdata_2 <= 8'h00; 
            tx_bdata_3 <= 8'h00;
          end
          CMOS_4L_PS_32X1: begin
            tx_bstrobe <= 8'h80;
            tx_bdata_0 <= tx_rddata[23:16];
            tx_bdata_1 <= tx_rddata[31:24];
            tx_bdata_2 <= tx_rddata[ 7: 0];
            tx_bdata_3 <= tx_rddata[15: 8];
          end
        endcase
      end
    end
  end

  always @(posedge tx_clk) begin
    if (tx_sdr1_ddr0 == 1'b0) begin
      tx_pstrobe <= tx_bstrobe;
      tx_pdata_0 <= tx_bdata_0;
      tx_pdata_1 <= tx_bdata_1;
      tx_pdata_2 <= tx_bdata_2;
      tx_pdata_3 <= tx_bdata_3;
    end else if (tx_bvalid == 1'b0) begin
      tx_pstrobe[7:6] <= {2{tx_bstrobe[7]}};
      tx_pstrobe[5:4] <= {2{tx_bstrobe[6]}};
      tx_pstrobe[3:2] <= {2{tx_bstrobe[5]}};
      tx_pstrobe[1:0] <= {2{tx_bstrobe[4]}};
      tx_pdata_0[7:6] <= {2{tx_bdata_0[7]}};
      tx_pdata_0[5:4] <= {2{tx_bdata_0[6]}};
      tx_pdata_0[3:2] <= {2{tx_bdata_0[5]}};
      tx_pdata_0[1:0] <= {2{tx_bdata_0[4]}};
      tx_pdata_1[7:6] <= {2{tx_bdata_1[7]}};
      tx_pdata_1[5:4] <= {2{tx_bdata_1[6]}};
      tx_pdata_1[3:2] <= {2{tx_bdata_1[5]}};
      tx_pdata_1[1:0] <= {2{tx_bdata_1[4]}};
      tx_pdata_2[7:6] <= {2{tx_bdata_2[7]}};
      tx_pdata_2[5:4] <= {2{tx_bdata_2[6]}};
      tx_pdata_2[3:2] <= {2{tx_bdata_2[5]}};
      tx_pdata_2[1:0] <= {2{tx_bdata_2[4]}};
      tx_pdata_3[7:6] <= {2{tx_bdata_3[7]}};
      tx_pdata_3[5:4] <= {2{tx_bdata_3[6]}};
      tx_pdata_3[3:2] <= {2{tx_bdata_3[5]}};
      tx_pdata_3[1:0] <= {2{tx_bdata_3[4]}};
    end else begin
      tx_pstrobe[7:6] <= {2{tx_bstrobe[3]}};
      tx_pstrobe[5:4] <= {2{tx_bstrobe[2]}};
      tx_pstrobe[3:2] <= {2{tx_bstrobe[1]}};
      tx_pstrobe[1:0] <= {2{tx_bstrobe[0]}};
      tx_pdata_0[7:6] <= {2{tx_bdata_0[3]}};
      tx_pdata_0[5:4] <= {2{tx_bdata_0[2]}};
      tx_pdata_0[3:2] <= {2{tx_bdata_0[1]}};
      tx_pdata_0[1:0] <= {2{tx_bdata_0[0]}};
      tx_pdata_1[7:6] <= {2{tx_bdata_1[3]}};
      tx_pdata_1[5:4] <= {2{tx_bdata_1[2]}};
      tx_pdata_1[3:2] <= {2{tx_bdata_1[1]}};
      tx_pdata_1[1:0] <= {2{tx_bdata_1[0]}};
      tx_pdata_2[7:6] <= {2{tx_bdata_2[3]}};
      tx_pdata_2[5:4] <= {2{tx_bdata_2[2]}};
      tx_pdata_2[3:2] <= {2{tx_bdata_2[1]}};
      tx_pdata_2[1:0] <= {2{tx_bdata_2[0]}};
      tx_pdata_3[7:6] <= {2{tx_bdata_3[3]}};
      tx_pdata_3[5:4] <= {2{tx_bdata_3[2]}};
      tx_pdata_3[3:2] <= {2{tx_bdata_3[1]}};
      tx_pdata_3[1:0] <= {2{tx_bdata_3[0]}};
    end
  end

  // mcs to strobe delay
 
  always @(negedge tx_cpu_resetn or posedge tx_clk) begin
    if (tx_cpu_resetn == 1'b0) begin
      tx_mcs_toggle_d <= 1'd0;
      tx_mcs_latency_start <= 1'd1;
      tx_mcs_latency_stop <= 1'd0;
      tx_mcs_latency_incr <= 4'd8;
      tx_mcs_latency_enable <= 1'd0;
      tx_mcs_latency_count <= 16'd0;
    end else begin
      tx_mcs_toggle_d <= tx_mcs_toggle;
      tx_mcs_latency_start <= tx_mcs_toggle ^ tx_mcs_toggle_d;
      if (tx_pstrobe == 8'd0) begin
        tx_mcs_latency_stop <= 1'd0;
      end else begin
        tx_mcs_latency_stop <= 1'd1;
      end
      if (tx_pstrobe == 8'd0) begin
        tx_mcs_latency_incr <= 4'd8;
      end else if (tx_pstrobe[7] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd0;
      end else if (tx_pstrobe[6] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd1;
      end else if (tx_pstrobe[5] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd2;
      end else if (tx_pstrobe[4] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd3;
      end else if (tx_pstrobe[3] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd4;
      end else if (tx_pstrobe[2] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd5;
      end else if (tx_pstrobe[1] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd6;
      end else if (tx_pstrobe[0] == 1'd1) begin
        tx_mcs_latency_incr <= 4'd7;
      end
      if (tx_mcs_latency_start == 1'd1) begin
        tx_mcs_latency_enable <= ~tx_mcs_latency_stop;
        tx_mcs_latency_count <= tx_mcs_latency_incr;
      end else if (tx_mcs_latency_enable == 1'd1) begin
        tx_mcs_latency_enable <= ~tx_mcs_latency_stop;
        tx_mcs_latency_count <= tx_mcs_latency_count + tx_mcs_latency_incr;
      end
    end
  end

  // processor initialization and reset

  assign tx_init_ack = ~tx_init_req_d & tx_init_req;

  always @(negedge tx_cpu_resetn or posedge tx_clk) begin
    if (tx_cpu_resetn == 1'b0) begin
      tx_resetn <= 1'b0;
      tx_init_req_d <= 1'd0;
      tx_init_cnt <= 4'd0;
    end else begin
      if (tx_init_cnt[3] == 1'd1) begin
        tx_resetn <= 1'd0;
      end else if ((tx_mcs_sync == 1'd0) || ((tx_mcs_sync == 1'd1) &&
        (tx_mcs_latency_start == 1'd1))) begin
        tx_resetn <= 1'd1;
      end
      tx_init_req_d <= tx_init_req;
      if (tx_init_ack == 1'd1) begin
        tx_init_cnt <= 4'h8;
      end else if (tx_init_cnt[3] == 1'b1) begin
        tx_init_cnt <= tx_init_cnt + 1'b1;
      end
    end
  end

  // cpu reset (software or mcs controlled)
 
  assign axilite_mcs_toggle_p = axilite_mcs_toggle ^ axilite_mcs_toggle_d;

  always @(negedge axilite_resetn or posedge axilite_clk) begin
    if (axilite_resetn == 1'b0) begin
      axilite_delay_unlocked <= 1'd1;
      axilite_clk_enable <= 1'd0;
      axilite_cpu_resetn <= 1'd0;
      axilite_cpu_reset <= 1'd1;
      axilite_mcs_sync_d <= 1'd0;
      axilite_mcs_toggle_d <= 1'd0;
      axilite_mcs_resetn <= 1'd0;
      axilite_mcs_clk_enable <= 1'd0;
      axilite_mcs_count <= 4'd0;
      axilite_sw_resetn <= 1'd0;
      axilite_sw_clk_enable <= 1'd0;
      axilite_sw_reset_in_d <= 1'd0;
      axilite_sw_reset_count <= 4'h0;
    end else begin
      axilite_delay_unlocked <= ~axilite_delay_locked;
      axilite_clk_enable <= axilite_sw_clk_enable & axilite_mcs_clk_enable;
      axilite_cpu_resetn <= ~axilite_cpu_reset;
      axilite_cpu_reset <= ~(axilite_sw_resetn & axilite_mcs_resetn);
      axilite_mcs_sync_d <= axilite_mcs_sync;
      axilite_mcs_toggle_d <= axilite_mcs_toggle;
      if (axilite_mcs_sync == 1'd0) begin
        axilite_mcs_resetn <= 1'd1;
        axilite_mcs_clk_enable <= 1'd1;
        axilite_mcs_count <= 4'd0;
      end else if (axilite_mcs_sync_d == 1'd0) begin
        axilite_mcs_resetn <= 1'd0;
        axilite_mcs_clk_enable <= 1'd0;
        axilite_mcs_count <= 4'd0;
      end else if (axilite_mcs_toggle_p == 1'd1) begin
        if (axilite_mcs_count >= axilite_mcs_active) begin
          axilite_mcs_resetn <= 1'd1;
        end
        axilite_mcs_clk_enable <= 1'd1;
        axilite_mcs_count <= axilite_mcs_count + 1'd1;
      end
      axilite_sw_resetn <= ~axilite_sw_reset_count[3];
      axilite_sw_clk_enable <= axilite_sw_reset_in_d | ~ axilite_sw_reset_in;
      axilite_sw_reset_in_d <= axilite_sw_reset_in;
      if ((axilite_sw_reset_in_d == 1'd0) && (axilite_sw_reset_in == 1'd1)) begin
        axilite_sw_reset_count <= 4'h8;
      end else if (axilite_sw_reset_count[3] == 1'd1) begin
        axilite_sw_reset_count <= axilite_sw_reset_count + 1'd1;
      end
    end
  end

  // io buffers
 
  assign tx_cssi_data1_lssi_data0_p = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_p[0] : tx_data_p[1];
  assign tx_cssi_data0_lssi_data0_n = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_n[0] : tx_data_p[0];
  assign tx_cssi_data3_lssi_data1_p = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_p[1] : tx_data_p[3];
  assign tx_cssi_data2_lssi_data1_n = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_n[1] : tx_data_p[2];

  assign tx_gpio_in = tx_cssi_gpio_in;
  assign tx_cssi_gpio_out = tx_gpio_out;
  assign tx_cssi_gpio_enb = tx_gpio_enb;

  // instantiations

  cdc #(.DATA_WIDTH(1)) i_tx_mcs_cdc (
    .src_data                               (mcs_toggle),
    .dest_resetn                            (tx_cpu_resetn),
    .dest_clk                               (tx_clk),
    .dest_data                              (tx_mcs_toggle));

  cdc_cntrl #(.DATA_WIDTH(16)) i_tx_mcs_cdc_cntrl (
    .src_resetn                             (tx_cpu_resetn),
    .src_clk                                (tx_clk),
    .src_data                               (tx_mcs_latency_count),
    .dest_resetn                            (axilite_resetn),
    .dest_clk                               (axilite_clk),
    .dest_data                              (axilite_mcs_latency_count));

  axi_adrv9001_tx_fifo i_tx_fifo (
    .rx_clk                                 (rx_clk),
    .rx_resetn                              (rx_resetn),
    .rx_valid                               (rx_valid),
    .rx_data                                (rx_data),
    .tx_clk                                 (tx_clk),
    .tx_resetn                              (tx_resetn),
    .tx_read                                (tx_read),
    .tx_valid                               (tx_lb_valid),
    .tx_data                                (tx_lb_data),
    .axilite_clk                            (axilite_clk),
    .axilite_resetn                         (axilite_resetn),
    .axilite_ovf                            (axilite_lb_ovf),
    .axilite_unf                            (axilite_lb_unf));

  axi_adrv9001_tx_dgen i_tx_dgen (
    .tx_clk                                 (tx_clk),
    .tx_resetn                              (tx_resetn),
    .tx_read                                (tx_read),
    .tx_valid                               (tx_dgen_valid),
    .tx_data                                (tx_dgen_data),
    .tx_mcs_latency_start                   (tx_mcs_latency_start),
    .tx_pattern                             (tx_pattern),
    .tx_pattern_pulse                       (tx_pattern_pulse),
    .tx_pattern_delay                       (tx_pattern_delay),
    .tx_pattern_ins                         (tx_pattern_ins),
    .tx_pattern_done                        (tx_pattern_done),
    .tx_num_of_bits                         (tx_num_of_bits),
    .tx_data_sel                            (tx_data_sel));

  clk_mon i_tx_clk_mon (
    .clk                                    (tx_clk),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_clk_mon_count                  (axilite_clk_mon_count));

  axi_adrv9001_tx_if #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pdata_0 (
    .tx_resetn                              (tx_resetn),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_pdata                               (tx_pdata_0),
    .tx_data_p                              (tx_data_p[0]),
    .tx_data_n                              (tx_data_n[0]),
    .tx_buf_enable                          (tx_buf_enable),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_delay_wrdata                   (axilite_wrdelay_0),
    .axilite_delay_rddata                   (axilite_rddelay_0));

  axi_adrv9001_tx_if #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pdata_1 (
    .tx_resetn                              (tx_resetn),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_pdata                               (tx_pdata_1),
    .tx_data_p                              (tx_data_p[1]),
    .tx_data_n                              (tx_data_n[1]),
    .tx_buf_enable                          (tx_buf_enable),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_delay_wrdata                   (axilite_wrdelay_1),
    .axilite_delay_rddata                   (axilite_rddelay_1));

  axi_adrv9001_tx_if #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pdata_2 (
    .tx_resetn                              (tx_resetn),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_pdata                               (tx_pdata_2),
    .tx_data_p                              (tx_data_p[2]),
    .tx_data_n                              (tx_data_n[2]),
    .tx_buf_enable                          (tx_buf_enable),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_delay_wrdata                   (axilite_wrdelay_2),
    .axilite_delay_rddata                   (axilite_rddelay_2));

  axi_adrv9001_tx_if #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pdata_3 (
    .tx_resetn                              (tx_resetn),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_pdata                               (tx_pdata_3),
    .tx_data_p                              (tx_data_p[3]),
    .tx_data_n                              (tx_data_n[3]),
    .tx_buf_enable                          (tx_buf_enable),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_delay_wrdata                   (axilite_wrdelay_3),
    .axilite_delay_rddata                   (axilite_rddelay_3));

  axi_adrv9001_tx_if #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pstrobe (
    .tx_resetn                              (tx_resetn),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_pdata                               (tx_pstrobe),
    .tx_data_p                              (tx_cssi_strobe_lssi_strobe_p),
    .tx_data_n                              (tx_cssi_nc_lssi_strobe_n),
    .tx_buf_enable                          (tx_buf_enable),
    .axilite_resetn                         (axilite_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_delay_wrdata                   (axilite_wrdelay_s),
    .axilite_delay_rddata                   (axilite_rddelay_s));

  axi_adrv9001_tx_clk #(
    .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
    .IODELAY_GROUP                          (IODELAY_GROUP))
  i_tx_pclock (
    .rx_buf_clk                             (rx_buf_clk),
    .tx_ref_clk_p                           (tx_cssi_refclk_lssi_refclk_p),
    .tx_ref_clk_n                           (tx_cssi_nc_lssi_refclk_n),
    .tx_hs_clk                              (tx_hs_clk),
    .tx_clk                                 (tx_clk),
    .tx_clock_p                             (tx_cssi_clk_lssi_clk_p),
    .tx_clock_n                             (tx_cssi_nc_lssi_clk_n),
    .axilite_clk                            (axilite_clk),
    .axilite_resetn                         (axilite_resetn),
    .axilite_cpu_resetn                     (axilite_cpu_resetn),
    .axilite_wrdelay_int_c                  (axilite_wrdelay_int_c),
    .axilite_rddelay_int_c                  (axilite_rddelay_int_c),
    .axilite_wrdelay_ext_c                  (axilite_wrdelay_ext_c),
    .axilite_rddelay_ext_c                  (axilite_rddelay_ext_c),
    .axilite_wrdelay_c                      (axilite_wrdelay_c),
    .axilite_rddelay_c                      (axilite_rddelay_c),
    .axilite_buf_enable                     (axilite_buf_enable),
    .axilite_clk_enable                     (axilite_clk_enable),
    .axilite_clk_reset                      (axilite_cpu_reset),
    .axilite_clk_msb                        (axilite_clk_msb),
    .axilite_clk_sel                        (axilite_clk_sel));

  cdc_cntrl #(.DATA_WIDTH(57)) i_tx_cdc_cntrl (
    .src_resetn                             (axilite_resetn),
    .src_clk                                (axilite_clk),
    .src_data                               ({axilite_init_req,
                                              axilite_mcs_sync,
                                              axilite_data_sel,
                                              axilite_pattern,
                                              axilite_pattern_pulse,
                                              axilite_pattern_delay,
                                              axilite_pattern_ins,
                                              axilite_q1_i0,
                                              axilite_lsb1_msb0,
                                              axilite_sdr1_ddr0,
                                              axilite_mlane1_slane0,
                                              axilite_lstrb1_pstrb0,
                                              axilite_num_of_bits,
                                              axilite_buf_enable}),
    .dest_resetn                            (tx_cpu_resetn),
    .dest_clk                               (tx_clk),
    .dest_data                              ({tx_init_req,
                                              tx_mcs_sync,
                                              tx_data_sel,
                                              tx_pattern,
                                              tx_pattern_pulse,
                                              tx_pattern_delay,
                                              tx_pattern_ins,
                                              tx_q1_i0,
                                              tx_lsb1_msb0,
                                              tx_sdr1_ddr0,
                                              tx_mlane1_slane0,
                                              tx_lstrb1_pstrb0,
                                              tx_num_of_bits,
                                              tx_buf_enable}));

  cdc #(.DATA_WIDTH(1)) i_tx_cdc (
    .src_data                               (axilite_cpu_resetn),
    .dest_resetn                            (1'b1),
    .dest_clk                               (tx_clk),
    .dest_data                              (tx_cpu_resetn));

  generate
  if (IODELAY_CTRL_ENABLE == 0) begin

  assign delay_locked = 1'b1;

  end else begin

  (* IODELAY_GROUP = IODELAY_GROUP *)
  IDELAYCTRL i_delay_ctrl (
    .RST                                    (~sys_200m_resetn),
    .REFCLK                                 (sys_200m_clk),
    .RDY                                    (delay_locked));

  end
  endgenerate

  cdc_event #(.DATA_WIDTH(1)) i_axilite_init_cdc_event (
    .src_resetn                             (tx_cpu_resetn),
    .src_clk                                (tx_clk),
    .src_data                               (tx_init_ack),
    .dest_resetn                            (axilite_resetn),
    .dest_clk                               (axilite_clk),
    .dest_data                              (axilite_init_ack));

  cdc #(.DATA_WIDTH(3)) i_axilite_cdc (
    .src_data                               ({tx_pattern_done,
                                              delay_locked,
                                              mcs_toggle}),
    .dest_resetn                            (axilite_resetn),
    .dest_clk                               (axilite_clk),
    .dest_data                              ({axilite_pattern_done,
                                              axilite_delay_locked,
                                              axilite_mcs_toggle}));

  axi_adrv9001_tx_regs #(.ID(ID)) i_tx_regs (
    .axilite_tx_cmos_lvds_id                (axilite_cmos_lvds_id),
    .axilite_tx_num_of_bits                 (axilite_num_of_bits),
    .axilite_tx_lstrb1_pstrb0               (axilite_lstrb1_pstrb0),
    .axilite_tx_mlane1_slane0               (axilite_mlane1_slane0),
    .axilite_tx_sdr1_ddr0                   (axilite_sdr1_ddr0),
    .axilite_tx_lsb1_msb0                   (axilite_lsb1_msb0),
    .axilite_tx_q1_i0                       (axilite_q1_i0),
    .axilite_tx_cmos1_lvds0                 (axilite_cmos1_lvds0),
    .axilite_tx_buf_enable                  (axilite_buf_enable),
    .axilite_tx_clk_sel                     (axilite_clk_sel),
    .axilite_tx_clk_msb                     (axilite_clk_msb),
    .axilite_tx_init                        (axilite_init_req),
    .axilite_tx_init_clr                    (axilite_init_ack),
    .axilite_tx_data_sel                    (axilite_data_sel),
    .axilite_tx_data_pat                    (axilite_pattern),
    .axilite_tx_data_pat_pulse              (axilite_pattern_pulse),
    .axilite_tx_data_pat_delay              (axilite_pattern_delay),
    .axilite_tx_data_pat_ins                (axilite_pattern_ins),
    .axilite_tx_data_pat_ins_clr            (axilite_pattern_done),
    .axilite_tx_lbovf                       (axilite_lb_ovf),
    .axilite_tx_lbunf                       (axilite_lb_unf),
    .axilite_tx_delay_unlocked              (axilite_delay_unlocked),
    .axilite_tx_wrdelay_0                   (axilite_wrdelay_0),
    .axilite_tx_rddelay_0                   (axilite_rddelay_0),
    .axilite_tx_wrdelay_1                   (axilite_wrdelay_1),
    .axilite_tx_rddelay_1                   (axilite_rddelay_1),
    .axilite_tx_wrdelay_2                   (axilite_wrdelay_2),
    .axilite_tx_rddelay_2                   (axilite_rddelay_2),
    .axilite_tx_wrdelay_3                   (axilite_wrdelay_3),
    .axilite_tx_rddelay_3                   (axilite_rddelay_3),
    .axilite_tx_wrdelay_s                   (axilite_wrdelay_s),
    .axilite_tx_rddelay_s                   (axilite_rddelay_s),
    .axilite_tx_wrdelay_c                   (axilite_wrdelay_c),
    .axilite_tx_rddelay_c                   (axilite_rddelay_c),
    .axilite_tx_wrdelay_int_c               (axilite_wrdelay_int_c),
    .axilite_tx_rddelay_int_c               (axilite_rddelay_int_c),
    .axilite_tx_wrdelay_ext_c               (axilite_wrdelay_ext_c),
    .axilite_tx_rddelay_ext_c               (axilite_rddelay_ext_c),
    .axilite_tx_timer                       (),
    .axilite_tx_clk_mon_count               (axilite_clk_mon_count),
    .axilite_tx_mcs_latency_count           (axilite_mcs_latency_count),
    .axilite_tx_mcs_sync                    (axilite_mcs_sync),
    .axilite_tx_mcs_active                  (axilite_mcs_active),
    .axilite_tx_sw_reset                    (axilite_sw_reset_in),
    .axilite_tx_sw_reset_clr                (axilite_cpu_resetn),
    .axilite_clk                            (axilite_clk),
    .axilite_resetn                         (axilite_resetn),
    .axilite_wrreq                          (axilite_wrreq),
    .axilite_wraddr                         (axilite_wraddr),
    .axilite_wrdata                         (axilite_wrdata),
    .axilite_wrack                          (axilite_wrack),
    .axilite_rdreq                          (axilite_rdreq),
    .axilite_rdaddr                         (axilite_rdaddr),
    .axilite_rddata                         (axilite_rddata),
    .axilite_rdack                          (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
