#!/usr/bin/perl
## ##################################################################################
## ##################################################################################
## ----------------------------------------------------------------------------------
## ################
## ##   ###########   Analog Devices Inc.
## ##      ########
## ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
## ##            ##   This file is the confidential and proprietary property of ADI.
## ##         #####   Possession or use of this file requires a written license.
## ##      ########   The licensing information may be found at: www.analog.com
## ##   ###########
## ################
## ----------------------------------------------------------------------------------
## Author:            Rejeesh Kutty
## Description:       AXI_ADRV9001 TDD registers
## ----------------------------------------------------------------------------------
## ##################################################################################
## ##################################################################################

use autodie;
require 'dooku.pl';

$addr_width = 16;
$addr_width_used = 12;

setdooku(@ARGV);
module(0x0, 'axi_adrv9001_tdd');

## ##################################################################################
## ##################################################################################
## Timing Reference. The following registers control the timing of all TDD related
## signals. Hardware maintains the following two counters.
##
## FRAME-PERIOD-COUNT:
##    1.  The counter starts if **BOTH** the following conditions are true.
##            A.  Software sets (write 0x1) to the START bit. (NOTE: the START bit
##                must be read as 0x0, otherwise see below).
##          AND
##            B.  A trigger condition set by the TRIGGER_MODE bits must occur.
##                Hardware supports four trigger inputs. This allows synchronization,
##                as an example the TDD frame may optionally be started with a PPS
##                event or device MCS event.
##    2.  The counter resets if **EITHER** of the following conditions are true.
##            A.  The count has reached FRAME_PERIOD value.
##          OR
##            B.  The count has reached FRAME_SWITCH_PERIOD value **AND**
##                FRAME-COUNT has reached the FRAME_SWITCH_NUMBER value **AND**
##                FRAME_SWITCH_ENABLE bit is set.
##                This is a premature termination of the counter.
##    3.  The counter restarts if
##            A.  The count has reached its reset point (#2 above) **AND**
##                FRAME-COUNT has reached the NUM_OF_FRAMES value **AND**
##                SEQUENCE_REPEAT bit is set **AND**
##                STOP bit is **NOT** set.
##                This is the graceful termination of the counter.
##
## FRAME-COUNT:
##    The frame count follows the FRAME-PERIOD-COUNT, with the exception that on
##    premature termination of the FRAME-PERIOD-COUNT (#2 above) it does **NOT**
##    restart or stop, instead it advances to the next frame. That is, the frame
##    is **SHORTENED** but seuqnece continues.
## ##################################################################################
## ##################################################################################

register(0x000, 'REG_FRAME_PERIOD');
field(0, 32, RW, 0x0, 'FRAME_PERIOD', 'The number of clock cycles in the TDD frame');

register(0x004, 'REG_NUM_OF_FRAMES');
field(0, 32, RW, 0x0, 'NUM_OF_FRAMES', 'The number of frames in the TDD sequence');

register(0x008, 'REG_FRAME_SWITCH_PERIOD');
field(0, 32, RW, 0x0, 'FRAME_SWITCH_PERIOD', 'The clock cycles value for premature
  termination of the current frame.');

register(0x00c, 'REG_FRAME_SWITCH_NUMBER');
field(0, 32, RW, 0x0, 'FRAME_SWITCH_NUMBER', 'The frame number for premature
  termination of the current clock count. This will be a shortened sequence.');

register(0x010, 'REG_TRIGGER_MODE');
field( 0,  3, RW, 0x0, 'TRIGGER_MODE_0', 'The start of a TDD sequence may optionally
  be synchronized to the hardware events. There are four trigger inputs, these bits
  controls each of the trigger inputs independently.
    0x0: This trigger source is disabled or ignored.
    0x4: TDD is started at the rising edge of this trigger input.
    0x5: TDD is started at the falling edge of this trigger input.
    0x6: TDD is started at any edge of this trigger input.');
field( 4,  3, RW, 0x0, 'TRIGGER_MODE_1', 'The same as above for trigger input 1');
field( 8,  3, RW, 0x0, 'TRIGGER_MODE_2', 'The same as above for trigger input 2');
field(12,  3, RW, 0x0, 'TRIGGER_MODE_3', 'The same as above for trigger input 3');

register(0x020, 'REG_SEQUENCE_CONTROL');
field(0, 1, RW1HC, 0x0, 'START', 'If set, indicates a request to start the TDD
  sequence, all sequence parameters above must be programmed first before setting
  this bit. Hardware clears this bit once the sequence is completed OR a stop
  request is initiated.');
field(1, 1, RW1HC, 0x0, 'STOP', 'If set, indicates a request to stop the current
  sequence, Hardware clears this bit once the sequence is stopped.');
field(2, 1, RW, 0x0, 'SEQUENCE_REPEAT', 'If set, the current sequence is repeated
  until a STOP request is received.');
field(3, 1, RW, 0x0, 'FRAME_SWITCH_ENABLE', 'If set, a sequence is shortened and the
  set frame is prematurely terminated.');

register(0x024, 'REG_SWRESET');
field(0, 1, RW1HC, 0x0, 'SWRESET', 'If set, the counters are reset. Hardware clears
  this bit once the internal reset is completed.');

register(0x028, 'REG_TIMER');
field(0, 32, RW, 0x0, 'TIMER', 'General purpose timer register, this is a count
  down timer, software may write desired interval and wait for it to reach zero.
  The count can be prematurely terminated, runs at AXILite clock (usually 100MHz).');

register(0x030, 'REG_CLK_MON');
field(0, 32, RO, 0x0, 'CLK_MON_COUNT', 'Clock monitor frequency count. If read 0x0,
  indicates a loss of clock. The format is 16.16 x 100MHz');

## ##################################################################################
## ##################################################################################
## Enable Generation. The individual enables below are all controlled by a set of
## programmable registers as noted below.
##
##  The Enable signal is ASSERTED at a TDD clock cycle if:
##      1.  FRAME-PERIOD-COUNT has reached the PRIMARY_ASSERT value **AND**
##          FRAME-COUNT has reached the PRIMARY_FRAME_ASSERT value
##    OR
##      2.  FRAME-PERIOD-COUNT has reached the SECONDARY_ASSERT value **AND**
##          FRAME-COUNT has reached the SECONDARY_FRAME_ASSERT value
##
##  The Enable signal is DEASSERTED at a TDD clock cycle if:
##      1.  FRAME-PERIOD-COUNT has reached the PRIMARY_DEASSERT value **AND**
##          FRAME-COUNT has reached the PRIMARY_FRAME_DEASSERT value
##    OR
##      2.  FRAME-PERIOD-COUNT has reached the SECONDARY_DEASSERT value **AND**
##          FRAME-COUNT has reached the SECONDARY_FRAME_DEASSERT value
## ##################################################################################
## ##################################################################################

$enable_list[ 0] = "RX0_DEVICE";
$enable_list[ 1] = "TX0_DEVICE";
$enable_list[ 2] = "RX1_DEVICE";
$enable_list[ 3] = "TX1_DEVICE";
$enable_list[ 4] = "ORX0_GPIO";
$enable_list[ 5] = "ORX1_GPIO";
$enable_list[ 6] = "AUX0_GPIO";
$enable_list[ 7] = "AUX1_GPIO";
$enable_list[ 8] = "CTL0_GPIO";
$enable_list[ 9] = "CTL1_GPIO";
$enable_list[10] = "CTL2_GPIO";
$enable_list[11] = "CTL3_GPIO";
$enable_list[12] = "TRIG0_EXT";
$enable_list[13] = "TRIG1_EXT";
$enable_cnt = 14;

for ($n = 0; $n < $enable_cnt; $n++) {

  register(((($n+1)*0x100)+0x00), "REG_$enable_list[$n]_PRIMARY_ASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_PRIMARY_ASSERT", "The clock count at which the
    $enable_list[$n]_ENABLE signal is asserted high.");

  register(((($n+1)*0x100)+0x04), "REG_$enable_list[$n]_PRIMARY_DEASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_PRIMARY_DEASSERT", "The clock count at which the
    $enable_list[$n]_ENABLE signal is deasserted low.");

  register(((($n+1)*0x100)+0x08), "REG_$enable_list[$n]_SECONDARY_ASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_SECONDARY_ASSERT", "The clock count at which the
    $enable_list[$n]_ENABLE signal is asserted high.");

  register(((($n+1)*0x100)+0x0c), "REG_$enable_list[$n]_SECONDARY_DEASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_SECONDARY_DEASSERT", "The clock count at which the
    $enable_list[$n]_ENABLE signal is deasserted low.");

  register(((($n+1)*0x100)+0x10), "REG_$enable_list[$n]_PRIMARY_FRAME_ASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_PRIMARY_FRAME_ASSERT", "The frame count at which the
    $enable_list[$n]_ENABLE signal is asserted high.");

  register(((($n+1)*0x100)+0x14), "REG_$enable_list[$n]_PRIMARY_FRAME_DEASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_PRIMARY_FRAME_DEASSERT", "The frame count at which the
    $enable_list[$n]_ENABLE signal is deasserted low.");

  register(((($n+1)*0x100)+0x18), "REG_$enable_list[$n]_SECONDARY_FRAME_ASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_SECONDARY_FRAME_ASSERT", "The frame count at which the
    $enable_list[$n]_ENABLE signal is asserted high.");

  register(((($n+1)*0x100)+0x1c), "REG_$enable_list[$n]_SECONDARY_FRAME_DEASSERT");
  field(0, 32, RW, 0x0, "$enable_list[$n]_SECONDARY_FRAME_DEASSERT", "The frame count at which the
    $enable_list[$n]_ENABLE signal is deasserted low.");

  register(((($n+1)*0x100)+0x20), "REG_$enable_list[$n]_MODE");
  field(0,  2, RW, 0x0, "$enable_list[$n]_MODE", "The software control of
    $enable_list[$n]_ENABLE signal.
      2b00: Software mode, enable is deasserted (low).
      2b01: Software mode, enable is asserted (high).
      2b1x: Hardware mode, enable is controlled by counters above.");
}

## ##################################################################################
## ##################################################################################
## NOTES:
##
##  1.  All counters start from 0x0 and upto and including the programmed values.
##        That is, program N-1 to all counters.
##  2.  If start bit is already set, setting it again will NOT restart the TDD
##        sequence, it is just ignored. Ideally if start bit is set, set STOP bit
##        the wait for both to clear and then proceed.
##  3.  If stop bit is already set, setting it again has no effect, if a reasonable
##        amount of time has passed and the bit doesn't self clear, reset is the
##        only option.
##  4.  There are no individual counter enable/disable bits. Instead, to disable the
##        enables, set the values to 0x0 (or same).
##  5.  It is possible to set DEASSERT values to be less than ASSERT values, this
##        will extend the enable over the frame boundary. However, be careful at the
##        sequence boundaries.
##  6.  Similar to #5 above, be careful about frame switch and premature termination.
##        Avoid (unless intentional) switching that masks assertion/de-assertion.
## ##################################################################################
## ##################################################################################

endregisters();
endmodule();
generate();

## ##################################################################################
## ##################################################################################
