// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       AXI_ADRV9001, receive data monitor
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_rx_dmon (

  // receive interface

  input   wire            rx_clk,
  input   wire            rx_resetn,
  input   wire            rx_valid,
  input   wire  [ 63:0]   rx_data,
  input   wire  [ 63:0]   rx_pattern,
  input   wire  [  2:0]   rx_data_sel,
  input   wire  [  2:0]   rx_num_of_bits,
  output  reg   [ 63:0]   rx_mon_data = 'd0,
  output  reg             rx_mon_oos = 'd0,
  output  reg             rx_mon_err = 'd0,
  output  reg             rx_pattern_state = 'd0,
  output  reg   [ 31:0]   rx_pattern_count = 'd0,

  // transmit interface

  input   wire            tx_pattern_done);

  // internal registers
 
  reg                     rx_valid_r = 'd0;
  reg           [ 63:0]   rx_data_r = 'd0;
  reg           [ 63:0]   rx_data_e = 'd0;
  reg           [ 63:0]   rx_ramp64_e = 'd0;
  reg           [ 63:0]   rx_nibb32_e = 'd0;
  reg           [ 63:0]   rx_ramp32_e = 'd0;
  reg           [ 63:0]   rx_pn1532_e = 'd0;
  reg           [ 63:0]   rx_pn0732_e = 'd0;
  reg           [ 63:0]   rx_ramp24_e = 'd0;
  reg           [ 63:0]   rx_ramp16_e = 'd0;
  reg           [ 63:0]   rx_ramp08_e = 'd0;
  reg           [ 63:0]   rx_ramp02_e = 'd0;
  reg           [  3:0]   rx_mon_cnt = 'd0;
  reg                     rx_pattern_done_d = 'd0;
  reg                     rx_lat_valid = 'd0;
  reg                     rx_lat_stop = 'd0;
  reg           [  3:0]   rx_lat_incr = 'd0;
  reg                     rx_lat64_stop = 'd0;
  reg           [  3:0]   rx_lat64_incr = 'd0;
  reg                     rx_lat32_stop = 'd0;
  reg           [  3:0]   rx_lat32_incr = 'd0;
  reg                     rx_lat24_stop = 'd0;
  reg           [  3:0]   rx_lat24_incr = 'd0;
  reg                     rx_lat16_stop = 'd0;
  reg           [  3:0]   rx_lat16_incr = 'd0;
  reg                     rx_lat08_stop = 'd0;
  reg           [  3:0]   rx_lat08_incr = 'd0;
  reg                     rx_lat02_stop = 'd0;
  reg           [  3:0]   rx_lat02_incr = 'd0;

  // internal signals

  wire          [ 63:0]   rx_patt64_e;
  wire          [ 63:0]   rx_ramp64_in;
  wire          [ 63:0]   rx_patt32_e;
  wire          [ 63:0]   rx_nibb32_in;
  wire          [ 63:0]   rx_ramp32_in;
  wire          [ 63:0]   rx_pn1532_in;
  wire          [ 63:0]   rx_pn1532_fn;
  wire          [ 63:0]   rx_pn0732_in;
  wire          [ 63:0]   rx_pn0732_fn;
  wire          [ 63:0]   rx_patt24_e;
  wire          [ 63:0]   rx_ramp24_in;
  wire          [ 63:0]   rx_patt16_e;
  wire          [ 63:0]   rx_ramp16_in;
  wire          [ 63:0]   rx_patt08_e;
  wire          [ 63:0]   rx_ramp08_in;
  wire          [ 63:0]   rx_patt02_e;
  wire          [ 63:0]   rx_ramp02_in;
  wire                    rx_match;
  wire                    rx_update;
  wire                    rx_pattern_done_p;
  wire                    rx_pattern_done;

  // pn7 function

  function [31:0] pn7;
    input [31:0] din;
    reg   [31:0] dout;
    begin
      dout[31] = din[ 6] ^ din[ 5];
      dout[30] = din[ 5] ^ din[ 4];
      dout[29] = din[ 4] ^ din[ 3];
      dout[28] = din[ 3] ^ din[ 2];
      dout[27] = din[ 2] ^ din[ 1];
      dout[26] = din[ 1] ^ din[ 0];
      dout[25] = din[ 0] ^ din[ 6] ^ din[ 5];
      dout[24] = din[ 6] ^ din[ 4];
      dout[23] = din[ 5] ^ din[ 3];
      dout[22] = din[ 4] ^ din[ 2];
      dout[21] = din[ 3] ^ din[ 1];
      dout[20] = din[ 2] ^ din[ 0];
      dout[19] = din[ 1] ^ din[ 6] ^ din[ 5];
      dout[18] = din[ 0] ^ din[ 5] ^ din[ 4];
      dout[17] = din[ 6] ^ din[ 4] ^ din[ 5] ^ din[ 3];
      dout[16] = din[ 5] ^ din[ 3] ^ din[ 4] ^ din[ 2];
      dout[15] = din[ 4] ^ din[ 2] ^ din[ 3] ^ din[ 1];
      dout[14] = din[ 3] ^ din[ 1] ^ din[ 2] ^ din[ 0];
      dout[13] = din[ 2] ^ din[ 0] ^ din[ 1] ^ din[ 6] ^ din[ 5];
      dout[12] = din[ 1] ^ din[ 6] ^ din[ 0] ^ din[ 4];
      dout[11] = din[ 0] ^ din[ 6] ^ din[ 3];
      dout[10] = din[ 6] ^ din[ 2];
      dout[ 9] = din[ 5] ^ din[ 1];
      dout[ 8] = din[ 4] ^ din[ 0];
      dout[ 7] = din[ 3] ^ din[ 6] ^ din[ 5];
      dout[ 6] = din[ 2] ^ din[ 5] ^ din[ 4];
      dout[ 5] = din[ 1] ^ din[ 4] ^ din[ 3];
      dout[ 4] = din[ 0] ^ din[ 3] ^ din[ 2];
      dout[ 3] = din[ 6] ^ din[ 2] ^ din[ 5] ^ din[ 1];
      dout[ 2] = din[ 5] ^ din[ 1] ^ din[ 4] ^ din[ 0];
      dout[ 1] = din[ 4] ^ din[ 0] ^ din[ 3] ^ din[ 6] ^ din[ 5];
      dout[ 0] = din[ 3] ^ din[ 6] ^ din[ 2] ^ din[ 4];
      pn7 = dout;
    end
  endfunction

  // pn15 function

  function [31:0] pn15;
    input [31:0] din;
    reg   [31:0] dout;
    begin
      dout[31] = din[14] ^ din[13];
      dout[30] = din[13] ^ din[12];
      dout[29] = din[12] ^ din[11];
      dout[28] = din[11] ^ din[10];
      dout[27] = din[10] ^ din[ 9];
      dout[26] = din[ 9] ^ din[ 8];
      dout[25] = din[ 8] ^ din[ 7];
      dout[24] = din[ 7] ^ din[ 6];
      dout[23] = din[ 6] ^ din[ 5];
      dout[22] = din[ 5] ^ din[ 4];
      dout[21] = din[ 4] ^ din[ 3];
      dout[20] = din[ 3] ^ din[ 2];
      dout[19] = din[ 2] ^ din[ 1];
      dout[18] = din[ 1] ^ din[ 0];
      dout[17] = din[ 0] ^ din[14] ^ din[13];
      dout[16] = din[14] ^ din[12];
      dout[15] = din[13] ^ din[11];
      dout[14] = din[12] ^ din[10];
      dout[13] = din[11] ^ din[ 9];
      dout[12] = din[10] ^ din[ 8];
      dout[11] = din[ 9] ^ din[ 7];
      dout[10] = din[ 8] ^ din[ 6];
      dout[ 9] = din[ 7] ^ din[ 5];
      dout[ 8] = din[ 6] ^ din[ 4];
      dout[ 7] = din[ 5] ^ din[ 3];
      dout[ 6] = din[ 4] ^ din[ 2];
      dout[ 5] = din[ 3] ^ din[ 1];
      dout[ 4] = din[ 2] ^ din[ 0];
      dout[ 3] = din[ 1] ^ din[14] ^ din[13];
      dout[ 2] = din[ 0] ^ din[13] ^ din[12];
      dout[ 1] = din[14] ^ din[12] ^ din[13] ^ din[11];
      dout[ 0] = din[13] ^ din[11] ^ din[12] ^ din[10];
      pn15 = dout;
    end
  endfunction

  // receive data sources
 
  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_mon_data <= rx_data;
    end
  end

  always @(posedge rx_clk) begin
    rx_valid_r <= rx_valid;
    rx_data_r <= rx_data;
    case ({rx_num_of_bits, rx_data_sel})
      6'b101011: rx_data_e <= rx_patt64_e;
      6'b101101: rx_data_e <= rx_ramp64_e;
      6'b100011: rx_data_e <= rx_patt32_e;
      6'b100100: rx_data_e <= rx_nibb32_e;
      6'b100101: rx_data_e <= rx_ramp32_e;
      6'b100110: rx_data_e <= rx_pn1532_e;
      6'b100111: rx_data_e <= rx_pn0732_e;
      6'b011011: rx_data_e <= rx_patt24_e;
      6'b011101: rx_data_e <= rx_ramp24_e;
      6'b010011: rx_data_e <= rx_patt16_e;
      6'b010101: rx_data_e <= rx_ramp16_e;
      6'b001011: rx_data_e <= rx_patt08_e;
      6'b001101: rx_data_e <= rx_ramp08_e;
      6'b000011: rx_data_e <= rx_patt02_e;
      6'b000101: rx_data_e <= rx_ramp02_e;
      default: rx_data_e <= {64{1'b1}};
    endcase
  end

  // 64bit (32I, 32Q) mode
 
  assign rx_patt64_e[63:0] = rx_pattern;

  assign rx_ramp64_in = (rx_mon_oos == 1'b0) ? rx_ramp64_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp64_e[63:32] <= rx_ramp64_in[31:0] + 1'h1;
      rx_ramp64_e[31: 0] <= rx_ramp64_in[31:0] + 1'h1;
    end
  end

  // 32bit (16I, 16Q) mode
 
  assign rx_patt32_e[63:32] = rx_pattern[31:0];
  assign rx_patt32_e[31: 0] = rx_pattern[31:0];

  assign rx_nibb32_in = (rx_mon_oos == 1'b0) ? rx_nibb32_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_nibb32_e[63:60] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[59:56] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[55:52] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[51:48] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[47:44] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[43:40] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[39:36] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[35:32] <= rx_nibb32_in[3:0] + 2'h3;
      rx_nibb32_e[31:28] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[27:24] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[23:20] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[19:16] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[15:12] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[11: 8] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[ 7: 4] <= rx_nibb32_in[3:0] + 2'h2;
      rx_nibb32_e[ 3: 0] <= rx_nibb32_in[3:0] + 2'h2;
    end
  end

  assign rx_ramp32_in = (rx_mon_oos == 1'b0) ? rx_ramp32_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp32_e[63:48] <= rx_ramp32_in[15:0] + 2'h3;
      rx_ramp32_e[47:32] <= rx_ramp32_in[15:0] + 2'h3;
      rx_ramp32_e[31:16] <= rx_ramp32_in[15:0] + 2'h2;
      rx_ramp32_e[15: 0] <= rx_ramp32_in[15:0] + 2'h2;
    end
  end

  assign rx_pn1532_in = (rx_mon_oos == 1'b0) ? rx_pn1532_e :
    ((rx_data == 64'd0) ? ~rx_data : rx_data);
  assign rx_pn1532_fn = pn15({rx_pn1532_in[47:32], rx_pn1532_in[15:0]});

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_pn1532_e[63:48] <= rx_pn1532_fn[31:16];
      rx_pn1532_e[47:32] <= rx_pn1532_fn[31:16];
      rx_pn1532_e[31:16] <= rx_pn1532_fn[15: 0];
      rx_pn1532_e[15: 0] <= rx_pn1532_fn[15: 0];
    end
  end

  assign rx_pn0732_in = (rx_mon_oos == 1'b0) ? rx_pn0732_e :
    ((rx_data == 64'd0) ? ~rx_data : rx_data);
  assign rx_pn0732_fn = pn7({rx_pn0732_in[47:32], rx_pn0732_in[15:0]});

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_pn0732_e[63:48] <= rx_pn0732_fn[31:16];
      rx_pn0732_e[47:32] <= rx_pn0732_fn[31:16];
      rx_pn0732_e[31:16] <= rx_pn0732_fn[15: 0];
      rx_pn0732_e[15: 0] <= rx_pn0732_fn[15: 0];
    end
  end

  // 24bit (12I, 12Q) mode
 
  assign rx_patt24_e[63:48] = {4'd0, rx_pattern[27:16]};
  assign rx_patt24_e[47:32] = {4'd0, rx_pattern[11: 0]};
  assign rx_patt24_e[31:16] = {4'd0, rx_pattern[27:16]};
  assign rx_patt24_e[15: 0] = {4'd0, rx_pattern[11: 0]};

  assign rx_ramp24_in = (rx_mon_oos == 1'b0) ? rx_ramp24_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp24_e[63:60] <= 4'd0;
      rx_ramp24_e[59:48] <= rx_ramp24_in[11:0] + 2'h3;
      rx_ramp24_e[47:44] <= 4'd0;
      rx_ramp24_e[43:32] <= rx_ramp24_in[11:0] + 2'h3;
      rx_ramp24_e[31:28] <= 4'd0;
      rx_ramp24_e[27:16] <= rx_ramp24_in[11:0] + 2'h2;
      rx_ramp24_e[15:12] <= 4'd0;
      rx_ramp24_e[11: 0] <= rx_ramp24_in[11:0] + 2'h2;
    end
  end

  // 16bit (16I) mode
 
  assign rx_patt16_e[63:48] = rx_pattern[15:0];
  assign rx_patt16_e[47:32] = rx_pattern[15:0];
  assign rx_patt16_e[31:16] = rx_pattern[15:0];
  assign rx_patt16_e[15: 0] = rx_pattern[15:0];

  assign rx_ramp16_in = (rx_mon_oos == 1'b0) ? rx_ramp16_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp16_e[63:48] <= rx_ramp16_in[15:0] + 3'h7;
      rx_ramp16_e[47:32] <= rx_ramp16_in[15:0] + 3'h6;
      rx_ramp16_e[31:16] <= rx_ramp16_in[15:0] + 3'h5;
      rx_ramp16_e[15: 0] <= rx_ramp16_in[15:0] + 3'h4;
    end
  end

  // 8bit (8I) mode
 
  assign rx_patt08_e[63:56] = rx_pattern[7:0];
  assign rx_patt08_e[55:48] = rx_pattern[7:0];
  assign rx_patt08_e[47:40] = rx_pattern[7:0];
  assign rx_patt08_e[39:32] = rx_pattern[7:0];
  assign rx_patt08_e[31:24] = rx_pattern[7:0];
  assign rx_patt08_e[23:16] = rx_pattern[7:0];
  assign rx_patt08_e[15: 8] = rx_pattern[7:0];
  assign rx_patt08_e[ 7: 0] = rx_pattern[7:0];

  assign rx_ramp08_in = (rx_mon_oos == 1'b0) ? rx_ramp08_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp08_e[63:56] <= rx_ramp08_in[7:0] + 4'hf;
      rx_ramp08_e[55:48] <= rx_ramp08_in[7:0] + 4'he;
      rx_ramp08_e[47:40] <= rx_ramp08_in[7:0] + 4'hd;
      rx_ramp08_e[39:32] <= rx_ramp08_in[7:0] + 4'hc;
      rx_ramp08_e[31:24] <= rx_ramp08_in[7:0] + 4'hb;
      rx_ramp08_e[23:16] <= rx_ramp08_in[7:0] + 4'ha;
      rx_ramp08_e[15: 8] <= rx_ramp08_in[7:0] + 4'h9;
      rx_ramp08_e[ 7: 0] <= rx_ramp08_in[7:0] + 4'h8;
    end
  end

  // 2bit (2I) mode
 
  assign rx_patt02_e[63:56] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[55:48] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[47:40] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[39:32] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[31:24] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[23:16] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[15: 8] = {6'd0, rx_pattern[1:0]};
  assign rx_patt02_e[ 7: 0] = {6'd0, rx_pattern[1:0]};

  assign rx_ramp02_in = (rx_mon_oos == 1'b0) ? rx_ramp02_e : rx_data;

  always @(posedge rx_clk) begin
    if (rx_valid == 1'b1) begin
      rx_ramp02_e[63:58] <= 6'd0;
      rx_ramp02_e[57:56] <= rx_ramp02_in[1:0] + 2'h3;
      rx_ramp02_e[55:50] <= 6'd0;
      rx_ramp02_e[49:48] <= rx_ramp02_in[1:0] + 2'h2;
      rx_ramp02_e[47:42] <= 6'd0;
      rx_ramp02_e[41:40] <= rx_ramp02_in[1:0] + 2'h1;
      rx_ramp02_e[39:34] <= 6'd0;
      rx_ramp02_e[33:32] <= rx_ramp02_in[1:0];
      rx_ramp02_e[31:26] <= 6'd0;
      rx_ramp02_e[25:24] <= rx_ramp02_in[1:0] + 2'h3;
      rx_ramp02_e[23:18] <= 6'd0;
      rx_ramp02_e[17:16] <= rx_ramp02_in[1:0] + 2'h2;
      rx_ramp02_e[15:10] <= 6'd0;
      rx_ramp02_e[ 9: 8] <= rx_ramp02_in[1:0] + 2'h1;
      rx_ramp02_e[ 7: 2] <= 6'd0;
      rx_ramp02_e[ 1: 0] <= rx_ramp02_in[1:0];
    end
  end

  // oos (16 good or bad data to go in or out of sync)

  assign rx_match = (rx_data_r == rx_data_e) ? 1'b1 : 1'b0;
  assign rx_update = ~(rx_mon_oos ^ rx_match);

  always @(negedge rx_resetn or posedge rx_clk) begin
    if (rx_resetn == 1'b0) begin
      rx_mon_err <= 1'd0;
      rx_mon_oos <= 1'd1;
      rx_mon_cnt <= 4'd0;
    end else begin
      if (rx_valid_r == 1'b1) begin
        rx_mon_err <= ~(rx_mon_oos | rx_match);
        if ((rx_update == 1'b1) && (rx_mon_cnt >= 4'hf)) begin
          rx_mon_oos <= ~rx_mon_oos;
        end
        if (rx_update == 1'b1) begin
          rx_mon_cnt <= rx_mon_cnt + 1'b1;
        end else begin
          rx_mon_cnt <= 4'd0;
        end
      end
    end
  end

  // round trip pattern latency
 
  assign rx_pattern_done_p = ~rx_pattern_done_d & rx_pattern_done;

  always @(negedge rx_resetn or posedge rx_clk) begin
    if (rx_resetn == 1'd0) begin
      rx_pattern_done_d <= 1'd0;
      rx_pattern_state <= 1'd0;
      rx_pattern_count <= 32'd0;
    end else begin
      rx_pattern_done_d <= rx_pattern_done;
      if ((rx_pattern_state == 1'd1) && (rx_lat_valid == 1'd1)) begin
        rx_pattern_state <= ~rx_lat_stop;
        if (rx_pattern_count[31] == 1'd0) begin
          rx_pattern_count <= rx_pattern_count + rx_lat_incr;
        end
      end else if (rx_pattern_done_p == 1'd1) begin
        rx_pattern_state <= ~rx_lat_valid | ~rx_lat_stop;
        rx_pattern_count <= (rx_lat_valid == 1'd1) ? rx_lat_incr : 32'd0;
      end
    end
  end

  always @(posedge rx_clk) begin
    rx_lat_valid <= rx_valid_r;
    case (rx_num_of_bits)
      3'b101: begin
        rx_lat_stop <= rx_lat64_stop;
        rx_lat_incr <= rx_lat64_incr;
      end
      3'b100: begin
        rx_lat_stop <= rx_lat32_stop;
        rx_lat_incr <= rx_lat32_incr;
      end
      3'b011: begin
        rx_lat_stop <= rx_lat24_stop;
        rx_lat_incr <= rx_lat24_incr;
      end
      3'b010: begin
        rx_lat_stop <= rx_lat16_stop;
        rx_lat_incr <= rx_lat16_incr;
      end
      3'b001: begin
        rx_lat_stop <= rx_lat08_stop;
        rx_lat_incr <= rx_lat08_incr;
      end
      3'b000: begin
        rx_lat_stop <= rx_lat02_stop;
        rx_lat_incr <= rx_lat02_incr;
      end
      default: begin
        rx_lat_stop <= 1'd0;
        rx_lat_incr <= 4'd0;
      end
    endcase
  end

  always @(posedge rx_clk) begin
    if (rx_data == rx_patt64_e) begin
      rx_lat64_stop <= 1'd1;
      rx_lat64_incr <= 4'd0;
    end else begin
      rx_lat64_stop <= 1'd0;
      rx_lat64_incr <= 4'd1;
    end
    if (rx_data[31:0] == rx_patt32_e[31:0]) begin
      rx_lat32_stop <= 1'd1;
      rx_lat32_incr <= 4'd0;
    end else if (rx_data[63:32] == rx_patt32_e[31:0]) begin
      rx_lat32_stop <= 1'd1;
      rx_lat32_incr <= 4'd1;
    end else begin
      rx_lat32_stop <= 1'd0;
      rx_lat32_incr <= 4'd2;
    end
    if (rx_data[31:0] == rx_patt24_e[31:0]) begin
      rx_lat24_stop <= 1'd1;
      rx_lat24_incr <= 4'd0;
    end else if (rx_data[63:32] == rx_patt24_e[31:0]) begin
      rx_lat24_stop <= 1'd1;
      rx_lat24_incr <= 4'd1;
    end else begin
      rx_lat24_stop <= 1'd0;
      rx_lat24_incr <= 4'd2;
    end
    if (rx_data[15:0] == rx_patt16_e[15:0]) begin
      rx_lat16_stop <= 1'd1;
      rx_lat16_incr <= 4'd0;
    end else if (rx_data[31:16] == rx_patt16_e[15:0]) begin
      rx_lat16_stop <= 1'd1;
      rx_lat16_incr <= 4'd1;
    end else if (rx_data[47:32] == rx_patt16_e[15:0]) begin
      rx_lat16_stop <= 1'd1;
      rx_lat16_incr <= 4'd2;
    end else if (rx_data[63:48] == rx_patt16_e[15:0]) begin
      rx_lat16_stop <= 1'd1;
      rx_lat16_incr <= 4'd3;
    end else begin
      rx_lat16_stop <= 1'd0;
      rx_lat16_incr <= 4'd4;
    end
    if (rx_data[7:0] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd0;
    end else if (rx_data[15:8] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd1;
    end else if (rx_data[23:16] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd2;
    end else if (rx_data[31:24] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd3;
    end else if (rx_data[39:32] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd4;
    end else if (rx_data[47:40] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd5;
    end else if (rx_data[55:48] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd6;
    end else if (rx_data[63:56] == rx_patt08_e[7:0]) begin
      rx_lat08_stop <= 1'd1;
      rx_lat08_incr <= 4'd7;
    end else begin
      rx_lat08_stop <= 1'd0;
      rx_lat08_incr <= 4'd8;
    end
    if (rx_data[7:0] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd0;
    end else if (rx_data[15:8] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd1;
    end else if (rx_data[23:16] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd2;
    end else if (rx_data[31:24] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd3;
    end else if (rx_data[39:32] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd4;
    end else if (rx_data[47:40] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd5;
    end else if (rx_data[55:48] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd6;
    end else if (rx_data[63:56] == rx_patt02_e[7:0]) begin
      rx_lat02_stop <= 1'd1;
      rx_lat02_incr <= 4'd7;
    end else begin
      rx_lat02_stop <= 1'd0;
      rx_lat02_incr <= 4'd8;
    end
  end

  // instantiations

  cdc #(.DATA_WIDTH(1)) i_cdc (
    .src_data       (tx_pattern_done),
    .dest_resetn    (rx_resetn),
    .dest_clk       (rx_clk),
    .dest_data      (rx_pattern_done));

endmodule

// **********************************************************************************
// **********************************************************************************
