// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty
// Description:       AXI_ADRV9001, receive decimation
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_rx_dec (

  // transmit interface

  input   wire            rx_clk,
  input   wire            rx_resetn,
  input   wire            rx_enable,
  input   wire            rx_valid,
  input   wire  [ 63:0]   rx_data,
  input   wire  [  3:0]   rx_dec_ratio,
  input   wire  [  2:0]   rx_num_of_bits,
  output  reg             rx_dec_valid = 'd0,
  output  reg   [ 63:0]   rx_dec_data = 'd0);

  // internal registers
 
  reg                     rx_dec00_valid = 'd0;
  reg           [ 63:0]   rx_dec08_data = 'd0;
  reg           [ 63:0]   rx_dec16_data = 'd0;
  reg           [ 63:0]   rx_dec32_data = 'd0;
  reg           [ 63:0]   rx_dec64_data = 'd0;
  reg           [  3:0]   rx_dec_cnt = 'd0;

  // internal signals

  wire          [  7:0]   rx_dec_sel;

  // decimation

  always @(posedge rx_clk) begin
    rx_dec_valid <= rx_enable & rx_dec00_valid;
    case ({rx_enable, rx_num_of_bits})
      4'b1101: rx_dec_data <= rx_dec64_data;
      4'b1100: rx_dec_data <= rx_dec32_data;
      4'b1011: rx_dec_data <= rx_dec32_data;
      4'b1010: rx_dec_data <= rx_dec16_data;
      4'b1001: rx_dec_data <= rx_dec08_data;
      4'b1000: rx_dec_data <= rx_dec08_data;
      default: rx_dec_data <= 64'd0;
    endcase
  end

  always @(posedge rx_clk) begin
    if (rx_dec_cnt >= rx_dec_ratio) begin
      rx_dec00_valid <= rx_valid;
    end else begin
      rx_dec00_valid <= 1'd0;
    end
  end

  always @(posedge rx_clk) begin
    if (rx_valid == 1'd1) begin
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h22, 8'h33, 8'h44, 8'h55, 8'h66, 8'h77,
        8'h87, 8'h98, 8'ha9, 8'hba, 8'hcb, 8'hdc, 8'hed, 8'hfe: begin
          case (rx_dec_ratio[2:0])
            3'd0: rx_dec08_data[63:56] <= rx_data[63:56];
            3'd1: rx_dec08_data[63:56] <= rx_data[55:48];
            3'd2: rx_dec08_data[63:56] <= rx_data[47:40];
            3'd3: rx_dec08_data[63:56] <= rx_data[39:32];
            3'd4: rx_dec08_data[63:56] <= rx_data[31:24];
            3'd5: rx_dec08_data[63:56] <= rx_data[23:16];
            3'd6: rx_dec08_data[63:56] <= rx_data[15: 8];
            3'd7: rx_dec08_data[63:56] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h22, 8'h33, 8'h43, 8'h54, 8'h65, 8'h76,
        8'h86, 8'h97, 8'ha8, 8'hb9, 8'hc9, 8'hda, 8'heb, 8'hfc: begin
          case (rx_dec_ratio[1:0])
            2'd0: rx_dec08_data[55:48] <= rx_data[55:48];
            2'd1: rx_dec08_data[55:48] <= rx_data[39:32];
            2'd2: rx_dec08_data[55:48] <= rx_data[23:16];
            2'd3: rx_dec08_data[55:48] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h21, 8'h32, 8'h43, 8'h53, 8'h64, 8'h75,
        8'h85, 8'h96, 8'ha6, 8'hb7, 8'hc8, 8'hd8, 8'he9, 8'hfa: begin
          case (rx_dec_ratio[2:0])
            3'd0: rx_dec08_data[47:40] <= rx_data[47:40];
            3'd1: rx_dec08_data[47:40] <= rx_data[23:16];
            3'd2: rx_dec08_data[47:40] <= rx_data[63:56];
            3'd3: rx_dec08_data[47:40] <= rx_data[39:32];
            3'd4: rx_dec08_data[47:40] <= rx_data[15: 8];
            3'd5: rx_dec08_data[47:40] <= rx_data[55:48];
            3'd6: rx_dec08_data[47:40] <= rx_data[31:24];
            3'd7: rx_dec08_data[47:40] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h21, 8'h32, 8'h42, 8'h53, 8'h63, 8'h74,
        8'h84, 8'h95, 8'ha5, 8'hb6, 8'hc6, 8'hd7, 8'he7, 8'hf8: begin
          case (rx_dec_ratio[0])
            1'd0: rx_dec08_data[39:32] <= rx_data[39:32];
            1'd1: rx_dec08_data[39:32] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h21, 8'h31, 8'h41, 8'h52, 8'h62, 8'h73,
        8'h83, 8'h93, 8'ha4, 8'hb4, 8'hc4, 8'hd5, 8'he5, 8'hf6: begin
          case (rx_dec_ratio[2:0])
            3'd0: rx_dec08_data[31:24] <= rx_data[31:24];
            3'd1: rx_dec08_data[31:24] <= rx_data[55:48];
            3'd2: rx_dec08_data[31:24] <= rx_data[15: 8];
            3'd3: rx_dec08_data[31:24] <= rx_data[39:32];
            3'd4: rx_dec08_data[31:24] <= rx_data[63:56];
            3'd5: rx_dec08_data[31:24] <= rx_data[23:16];
            3'd6: rx_dec08_data[31:24] <= rx_data[47:40];
            3'd7: rx_dec08_data[31:24] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h31, 8'h41, 8'h51, 8'h61, 8'h72,
        8'h82, 8'h92, 8'ha2, 8'hb3, 8'hc3, 8'hd3, 8'he3, 8'hf4: begin
          case (rx_dec_ratio[1:0])
            2'd0: rx_dec08_data[23:16] <= rx_data[23:16];
            2'd1: rx_dec08_data[23:16] <= rx_data[39:32];
            2'd2: rx_dec08_data[23:16] <= rx_data[55:48];
            2'd3: rx_dec08_data[23:16] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h30, 8'h40, 8'h50, 8'h60, 8'h71,
        8'h81, 8'h91, 8'ha1, 8'hb1, 8'hc1, 8'hd1, 8'he1, 8'hf2: begin
          case (rx_dec_ratio[2:0])
            3'd0: rx_dec08_data[15: 8] <= rx_data[15: 8];
            3'd1: rx_dec08_data[15: 8] <= rx_data[23:16];
            3'd2: rx_dec08_data[15: 8] <= rx_data[31:24];
            3'd3: rx_dec08_data[15: 8] <= rx_data[39:32];
            3'd4: rx_dec08_data[15: 8] <= rx_data[47:40];
            3'd5: rx_dec08_data[15: 8] <= rx_data[55:48];
            3'd6: rx_dec08_data[15: 8] <= rx_data[63:56];
            3'd7: rx_dec08_data[15: 8] <= rx_data[ 7: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h30, 8'h40, 8'h50, 8'h60, 8'h70,
        8'h80, 8'h90, 8'ha0, 8'hb0, 8'hc0, 8'hd0, 8'he0, 8'hf0: begin
          rx_dec08_data[ 7: 0] <= rx_data[ 7: 0];
        end
      endcase
    end
  end

  always @(posedge rx_clk) begin
    if (rx_valid == 1'd1) begin
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h22, 8'h33, 8'h43, 8'h54, 8'h65, 8'h76,
        8'h86, 8'h97, 8'ha8, 8'hb9, 8'hc9, 8'hda, 8'heb, 8'hfc: begin
          case (rx_dec_ratio[1:0])
            2'd0: rx_dec16_data[63:48] <= rx_data[63:48];
            2'd1: rx_dec16_data[63:48] <= rx_data[47:32];
            2'd2: rx_dec16_data[63:48] <= rx_data[31:16];
            2'd3: rx_dec16_data[63:48] <= rx_data[15: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h21, 8'h32, 8'h42, 8'h53, 8'h63, 8'h74,
        8'h84, 8'h95, 8'ha5, 8'hb6, 8'hc6, 8'hd7, 8'he7, 8'hf8: begin
          case (rx_dec_ratio[0])
            1'd0: rx_dec16_data[47:32] <= rx_data[47:32];
            1'd1: rx_dec16_data[47:32] <= rx_data[15: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h31, 8'h41, 8'h51, 8'h61, 8'h72,
        8'h82, 8'h92, 8'ha2, 8'hb3, 8'hc3, 8'hd3, 8'he3, 8'hf4: begin
          case (rx_dec_ratio[1:0])
            2'd0: rx_dec16_data[31:16] <= rx_data[31:16];
            2'd1: rx_dec16_data[31:16] <= rx_data[47:32];
            2'd2: rx_dec16_data[31:16] <= rx_data[63:48];
            2'd3: rx_dec16_data[31:16] <= rx_data[15: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h30, 8'h40, 8'h50, 8'h60, 8'h70,
        8'h80, 8'h90, 8'ha0, 8'hb0, 8'hc0, 8'hd0, 8'he0, 8'hf0: begin
          rx_dec16_data[15: 0] <= rx_data[15: 0];
        end
      endcase
    end
  end

  always @(posedge rx_clk) begin
    if (rx_valid == 1'd1) begin
      case (rx_dec_sel)
        8'h00, 8'h11, 8'h21, 8'h32, 8'h42, 8'h53, 8'h63, 8'h74,
        8'h84, 8'h95, 8'ha5, 8'hb6, 8'hc6, 8'hd7, 8'he7, 8'hf8: begin
          case (rx_dec_ratio[0])
            1'd0: rx_dec32_data[63:32] <= rx_data[63:32];
            1'd1: rx_dec32_data[63:32] <= rx_data[31: 0];
          endcase
        end
      endcase
      case (rx_dec_sel)
        8'h00, 8'h10, 8'h20, 8'h30, 8'h40, 8'h50, 8'h60, 8'h70,
        8'h80, 8'h90, 8'ha0, 8'hb0, 8'hc0, 8'hd0, 8'he0, 8'hf0: begin
          rx_dec32_data[31: 0] <= rx_data[31: 0];
        end
      endcase
    end
  end

  always @(posedge rx_clk) begin
    if (rx_dec_cnt >= rx_dec_ratio) begin
      rx_dec64_data <= rx_data;
    end else begin
      rx_dec64_data <= 64'd0;
    end
  end

  assign rx_dec_sel = {rx_dec_ratio, rx_dec_cnt};

  always @(negedge rx_resetn or posedge rx_clk) begin
    if (rx_resetn == 1'b0) begin
      rx_dec_cnt <= 4'd0;
    end else if (rx_valid == 1'b1) begin
      if (rx_dec_cnt >= rx_dec_ratio) begin
        rx_dec_cnt <= 4'd0;
      end else begin
        rx_dec_cnt <= rx_dec_cnt + 1'b1;
      end
    end
  end

endmodule

// **********************************************************************************
// **********************************************************************************
