// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty, Jim Marvill
// Description:       AXI_ADRV9001, MCS Generation
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_mcs #(

  parameter INSTANCE_ID = 0) (

  // system interface

  input   wire            dev_clk,
  input   wire            mcs_in_p,
  input   wire            mcs_in_n,
  output  wire            mcs_out_p,
  output  wire            mcs_out_n,
  output  reg             mcs_out_int = 1'd0,
  output  reg             mcs_toggle = 1'd0,
  input   wire            mcs_int,
  output  wire            mcs,
  
  // axilite interface

  input   wire            axilite_clk,
  input   wire            axilite_resetn,
  input   wire            axilite_mcs_request,
  output  wire            axilite_mcs_done,
  input   wire  [ 31:0]   axilite_mcs_period,
  input   wire  [  3:0]   axilite_mcs_width,
  input   wire  [  3:0]   axilite_mcs_num,
  input   wire            axilite_mcs_edge_fall1_rise0,
  input   wire            axilite_mcs_select,
  output  wire  [ 31:0]   axilite_mcs_clk_mon_count);

  // internal registers
 
  reg                     mcs_sw = 'd0;
  reg                     mcs_request_d = 'd0;
  reg                     mcs_status = 'd0;
  reg           [  3:0]   mcs_num_count = 'd0;
  reg           [ 31:0]   mcs_period_count = 'd0;
  reg                     mcs_d2 = 'd0;
  reg                     mcs_d1 = 'd0;

  // internal signals
 
  wire                    dev_resetn;
  wire                    mcs_hw;
  wire                    mcs_out;
  wire                    mcs_request;
  wire          [ 31:0]   mcs_period;
  wire          [  3:0]   mcs_width;
  wire          [  3:0]   mcs_num;
  wire                    mcs_edge_fall1_rise0;
  wire                    mcs_select;

  // mcs internal

  always @(negedge dev_resetn or posedge dev_clk) begin
    if (dev_resetn == 1'b0) begin
      mcs_out_int <= 1'd0;
      mcs_sw <= 1'b0;
      mcs_request_d <= 1'd0;
      mcs_status <= 1'd0;
      mcs_num_count <= 4'd0;
      mcs_period_count <= 32'd0;
    end else begin
      mcs_out_int <= (mcs_select == 1'd1) ? mcs_hw : mcs_sw;
      mcs_sw <= (mcs_period_count > mcs_width) ? 1'd0 : mcs_status;
      mcs_request_d <= mcs_request;
      if (mcs_status == 1'd1) begin
        if ((mcs_period_count >= mcs_period) && (mcs_num_count >= mcs_num)) begin
          mcs_status <= 1'd0;
        end
        if (mcs_period_count >= mcs_period) begin
          mcs_num_count <= mcs_num_count + 1'd1;
        end
        if (mcs_period_count >= mcs_period) begin
          mcs_period_count <= 8'd0;
        end else begin
          mcs_period_count <= mcs_period_count + 1'd1;
        end
      end else begin
        mcs_status <= mcs_request & ~mcs_request_d;
        mcs_num_count <= 4'd0;
        mcs_period_count <= 32'd0;
      end
    end
  end

  // mcs out

  assign mcs = (INSTANCE_ID == 0) ? mcs_out_int : mcs_int;

  always @(negedge dev_resetn or posedge dev_clk) begin
    if (dev_resetn == 1'b0) begin
      mcs_d2 <= 1'b0;
      mcs_d1 <= 1'b0;
      mcs_toggle <= 1'd0;
    end else begin
      mcs_d2 <= mcs;
      mcs_d1 <= (mcs_edge_fall1_rise0 == 1'd1) ? mcs_d2 : mcs;
      if ((mcs == 1'd1) && (mcs_d2 == 1'd0)) begin
        mcs_toggle <= ~mcs_toggle;
      end
    end
  end

  // instantiations

  IBUFDS i_ibuf_mcs (
    .I                      (mcs_in_p),
    .IB                     (mcs_in_n),
    .O                      (mcs_hw));

  OBUFDS i_obuf_mcs (
    .I                      (mcs_out),
    .O                      (mcs_out_p),
    .OB                     (mcs_out_n));

  ODDR #(.DDR_CLK_EDGE("SAME_EDGE")) i_oddr_mcs (
    .CE                     (1'b1),
    .R                      (1'b0),
    .S                      (1'b0),
    .C                      (dev_clk),
    .D1                     (mcs_d1),
    .D2                     (mcs_d2),
    .Q                      (mcs_out));

  clk_mon i_clk_mon (
    .clk                    (dev_clk),
    .axilite_resetn         (axilite_resetn),
    .axilite_clk            (axilite_clk),
    .axilite_clk_mon_count  (axilite_mcs_clk_mon_count));

  cdc_cntrl #(.DATA_WIDTH(43)) i_cdc_cntrl (
    .src_resetn             (axilite_resetn),
    .src_clk                (axilite_clk),
    .src_data               ({axilite_mcs_request,
                              axilite_mcs_period,
                              axilite_mcs_width,
                              axilite_mcs_num,
                              axilite_mcs_edge_fall1_rise0,
                              axilite_mcs_select}),
    .dest_resetn            (dev_resetn),
    .dest_clk               (dev_clk),
    .dest_data              ({mcs_request,
                              mcs_period,
                              mcs_width,
                              mcs_num,
                              mcs_edge_fall1_rise0,
                              mcs_select}));

  cdc_cntrl #(.DATA_WIDTH(1)) i_cdc_status (
    .src_resetn             (dev_resetn),
    .src_clk                (dev_clk),
    .src_data               (~mcs_status),
    .dest_resetn            (axilite_resetn),
    .dest_clk               (axilite_clk),
    .dest_data              (axilite_mcs_done));

  cdc #(.DATA_WIDTH(1)) i_cdc (
    .src_data               (axilite_resetn),
    .dest_resetn            (1'b1),
    .dest_clk               (dev_clk),
    .dest_data              (dev_resetn));

endmodule

// **********************************************************************************
// **********************************************************************************
