// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Author:            Rejeesh Kutty, Jim Marvill
// Description:       AXI_ADRV9001, GPIO
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_gpio #(

  parameter ID = 0) (

  // system interface

  inout   wire  [ 11:0]   dgpio,
  input   wire  [  7:0]   rx0_gpio_in,
  output  wire  [  7:0]   rx0_gpio_out,
  output  wire  [  7:0]   rx0_gpio_enb,
  input   wire  [  7:0]   rx1_gpio_in,
  output  wire  [  7:0]   rx1_gpio_out,
  output  wire  [  7:0]   rx1_gpio_enb,
  input   wire  [  9:0]   tx0_gpio_in,
  output  wire  [  9:0]   tx0_gpio_out,
  output  wire  [  9:0]   tx0_gpio_enb,
  input   wire  [  9:0]   tx1_gpio_in,
  output  wire  [  9:0]   tx1_gpio_out,
  output  wire  [  9:0]   tx1_gpio_enb,

  // gpio interface
 
  input   wire            gpio_clk,
  output  wire            gpio_resetn,
  output  reg             gpio_trig = 'd0,
  output  wire            gpio_gainindex_valid,
  output  reg   [  7:0]   gpio_gainindex = 'd0,

  // tdd enables

  input   wire            tdd_clk,
  input   wire            tdd_resetn,
  input   wire            tdd_tx0_dev_enable,
  input   wire            tdd_rx0_dev_enable,
  input   wire            tdd_tx1_dev_enable,
  input   wire            tdd_rx1_dev_enable,
  input   wire            tdd_orx0_gpio_enable,
  input   wire            tdd_orx1_gpio_enable,
  input   wire            tdd_aux0_gpio_enable,
  input   wire            tdd_aux1_gpio_enable,
  input   wire            tdd_ctl0_gpio_enable,
  input   wire            tdd_ctl1_gpio_enable,
  input   wire            tdd_ctl2_gpio_enable,
  input   wire            tdd_ctl3_gpio_enable,

  // axi interface

  input   wire            axilite_clk,
  input   wire            axilite_resetn,
  input   wire            axilite_wrreq,
  input   wire  [15:0]    axilite_wraddr,
  input   wire  [31:0]    axilite_wrdata,
  output  wire            axilite_wrack,
  input   wire            axilite_rdreq,
  input   wire  [15:0]    axilite_rdaddr,
  output  wire  [31:0]    axilite_rddata,
  output  wire            axilite_rdack);

  // internal registers
 
  reg           [ 11:0]   tdd_dgpio_out = 'd0;

  // internal signals
 
  wire          [ 11:0]   dgpio_in;
  wire          [ 11:0]   dgpio_out;
  wire          [ 11:0]   dgpio_enb;
  wire          [ 11:0]   dgpio_mode;
  wire          [ 11:0]   dgpio_data_out;
  wire          [ 47:0]   tdd_dgpio_sel;
  wire          [ 11:0]   axilite_dgpio_pio_data;
  wire          [ 47:0]   axilite_dgpio_pio_sel;
  wire          [  7:0]   axilite_rx0_pio_data;
  wire          [  7:0]   axilite_rx1_pio_data;
  wire          [  9:0]   axilite_tx0_pio_data;
  wire          [  9:0]   axilite_tx1_pio_data;
  wire          [  3:0]   axilite_trig_src_group;
  wire          [  3:0]   axilite_trig_src_pin;
  wire          [  3:0]   axilite_gainindex_src_group;
  wire          [ 31:0]   axilite_gainindex_src_pin;
  wire          [ 31:0]   axilite_clk_mon_count;
  wire          [ 11:0]   gpio_trig_in;
  wire          [ 11:0]   gpio_gainindex_in;
  wire          [ 11:0]   gpio_dgpio_data;
  wire          [  7:0]   gpio_rx0_data;
  wire          [  7:0]   gpio_rx1_data;
  wire          [  9:0]   gpio_tx0_data;
  wire          [  9:0]   gpio_tx1_data;
  wire          [  3:0]   gpio_trig_src_group;
  wire          [  3:0]   gpio_trig_src_pin;
  wire          [  3:0]   gpio_gainindex_src_group;
  wire          [ 31:0]   gpio_gainindex_src_pin;

  // digital gpio

  genvar n;
  generate
  for (n = 0; n < 12; n = n + 1) begin: g_dgpio

  assign dgpio[n] = (dgpio_enb[n] == 1'b1) ? dgpio_out[n] : 1'bz;

  assign dgpio_in[n] = dgpio[n];
  assign dgpio_out[n] = (dgpio_mode[n] == 1'b0) ?
    dgpio_data_out[n] : tdd_dgpio_out[n];

  always @(negedge tdd_resetn or posedge tdd_clk) begin
    if (tdd_resetn == 1'd0) begin
      tdd_dgpio_out[n] <= 1'd0;
    end else begin
      case (tdd_dgpio_sel[((n*4)+3):(n*4)])
        4'b0000: tdd_dgpio_out[n] <= tdd_tx0_dev_enable;
        4'b0001: tdd_dgpio_out[n] <= tdd_rx0_dev_enable;
        4'b0010: tdd_dgpio_out[n] <= tdd_orx0_gpio_enable;
        4'b0011: tdd_dgpio_out[n] <= tdd_aux0_gpio_enable;
        4'b0100: tdd_dgpio_out[n] <= tdd_tx1_dev_enable;
        4'b0101: tdd_dgpio_out[n] <= tdd_rx1_dev_enable;
        4'b0110: tdd_dgpio_out[n] <= tdd_orx1_gpio_enable;
        4'b0111: tdd_dgpio_out[n] <= tdd_aux1_gpio_enable;
        4'b1000: tdd_dgpio_out[n] <= tdd_ctl0_gpio_enable;
        4'b1001: tdd_dgpio_out[n] <= tdd_ctl1_gpio_enable;
        4'b1010: tdd_dgpio_out[n] <= tdd_ctl2_gpio_enable;
        4'b1011: tdd_dgpio_out[n] <= tdd_ctl3_gpio_enable;
        default: tdd_dgpio_out[n] <= 1'd0;
      endcase
    end
  end

  end
  endgenerate

  // trigger select

  assign gpio_trig_in = (gpio_trig_src_group == 4'd1) ? gpio_dgpio_data :
    ((gpio_trig_src_group == 4'd4) ? {4'd0, gpio_rx0_data} :
    ((gpio_trig_src_group == 4'd5) ? {4'd0, gpio_rx1_data} :
    ((gpio_trig_src_group == 4'd6) ? {2'd0, gpio_tx0_data} :
    ((gpio_trig_src_group == 4'd7) ? {2'd0, gpio_tx1_data} : 12'd0))));

  always @(negedge gpio_resetn or posedge gpio_clk) begin
    if (gpio_resetn == 1'd0) begin
      gpio_trig <= 1'd0;
    end else begin
      case (gpio_trig_src_pin)
        4'b0000: gpio_trig <= gpio_trig_in[ 0];
        4'b0001: gpio_trig <= gpio_trig_in[ 1];
        4'b0010: gpio_trig <= gpio_trig_in[ 2];
        4'b0011: gpio_trig <= gpio_trig_in[ 3];
        4'b0100: gpio_trig <= gpio_trig_in[ 4];
        4'b0101: gpio_trig <= gpio_trig_in[ 5];
        4'b0110: gpio_trig <= gpio_trig_in[ 6];
        4'b0111: gpio_trig <= gpio_trig_in[ 7];
        4'b1000: gpio_trig <= gpio_trig_in[ 8];
        4'b1001: gpio_trig <= gpio_trig_in[ 9];
        4'b1010: gpio_trig <= gpio_trig_in[10];
        4'b1011: gpio_trig <= gpio_trig_in[11];
        default: gpio_trig <= 1'd0;
      endcase
    end
  end

  // gain index

  assign gpio_gainindex_in = (gpio_gainindex_src_group == 4'd1) ? gpio_dgpio_data :
    ((gpio_gainindex_src_group == 4'd4) ? {4'd0, gpio_rx0_data} :
    ((gpio_gainindex_src_group == 4'd5) ? {4'd0, gpio_rx1_data} :
    ((gpio_gainindex_src_group == 4'd6) ? {2'd0, gpio_tx0_data} :
    ((gpio_gainindex_src_group == 4'd7) ? {2'd0, gpio_tx1_data} : 12'd0))));

  assign gpio_gainindex_valid = 1'd1;

  generate
  for (n = 0; n < 8; n = n + 1) begin: g_gainindex

  always @(negedge gpio_resetn or posedge gpio_clk) begin
    if (gpio_resetn == 1'd0) begin
      gpio_gainindex[n] <= 1'd0;
    end else begin
      case (gpio_gainindex_src_pin[((n*4)+3):(n*4)])
        4'b0000: gpio_gainindex[n] <= gpio_gainindex_in[ 0];
        4'b0001: gpio_gainindex[n] <= gpio_gainindex_in[ 1];
        4'b0010: gpio_gainindex[n] <= gpio_gainindex_in[ 2];
        4'b0011: gpio_gainindex[n] <= gpio_gainindex_in[ 3];
        4'b0100: gpio_gainindex[n] <= gpio_gainindex_in[ 4];
        4'b0101: gpio_gainindex[n] <= gpio_gainindex_in[ 5];
        4'b0110: gpio_gainindex[n] <= gpio_gainindex_in[ 6];
        4'b0111: gpio_gainindex[n] <= gpio_gainindex_in[ 7];
        4'b1000: gpio_gainindex[n] <= gpio_gainindex_in[ 8];
        4'b1001: gpio_gainindex[n] <= gpio_gainindex_in[ 9];
        4'b1010: gpio_gainindex[n] <= gpio_gainindex_in[10];
        4'b1011: gpio_gainindex[n] <= gpio_gainindex_in[11];
        default: gpio_gainindex[n] <= 1'd0;
      endcase
    end
  end

  end
  endgenerate

  // instantiations

  cdc #(.DATA_WIDTH(48)) i_cdc_gpio_data (
    .src_data                           ({dgpio_in,
                                          rx0_gpio_in,
                                          rx1_gpio_in,
                                          tx0_gpio_in,
                                          tx1_gpio_in}),
    .dest_resetn                        (gpio_resetn),
    .dest_clk                           (gpio_clk),
    .dest_data                          ({gpio_dgpio_data,
                                          gpio_rx0_data,
                                          gpio_rx1_data,
                                          gpio_tx0_data,
                                          gpio_tx1_data}));

  cdc_cntrl #(.DATA_WIDTH(44)) i_cdc_cntrl_gpio (
    .src_resetn                         (axilite_resetn),
    .src_clk                            (axilite_clk),
    .src_data                           ({axilite_trig_src_group,
                                          axilite_trig_src_pin,
                                          axilite_gainindex_src_group,
                                          axilite_gainindex_src_pin}),
    .dest_resetn                        (gpio_resetn),
    .dest_clk                           (gpio_clk),
    .dest_data                          ({gpio_trig_src_group,
                                          gpio_trig_src_pin,
                                          gpio_gainindex_src_group,
                                          gpio_gainindex_src_pin}));

  cdc #(.DATA_WIDTH(48)) i_cdc_axilite_data (
    .src_data                           ({dgpio_in,
                                          rx0_gpio_in,
                                          rx1_gpio_in,
                                          tx0_gpio_in,
                                          tx1_gpio_in}),
    .dest_resetn                        (axilite_resetn),
    .dest_clk                           (axilite_clk),
    .dest_data                          ({axilite_dgpio_pio_data,
                                          axilite_rx0_pio_data,
                                          axilite_rx1_pio_data,
                                          axilite_tx0_pio_data,
                                          axilite_tx1_pio_data}));

  cdc_cntrl #(.DATA_WIDTH(48)) i_cdc_cntrl_tdd (
    .src_resetn                         (axilite_resetn),
    .src_clk                            (axilite_clk),
    .src_data                           (axilite_dgpio_pio_sel),
    .dest_resetn                        (tdd_resetn),
    .dest_clk                           (tdd_clk),
    .dest_data                          (tdd_dgpio_sel));

  cdc #(.DATA_WIDTH(1)) i_cdc_resetn (
    .src_data                           (axilite_resetn),
    .dest_resetn                        (1'b1),
    .dest_clk                           (gpio_clk),
    .dest_data                          (gpio_resetn));

  clk_mon i_gpio_clk_mon (
    .clk                                (gpio_clk),
    .axilite_resetn                     (axilite_resetn),
    .axilite_clk                        (axilite_clk),
    .axilite_clk_mon_count              (axilite_clk_mon_count));

  axi_adrv9001_gpio_regs #(.ID(ID)) i_regs (
    .axilite_timer                      (),
    .axilite_dgpio_pio_data             (axilite_dgpio_pio_data),
    .axilite_dgpio_pio_data_out         (dgpio_data_out),
    .axilite_dgpio_pio_data_out_enb     (dgpio_enb),
    .axilite_dgpio_pio_data_mode        (dgpio_mode),
    .axilite_dgpio_pio_sel_00           (axilite_dgpio_pio_sel[(( 0*4)+3):( 0*4)]),
    .axilite_dgpio_pio_sel_01           (axilite_dgpio_pio_sel[(( 1*4)+3):( 1*4)]),
    .axilite_dgpio_pio_sel_02           (axilite_dgpio_pio_sel[(( 2*4)+3):( 2*4)]),
    .axilite_dgpio_pio_sel_03           (axilite_dgpio_pio_sel[(( 3*4)+3):( 3*4)]),
    .axilite_dgpio_pio_sel_04           (axilite_dgpio_pio_sel[(( 4*4)+3):( 4*4)]),
    .axilite_dgpio_pio_sel_05           (axilite_dgpio_pio_sel[(( 5*4)+3):( 5*4)]),
    .axilite_dgpio_pio_sel_06           (axilite_dgpio_pio_sel[(( 6*4)+3):( 6*4)]),
    .axilite_dgpio_pio_sel_07           (axilite_dgpio_pio_sel[(( 7*4)+3):( 7*4)]),
    .axilite_dgpio_pio_sel_08           (axilite_dgpio_pio_sel[(( 8*4)+3):( 8*4)]),
    .axilite_dgpio_pio_sel_09           (axilite_dgpio_pio_sel[(( 9*4)+3):( 9*4)]),
    .axilite_dgpio_pio_sel_10           (axilite_dgpio_pio_sel[((10*4)+3):(10*4)]),
    .axilite_dgpio_pio_sel_11           (axilite_dgpio_pio_sel[((11*4)+3):(11*4)]),
    .axilite_rx0_pio_data               (axilite_rx0_pio_data),
    .axilite_rx0_pio_data_out           (rx0_gpio_out),
    .axilite_rx0_pio_data_out_enb       (rx0_gpio_enb),
    .axilite_rx1_pio_data               (axilite_rx1_pio_data),
    .axilite_rx1_pio_data_out           (rx1_gpio_out),
    .axilite_rx1_pio_data_out_enb       (rx1_gpio_enb),
    .axilite_tx0_pio_data               (axilite_tx0_pio_data),
    .axilite_tx0_pio_data_out           (tx0_gpio_out),
    .axilite_tx0_pio_data_out_enb       (tx0_gpio_enb),
    .axilite_tx1_pio_data               (axilite_tx1_pio_data),
    .axilite_tx1_pio_data_out           (tx1_gpio_out),
    .axilite_tx1_pio_data_out_enb       (tx1_gpio_enb),
    .axilite_trig_src_group             (axilite_trig_src_group),
    .axilite_trig_src_pin               (axilite_trig_src_pin),
    .axilite_gainindex_src_group        (axilite_gainindex_src_group),
    .axilite_gainindex0_src_pin         (axilite_gainindex_src_pin[((0*4)+3):(0*4)]),
    .axilite_gainindex1_src_pin         (axilite_gainindex_src_pin[((1*4)+3):(1*4)]),
    .axilite_gainindex2_src_pin         (axilite_gainindex_src_pin[((2*4)+3):(2*4)]),
    .axilite_gainindex3_src_pin         (axilite_gainindex_src_pin[((3*4)+3):(3*4)]),
    .axilite_gainindex4_src_pin         (axilite_gainindex_src_pin[((4*4)+3):(4*4)]),
    .axilite_gainindex5_src_pin         (axilite_gainindex_src_pin[((5*4)+3):(5*4)]),
    .axilite_gainindex6_src_pin         (axilite_gainindex_src_pin[((6*4)+3):(6*4)]),
    .axilite_gainindex7_src_pin         (axilite_gainindex_src_pin[((7*4)+3):(7*4)]),
    .axilite_clk_mon_count              (axilite_clk_mon_count),
    .axilite_clk                        (axilite_clk),
    .axilite_resetn                     (axilite_resetn),
    .axilite_wrreq                      (axilite_wrreq),
    .axilite_wraddr                     (axilite_wraddr),
    .axilite_wrdata                     (axilite_wrdata),
    .axilite_wrack                      (axilite_wrack),
    .axilite_rdreq                      (axilite_rdreq),
    .axilite_rdaddr                     (axilite_rdaddr),
    .axilite_rddata                     (axilite_rddata),
    .axilite_rdack                      (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
