/**
* Copyright 2015 - 2018 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information.
* see the "LICENSE.txt" file in this zip file.
*/
#ifdef __KERNEL__
#include <linux/firmware.h>
#include <linux/kernel.h>
#include <linux/spi/spi.h>

#include "linux_fw.h"
#include "adi_platform_types.h"

int32_t linux_ImagePageGet(void *devHalCfg, const char *ImagePath, uint32_t pageIndex,
                           uint32_t pageSize, uint8_t *rdBuff)
{
    adi_hal_Cfg_t *hal = devHalCfg;
    struct spi_device *spi = hal->spi;
    const struct firmware *fw;
    int ret;

    ret = request_firmware(&fw, ImagePath, &spi->dev);
    if (ret) {
        dev_err(&spi->dev,
		"request_firmware(%s) failed with %d\n", ImagePath, ret);

        return ADI_HAL_BBICCTRL_FAIL;
    }

    if ((pageIndex * pageSize) > fw->size) {
        ret = ADI_HAL_BBICCTRL_FAIL;
        goto release_fw;
    }

    memcpy(rdBuff, &fw->data[pageIndex * pageSize], pageSize);

release_fw:
    release_firmware(fw);

    return ret;
}

int32_t linux_RxGainTableEntryGet(void *devHalCfg, const char *rxGainTablePath, uint16_t lineCount,
                                  uint8_t *gainIndex, uint8_t *rxFeGain, uint8_t *tiaControl,
                                  uint8_t *adcControl, uint8_t *extControl, uint16_t *phaseOffset,
                                  int16_t *digGain)
{
    adi_hal_Cfg_t *hal = devHalCfg;
    struct spi_device *spi = hal->spi;
    const struct firmware *fw;
    char *line;
    int i = 0, ret, cnt = 0;
    static const int NUM_COLUMN = 7;

    /* setting index to 0 will break the caller loop in case of failure */
    *gainIndex = 0;

    ret = request_firmware(&fw, rxGainTablePath, &spi->dev);
    if (ret) {
	dev_err(&spi->dev,
		"request_firmware(%s) failed with %d\n", rxGainTablePath, ret);
	return ADI_HAL_BBICCTRL_FAIL;
    }

    do {
	line = strnchr(fw->data + cnt, fw->size - cnt, '\n');
	if (!line)
		goto release_fw;
	line++;
	cnt = line - (char *)fw->data;
    } while (i++ != lineCount && cnt < fw->size);

    /* EOF */
    if (cnt >= fw->size) {
        ret = NUM_COLUMN;
        goto release_fw;
    }

    line = skip_spaces(line);

    ret = sscanf(line, "%hhu,%hhu,%hhu,%hhu,%hhu,%hu,%hd",
		 gainIndex,
		 rxFeGain,
		 tiaControl,
		 adcControl,
		 extControl,
		 phaseOffset,
		 digGain);

release_fw:
    release_firmware(fw);

    return ret;
}

int32_t linux_TxAttenTableEntryGet(void *devHalCfg, const char *txAttenTablePath,
                                   uint16_t lineCount, uint16_t *attenIndex, uint8_t *txAttenHp,
                                   uint16_t *txAttenMult)
{
    adi_hal_Cfg_t *hal = devHalCfg;
    struct spi_device *spi = hal->spi;
    const struct firmware *fw;
    char *line;
    int i = 0, ret, cnt = 0;
    static const int NUM_COLUMN = 3;

    /* setting mult to 0 will break caller loop in case of failure */
    *txAttenMult = 0;

    ret = request_firmware(&fw, txAttenTablePath, &spi->dev);
    if (ret) {
	dev_err(&spi->dev,
		"request_firmware(%s) failed with %d\n", txAttenTablePath, ret);
	return ADI_HAL_BBICCTRL_FAIL;
    }

    do {
	line = strnchr(fw->data + cnt, fw->size - cnt, '\n');
	if (!line)
		goto release_fw;
	line++;
	cnt = line - (char *)fw->data;
    } while (i++ != lineCount && cnt < fw->size);

    /* EOF */
    if (cnt >= fw->size) {
        ret = NUM_COLUMN;
        goto release_fw;
    }

    line = skip_spaces(line);

    ret = sscanf(line, "%hu,%hhu,%hu", attenIndex, txAttenHp, txAttenMult);

release_fw:
    release_firmware(fw);

    return ret;
}
#endif
