/*
LTC3350: High Current Supercapacitor Backup Controller and System Monitor

@verbatim
The  LTC®3350 is a backup power controller that can charge and monitor a series
stack  of  one  to  four  super-capacitors.  The LTC3350's synchronous step-down
controller  drives  N-channel  MOSFETs  for  constant  current/constant  voltage
charging  with  programmable  input  current  limit.  In addition, the step-down
converter  can  run  in reverse as a step-up converter to deliver power from the
supercapacitor stack to the backup supply rail. Internal balancers eliminate the
need for external balance resistors and each capacitor has a shunt regulator for
overvoltage  protection.  The  LTC3350 monitors system voltages, currents, stack
capacitance  and  stack  ESR which can all be read over the I²C/SMBus. The dual
ideal  diode controller uses N-channel MOSFETs for low loss power paths from the
input and super-capacitors to the backup system supply. The LTC3350 is available
in a low profile 38-lead 5mm × 7mm × 0.75mm QFN surface mount package.
@endverbatim

http://www.linear.com/product/LTC3350

http://www.linear.com/product/LTC3350#demoboards

REVISION HISTORY
$Revision: $
$Date: $

Copyright (c) 2016, Linear Technology Corp.(LTC)
All rights reserved.

Linear Technology Confidential - For Customer Use Only

THIS  SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY  EXPRESS  OR  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES   OF  MERCHANTABILITY  AND  FITNESS  FOR  A  PARTICULAR  PURPOSE  ARE
DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY  DIRECT,  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING,  BUT  NOT  LIMITED  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS  OF  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT  LIABILITY,  OR TORT
(INCLUDING  NEGLIGENCE  OR  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/



/*! @file
 *  @ingroup LTC3350
 *  @brief Linux I2C Example
 *
 *  This is an example program to demonstrate use of the API with the Linux
 *  kernel ic2-tools driver.
 *  http://www.lm-sensors.org/wiki/I2CTools
 */

#include "LTC3350.h"
#include <fcntl.h>
#include <linux/i2c-dev.h> //#Make sure to copy corresponding smbus.c from i2c-tools to source dir!
#include <errno.h>
#include <stdio.h>

/*!  read_register  function  wraps read_word_data and places the returned
data  in  *data. It returns 0 for success and a non-zero error code for failure.
The API functions will return this error code in the event of an error.*/
int read_register(uint8_t addr, //!< Target IC's SMBus address
                  uint8_t command_code, //!< Command Code to be read from
                  uint16_t *data, //!< Pointer to data destination
                  port_configuration_t *port_configuration //!< Pointer to port configuration struct
                 )
{
  if (ioctl(port_configuration->file_descriptor, I2C_SLAVE, addr) < 0)
  {
    printf("Error setting slave address: %s\n", strerror(errno));
    return 1;
  }
  //__s32 i2c_smbus_read_byte_data(int file, __u8 command);
  //__s32 i2c_smbus_read_word_data(int file, __u8 command);
  __s32 ret_val = i2c_smbus_read_word_data(port_configuration->file_descriptor, command_code);
  if (ret_val == -1)
  {
    printf("Read error: %s\n", strerror(errno));
    return 1;
  }
  *data = (uint16_t)ret_val;
  return 0;
}

/*!  write_register  function  wraps  write_word_data.  It  returns  0 for
success  and  a  non-zero  error code for failure. The API functions will return
this error code in the event of an error.*/
int write_register(uint8_t addr, //!< Target IC's SMBus address
                   uint8_t command_code, //!< Command Code to be written to
                   uint16_t data, //!< Data to be written
                   port_configuration_t *port_configuration //!< Pointer to port configuration struct
                  )
{
  if (ioctl(port_configuration->file_descriptor, I2C_SLAVE, addr) < 0)
  {
    printf("Error setting slave address: %s\n", strerror(errno));
    return 1;
  }
  //__s32 i2c_smbus_write_byte_data(int file, __u8 command, __u8 value);
  //__s32 i2c_smbus_write_word_data(int file, __u8 command, __u16 value);
  __s32 ret_val = i2c_smbus_write_word_data(port_configuration->file_descriptor, command_code, data);
  if (ret_val == -1)
  {
    printf("Write error: %s\n", strerror(errno));
    return 1;
  }
  return 0;
}

int main(void)
{
  uint16_t data;
  //choose the correct i2c bus.  You can find the memory mapping at /sys/bus/i2c/devices/i2c*
  int fd = open("/dev/i2c-1", O_RDWR);
  if (fd < 0)
  {
    printf("Error opening file: %s\n", strerror(errno));
    return 1;
  }
  if (ioctl(fd, I2C_PEC, 0) < 0) //set non-zero to enable PEC
  {
    printf("Error disabling Packet Error Checking: %s\n", strerror(errno));
    return 1;
  }
  //Leave slave address ioctl setup for read/write function.
  //Could place here if only ever talking to one address.
  port_configuration_t pc =
  {
    .file_descriptor = fd
  };
  LTC3350_chip_cfg_t cfg =
  {
    .addr = LTC3350_ADDR_09,
    .read_register = read_register,
    .write_register = write_register,
    .port_configuration = &pc
  };

  printf("Initializing LTC3350\n");
  LTC3350 chip = LTC3350_init(&cfg);

  /* the API functions can be used to read and write individual bit fields
  within a command code */
  printf("Using Read Register for the LTC3350_MSK_GPI_UV bit field\n");
  LTC3350_read_register(chip, LTC3350_MSK_GPI_UV, &data);
  printf("Read: %d\n",data);
  printf("Using Read/Modify/Write Register for the LTC3350_MSK_GPI_UV bit field\n");
  LTC3350_write_register(chip, LTC3350_MSK_GPI_UV, 1);
  printf("Wrote: %d\n",1);
  printf("Using Read Register for the LTC3350_MSK_GPI_UV bit field\n");
  LTC3350_read_register(chip, LTC3350_MSK_GPI_UV, &data);
  printf("Read: %d\n",data);

  /* the API functions can also be used to read and write whole command codes */
  printf("Using Read Register for the LTC3350_VOUT_OV_LVL_CMD command code\n");
  LTC3350_read_register(chip, LTC3350_VOUT_OV_LVL_CMD, &data);
  printf("Read: %d\n",data);
  printf("Using Read/Modify/Write Register for the LTC3350_VOUT_OV_LVL_CMD command code\n");
  LTC3350_write_register(chip, LTC3350_VOUT_OV_LVL_CMD, 35787);
  printf("Wrote: %d\n",35787);
  printf("Using Read Register for the LTC3350_VOUT_OV_LVL_CMD command code\n");
  LTC3350_read_register(chip, LTC3350_VOUT_OV_LVL_CMD, &data);
  printf("Read: %d\n",data);

  return 0;
}
