/*
LTC3350: High Current Supercapacitor Backup Controller and System Monitor

@verbatim
The  LTC®3350 is a backup power controller that can charge and monitor a series
stack  of  one  to  four  super-capacitors.  The LTC3350's synchronous step-down
controller  drives  N-channel  MOSFETs  for  constant  current/constant  voltage
charging  with  programmable  input  current  limit.  In addition, the step-down
converter  can  run  in reverse as a step-up converter to deliver power from the
supercapacitor stack to the backup supply rail. Internal balancers eliminate the
need for external balance resistors and each capacitor has a shunt regulator for
overvoltage  protection.  The  LTC3350 monitors system voltages, currents, stack
capacitance  and  stack  ESR which can all be read over the I²C/SMBus. The dual
ideal  diode controller uses N-channel MOSFETs for low loss power paths from the
input and super-capacitors to the backup system supply. The LTC3350 is available
in a low profile 38-lead 5mm × 7mm × 0.75mm QFN surface mount package.
@endverbatim

http://www.linear.com/product/LTC3350

http://www.linear.com/product/LTC3350#demoboards

REVISION HISTORY
$Revision: $
$Date: $

Copyright (c) 2016, Linear Technology Corp.(LTC)
All rights reserved.

Linear Technology Confidential - For Customer Use Only

THIS  SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY  EXPRESS  OR  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES   OF  MERCHANTABILITY  AND  FITNESS  FOR  A  PARTICULAR  PURPOSE  ARE
DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY  DIRECT,  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING,  BUT  NOT  LIMITED  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS  OF  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT  LIABILITY,  OR TORT
(INCLUDING  NEGLIGENCE  OR  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/*! @file
 *  @ingroup LTC3350
 *  @brief LTC3350 communication library example
 *
 *  This is an example program to demonstrate use of the API. The
 *  my_write_register and my_read_register functions simply display data and
 *  return constant data to allow compiling and demonstration without hardware.
 */

#include "LTC3350.h"
#include <stdio.h>
#include <stdlib.h>

/*!
This  is  an  example  of an smbus_read_register function. In a real application
this  function  would  do  an SMBus read_word transaction of slave address
addr,  and  command  code command_code, and place the returned data in *data. It
should  return  a  0  for success and a non-zero error code for failure. The API
functions  will  return  this error code in the event of an error. If you do not
need error handling, it should simply always return 0.
*/
int my_read_register(uint8_t addr, //!< Target IC's SMBus address
                     uint8_t command_code, //!< Command Code to be read from
                     uint16_t *data, //!< Pointer to data destination
                     port_configuration_t *port_configuration //!< Pointer to port configuration struct
                    )
{
  printf("Read word Transaction:\n");
  printf("\tAddr: %d\n",addr);
  printf("\tCommand_code: %d\n",command_code);
  *data = rand() % (1<<16);
  printf("\tRaw read fake data: %d\n",*data);
  return 0;
}

/*!
This  is  an  example of an smbus_write_register function. In a real application
this  function  would  do an SMBus write_word transaction of slave address
addr,  and  command  code  command_code,  writing data. It should return a 0 for
success  and  a  non-zero  error code for failure. The API functions will return
this  error code in the event of an error. If you do not need error handling, it
should simply always return 0.
*/
int my_write_register(uint8_t addr, //!< Target IC's SMBus address
                      uint8_t command_code, //!< Command Code to be written to
                      uint16_t data, //!< Data to be written
                      port_configuration_t *port_configuration //!< Pointer to port configuration struct
                     )
{
  printf("Write word Transaction:\n");
  printf("\tAddr: %d\n",addr);
  printf("\tCommandCode: %d\n",command_code);
  printf("\tRaw Data: %d\n",data);
  return 0;
}

int main(void)
{
  uint16_t data;
  LTC3350_chip_cfg_t cfg =
  {
    .addr = LTC3350_ADDR_09,
    .read_register = my_read_register,
    .write_register = my_write_register,
    //!< Pointer to port configuration struct not used without physical port
    //.port_configuration = &pc
  };

  printf("Initializing LTC3350\n");
  LTC3350 chip = LTC3350_init(&cfg);

  /*! the API functions can be used to read and write individual bit fields
  within a command code */
  printf("Using Read/Modify/Write Register for the LTC3350_MSK_GPI_UV bit field\n");
  LTC3350_write_register(chip, LTC3350_MSK_GPI_UV, 0x1);
  printf("Using Read Register for the LTC3350_MSK_GPI_UV bit field\n");
  LTC3350_read_register(chip, LTC3350_MSK_GPI_UV, &data);
  printf("\tRead: %d\n",data);

  /*! the API functions can also be used to read and write whole command codes */
  printf("Using Read/Modify/Write Register for the LTC3350_VOUT_OV_LVL_CMD command code\n");
  LTC3350_write_register(chip, LTC3350_VOUT_OV_LVL_CMD, 0x1);
  printf("Using Read Register for the LTC3350_VOUT_OV_LVL_CMD command code\n");
  LTC3350_read_register(chip, LTC3350_VOUT_OV_LVL_CMD, &data);
  printf("\tRead: %d\n",data);


  return 0;
}
