#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import print_function
from scipy.interpolate import UnivariateSpline


''' Copyright (c) 2014, Linear Technology Corp.(LTC)
All rights reserved.

Linear Technology Confidential - For Customer Use Only

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. '''


#Generated on 2016-01-08

class LTC3350(object):
    '''API for the LTC3350 High Current Supercapacitor Backup Controller and System Monitor.

    Each bit field is read and written as an attribute of the class instance.
    Bit fields are changed in place with a read-modify-write algorithm to avoid clearing adjacent data in shared registers.
    Presets (enumerations) and formats (transformation functions to real-word units) are applied in both directions for interactive use, but can be disabled.'''
    def __init__(self, read_function, write_function=None, verbose=False):
        '''The user must supply appropriate functions to read from and write to the I2C/SMBus hardware.
        read_function should take arguments (addr_7bit, command_code) and return contents of LTC3350 register at command_code.
        write_function should take arguments (addr_7bit, command_code, data) and write data to LTC3350 register at command_code.
        Set verbose argument to True to enable printing of intermediate results of masking, shifting and data transformation operations.
        '''
        object.__setattr__(self, 'verbose', verbose)
        object.__setattr__(self, 'read_function', read_function)
        object.__setattr__(self, 'write_function', write_function)
        object.__setattr__(self, 'addr_7bit', 0x09)
        object.__setattr__(self, 'word_size', 16)
        self._update_xml_formatters()

    def get_active_format(self, bf_name):
        '''Returns string name of format currently active for field bf_name.'''
        return self._register_map[bf_name]['active_format']
    def set_active_format(self, bf_name, format_name, force=False):
        '''Changes currently active format for field bf_name to format_name.
        If format_name is not in list of allowed_formats, set force argument to True to disable check.
        '''
        if format_name is None or format_name.lower() == 'none':
            self._register_map[bf_name]['active_format'] = 'None'
        elif format_name in self._register_map[bf_name]['allowed_formats'] or force:
            self._register_map[bf_name]['active_format'] = format_name
        else:
            self._error('Format {} not allowed for field {}'.format(format_name,bf_name))
    def create_format(self, format_name, format_function, unformat_function, signed=False, description=None):
        '''Create a new format definition or modify an existing definition.

        format_function should take a single argument of integer raw data from the register and return a version of the data scaled to appropriate units.
        unformat_function should take a single argument of data in real units and return an integer version of the data scaled to the register LSB weight.
        If the data is signed in two's-complement format, set signed=True.
        After creating format, use set_active_format method to make the new format active.
        '''
        self._formatters[format_name] = {'format': format_function, 'unformat': unformat_function, 'description': description, 'signed': signed}
    def set_constant(self, constant, value):
        '''Sets the constants found in the datasheet used by the formatters to convert from real world values to digital value and back.'''
        self._constants[constant] = value
        self._update_xml_formatters()
    def get_constant(self,constant):
        '''Gets the constants found in the datasheet used by the formatters to convert from real world values to digital value and back.'''
        return self._constants[constant]
    def list_constants(self):
        '''Returns a dictionary of constants found in the datasheet used by the formatters to convert from real world values to digital value and back.'''
        return self._constants
    def disable_presets_and_formats(self):
        '''Remove all presets and formats.

        Read and write data will be passed through unmodified, except for masking and shifting.
        The dictionary of presets and formats is shared between all instances of the class and any other instances will reflect this change.
        This is permanent for the duration of the Python session.'''
        for register in self._register_map.values():
            register['presets'] = []
            register['active_format'] = 'None'
    def _error(self, message):
        '''Raises exceptions if bad data is passed to API.
        Bus Read and Write errors must be raised by the user-supplied functions.'''
        raise LTC3350_APIException(message)
    def __str__(self):
        return 'LTC3350 API instance at address 0x09'
    def __repr__(self):
        return self.__str__()
    def __setattr__(self, name, value):
        '''Override attribute storage mechanism to prevent writes to mis-spelled bit fields.'''
        try:
            #First try writes to bit field property decorator
            LTC3350.__dict__[name].fset(self, value)
        except KeyError as e:
            if getattr(self, name, None) is not None:
                #Then try writing to existing conventional instance attributes
                object.__setattr__(self, name, value)
            else:
                #Finally, prevent creation of new instance attributes
                self._error('Attempted write to non-existant field: {}.'.format(name))
    def _read_bf(self, bf_name):
        return self._format(bf_name,self._extract(self.read_function(self.addr_7bit, self._register_map[bf_name]['command_code']),self._register_map[bf_name]['size'],self._register_map[bf_name]['offset']))
    def _write_bf(self, bf_name, data):
        if self._register_map[bf_name]['size'] == self.word_size: #write-in-place
            data = self._unformat(bf_name,data)
            if data >= (1<<self.word_size) or data < 0:
                self._error('Data:{} does not fit in field:{} of size:{}.'.format(hex(data),bf_name,self.word_size))
            self.write_function(self.addr_7bit, self._register_map[bf_name]['command_code'],data)
        else: #read-modify-write
            self.write_function(self.addr_7bit, self._register_map[bf_name]['command_code'],self._pack(self._unformat(bf_name,data),self._register_map[bf_name]['size'],self._register_map[bf_name]['offset'],self.read_function(self.addr_7bit, self._register_map[bf_name]['command_code'])))
    def _extract(self, data, size, offset):
        result = (data >> offset) & ((1<<size)-1)
        if self.verbose:
            print('Extracting data:{}, size:{}, offset:{} from raw data:{:#018b}.'.format(bin(result),size,offset,data))
        return result
    def _pack(self, data, size, offset, old_register_contents=0):
        if self.verbose:
            print('Packing new data:{}, size:{}, offset:{} into {:#018b}'.format(bin(data),size,offset,old_register_contents))
        mask = ((1<<self.word_size)-1) ^ (((1<<size)-1)<<offset)
        masked = old_register_contents & mask
        if data >= (1<<size) or data < 0:
            self._error('Data:{} does not fit in field of size:{}.'.format(hex(data),size))
        data = (data<<offset) | masked
        if self.verbose:
            print('mask:{:#018b}, masked_old:{:#018b}, merged:{:#018b}'.format(mask,masked,data))
        return data
    def _transform_from_points(self, xlist, ylist, direction):
        '''Used internally to convert from register values to real world values and back again.'''
        x_evaled = []
        y_evaled = []
        only_constants = {}
        only_constants.update(self._constants)
        only_constants = {key:float(value) for key, value in self._constants.iteritems()}
        for xpoint in xlist:
            x_evaled.append(eval(xpoint, only_constants))
        for ypoint in ylist:
            y_evaled.append(eval(ypoint, only_constants))
        if direction == "format":
            z = sorted(zip(x_evaled, y_evaled), key = lambda x: x[0])
            return lambda x: float(UnivariateSpline(x = zip(*z)[0], y = zip(*z)[1], k=1, s = 0)(x))
        elif direction == "unformat":
            z = sorted(zip(x_evaled, y_evaled), key = lambda x: x[1])
            return lambda x: int(round(UnivariateSpline(x = zip(*z)[1], y = zip(*z)[0], k=1, s = 0)(x)))
        else:
            print("'transform_from_points()' requires one of either: 'format' or 'unformat'")
            return
    def _format(self, bf_name, data):
        for (preset,value) in self._register_map[bf_name]['presets']:
                if (data == value):
                    if self.verbose:
                        print('Read matched preset {} with value {}'.format(preset,value))
                    return preset
        if self.verbose:
            print('Applying format: {}'.format(self._register_map[bf_name]['active_format']))
        if self._formatters[self._register_map[bf_name]['active_format']]['signed']:
            data = self._twosComplementToSigned(data, self._register_map[bf_name]['size'])
        return self._formatters[self._register_map[bf_name]['active_format']]['format'](data)
    def _unformat(self, bf_name, data):
        for (preset,value) in self._register_map[bf_name]['presets']:
                if (data == preset):
                    if self.verbose:
                        print('Write matched preset {} with value {}'.format(preset,value))
                    return value
        if self.verbose:
            print('Un-applying format: {}'.format(self._register_map[bf_name]['active_format']))
        data = self._formatters[self._register_map[bf_name]['active_format']]['unformat'](data)
        if self._formatters[self._register_map[bf_name]['active_format']]['signed']:
            data = self._signedToTwosComplement(data, self._register_map[bf_name]['size'])
        else:
            if data < 0:
                print("WARNING: unformatted data {} clamped to fit bitfield".format(data))
                data = 0
            elif data > 2**self._register_map[bf_name]['size'] - 1:
                print("WARNING: unformatted data {} clamped to fit bitfield".format(data))
                data = 2**self._register_map[bf_name]['size'] - 1
        return data
    def _signedToTwosComplement(self, signed, bitCount):
        '''take python int and convert to two's complement representation using specified number of bits'''
        if signed > 2**(bitCount-1) - 1:
            print("WARNING: unformatted data {} clamped to fit signed bitfield".format(signed))
            signed = 2**(bitCount-1) - 1
        elif signed < -2**(bitCount-1):
            print("WARNING: unformatted data {} clamped to fit signed bitfield".format(signed))
            signed = -2**(bitCount-1)
        if signed < 0:
            if self.verbose:
                print("Converting negative number:{} of size:{} to two's complement.".format(signed, bitCount))
            signed += 2**bitCount
            signed &= 2**bitCount-1
        return signed
    def _twosComplementToSigned(self, binary, bitCount):
        '''take two's complement number with specified number of bits and convert to python int representation'''
        assert binary <= 2**bitCount - 1
        assert binary >= 0
        if binary >= 2**(bitCount-1):
            if self.verbose:
                print('Converting binary number:{} of size:{} to negative int.'.format(binary, bitCount))
            binary -= 2**bitCount
        return binary
    def _update_xml_formatters(self):
        self.create_format( format_name = 'None',
                            format_function = lambda x:x,
                            unformat_function = lambda x:x,
                            signed = False,
                            description = '''No formatting applied to data.''')  
        for fmt_name in self._xml_formats:
            xlist, ylist = zip(*self._xml_formats[fmt_name]["points"])
            self.create_format( format_name = fmt_name,
                                format_function = self._transform_from_points(xlist, ylist, "format"),
                                unformat_function = self._transform_from_points(xlist, ylist, "unformat"),
                                signed = self._xml_formats[fmt_name]["signed"],
                                description = self._xml_formats[fmt_name]["description"])

    clr_cap_lo = property(fget=lambda self: self._read_bf('clr_cap_lo'),
                                         fset=lambda self,data: self._write_bf('clr_cap_lo',data),
                                         doc='''Clear capacitance low alarm (1 bit, R/W)''')
    clr_esr_hi = property(fget=lambda self: self._read_bf('clr_esr_hi'),
                                         fset=lambda self,data: self._write_bf('clr_esr_hi',data),
                                         doc='''Clear ESR high alarm (1 bit, R/W)''')
    clr_dtemp_hot = property(fget=lambda self: self._read_bf('clr_dtemp_hot'),
                                         fset=lambda self,data: self._write_bf('clr_dtemp_hot',data),
                                         doc='''Clear die temperature hot alarm (1 bit, R/W)''')
    clr_dtemp_cold = property(fget=lambda self: self._read_bf('clr_dtemp_cold'),
                                         fset=lambda self,data: self._write_bf('clr_dtemp_cold',data),
                                         doc='''Clear die temperature cold alarm (1 bit, R/W)''')
    clr_ichg_uc = property(fget=lambda self: self._read_bf('clr_ichg_uc'),
                                         fset=lambda self,data: self._write_bf('clr_ichg_uc',data),
                                         doc='''Clear charge undercurrent alarm (1 bit, R/W)''')
    clr_iin_oc = property(fget=lambda self: self._read_bf('clr_iin_oc'),
                                         fset=lambda self,data: self._write_bf('clr_iin_oc',data),
                                         doc='''Clear input overcurrent alarm (1 bit, R/W)''')
    clr_vout_ov = property(fget=lambda self: self._read_bf('clr_vout_ov'),
                                         fset=lambda self,data: self._write_bf('clr_vout_ov',data),
                                         doc='''Clear VOUT overvoltage alarm (1 bit, R/W)''')
    clr_vout_uv = property(fget=lambda self: self._read_bf('clr_vout_uv'),
                                         fset=lambda self,data: self._write_bf('clr_vout_uv',data),
                                         doc='''Clear VOUT undervoltage alarm (1 bit, R/W)''')
    clr_vcap_ov = property(fget=lambda self: self._read_bf('clr_vcap_ov'),
                                         fset=lambda self,data: self._write_bf('clr_vcap_ov',data),
                                         doc='''Clear VCAP overvoltage alarm (1 bit, R/W)''')
    clr_vcap_uv = property(fget=lambda self: self._read_bf('clr_vcap_uv'),
                                         fset=lambda self,data: self._write_bf('clr_vcap_uv',data),
                                         doc='''Clear VCAP undervoltage alarm (1 bit, R/W)''')
    clr_vin_ov = property(fget=lambda self: self._read_bf('clr_vin_ov'),
                                         fset=lambda self,data: self._write_bf('clr_vin_ov',data),
                                         doc='''Clear VIN overvoltage alarm (1 bit, R/W)''')
    clr_vin_uv = property(fget=lambda self: self._read_bf('clr_vin_uv'),
                                         fset=lambda self,data: self._write_bf('clr_vin_uv',data),
                                         doc='''Clear VIN undervoltage alarm (1 bit, R/W)''')
    clr_gpi_ov = property(fget=lambda self: self._read_bf('clr_gpi_ov'),
                                         fset=lambda self,data: self._write_bf('clr_gpi_ov',data),
                                         doc='''Clear GPI overvoltage alarm (1 bit, R/W)''')
    clr_gpi_uv = property(fget=lambda self: self._read_bf('clr_gpi_uv'),
                                         fset=lambda self,data: self._write_bf('clr_gpi_uv',data),
                                         doc='''Clear GPI undervoltage alarm (1 bit, R/W)''')
    clr_cap_ov = property(fget=lambda self: self._read_bf('clr_cap_ov'),
                                         fset=lambda self,data: self._write_bf('clr_cap_ov',data),
                                         doc='''Clear capacitor overvoltage alarm (1 bit, R/W)''')
    clr_cap_uv = property(fget=lambda self: self._read_bf('clr_cap_uv'),
                                         fset=lambda self,data: self._write_bf('clr_cap_uv',data),
                                         doc='''Clear capacitor undervoltage alarm (1 bit, R/W)''')
    clr_alarms = property(fget=lambda self: self._read_bf('clr_alarms'),
                                         fset=lambda self,data: self._write_bf('clr_alarms',data),
                                         doc='''Clear Alarms Register: This register is used to clear alarms caused by exceeding a programmed limit. Writing a one to any bit in this register will cause its respective alarm to be cleared. The one written to this register is automatically cleared when its respective alarm is cleared.''')
    msk_cap_uv = property(fget=lambda self: self._read_bf('msk_cap_uv'),
                                         fset=lambda self,data: self._write_bf('msk_cap_uv',data),
                                         doc='''Enable capacitor undervoltage alarm (1 bit, R/W)''')
    msk_cap_ov = property(fget=lambda self: self._read_bf('msk_cap_ov'),
                                         fset=lambda self,data: self._write_bf('msk_cap_ov',data),
                                         doc='''Enable capacitor over voltage alarm (1 bit, R/W)''')
    msk_gpi_uv = property(fget=lambda self: self._read_bf('msk_gpi_uv'),
                                         fset=lambda self,data: self._write_bf('msk_gpi_uv',data),
                                         doc='''Enable GPI undervoltage alarm (1 bit, R/W)''')
    msk_gpi_ov = property(fget=lambda self: self._read_bf('msk_gpi_ov'),
                                         fset=lambda self,data: self._write_bf('msk_gpi_ov',data),
                                         doc='''Enable GPI overvoltage alarm (1 bit, R/W)''')
    msk_vin_uv = property(fget=lambda self: self._read_bf('msk_vin_uv'),
                                         fset=lambda self,data: self._write_bf('msk_vin_uv',data),
                                         doc='''Enable VIN undervoltage alarm (1 bit, R/W)''')
    msk_vin_ov = property(fget=lambda self: self._read_bf('msk_vin_ov'),
                                         fset=lambda self,data: self._write_bf('msk_vin_ov',data),
                                         doc='''Enable VIN overvoltage alarm (1 bit, R/W)''')
    msk_vcap_uv = property(fget=lambda self: self._read_bf('msk_vcap_uv'),
                                         fset=lambda self,data: self._write_bf('msk_vcap_uv',data),
                                         doc='''Enable VCAP undervoltage alarm (1 bit, R/W)''')
    msk_vcap_ov = property(fget=lambda self: self._read_bf('msk_vcap_ov'),
                                         fset=lambda self,data: self._write_bf('msk_vcap_ov',data),
                                         doc='''Enable VCAP overvoltage alarm (1 bit, R/W)''')
    msk_vout_uv = property(fget=lambda self: self._read_bf('msk_vout_uv'),
                                         fset=lambda self,data: self._write_bf('msk_vout_uv',data),
                                         doc='''Enable VOUT undervoltage alarm (1 bit, R/W)''')
    msk_vout_ov = property(fget=lambda self: self._read_bf('msk_vout_ov'),
                                         fset=lambda self,data: self._write_bf('msk_vout_ov',data),
                                         doc='''Enable VOUT overvoltage alarm (1 bit, R/W)''')
    msk_iin_oc = property(fget=lambda self: self._read_bf('msk_iin_oc'),
                                         fset=lambda self,data: self._write_bf('msk_iin_oc',data),
                                         doc='''Enable input overcurrent alarm (1 bit, R/W)''')
    msk_ichg_uc = property(fget=lambda self: self._read_bf('msk_ichg_uc'),
                                         fset=lambda self,data: self._write_bf('msk_ichg_uc',data),
                                         doc='''Enable charge undercurrent alarm (1 bit, R/W)''')
    msk_dtemp_cold = property(fget=lambda self: self._read_bf('msk_dtemp_cold'),
                                         fset=lambda self,data: self._write_bf('msk_dtemp_cold',data),
                                         doc='''Enable die temperature cold alarm (1 bit, R/W)''')
    msk_dtemp_hot = property(fget=lambda self: self._read_bf('msk_dtemp_hot'),
                                         fset=lambda self,data: self._write_bf('msk_dtemp_hot',data),
                                         doc='''Enable die temperature hot alarm (1 bit, R/W)''')
    msk_esr_hi = property(fget=lambda self: self._read_bf('msk_esr_hi'),
                                         fset=lambda self,data: self._write_bf('msk_esr_hi',data),
                                         doc='''Enable ESR high alarm (1 bit, R/W)''')
    msk_cap_lo = property(fget=lambda self: self._read_bf('msk_cap_lo'),
                                         fset=lambda self,data: self._write_bf('msk_cap_lo',data),
                                         doc='''Enable capacitance low alarm (1 bit, R/W)''')
    msk_alarms = property(fget=lambda self: self._read_bf('msk_alarms'),
                                         fset=lambda self,data: self._write_bf('msk_alarms',data),
                                         doc='''Mask Alarms Register: Writing a one to any bit in the Mask Alarms Register enables its respective alarm to trigger an SMBALERT.''')
    msk_mon_capesr_active = property(fget=lambda self: self._read_bf('msk_mon_capesr_active'),
                                         fset=lambda self,data: self._write_bf('msk_mon_capesr_active',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_capesr_active (1 bit, R/W)''')
    msk_mon_capesr_scheduled = property(fget=lambda self: self._read_bf('msk_mon_capesr_scheduled'),
                                         fset=lambda self,data: self._write_bf('msk_mon_capesr_scheduled',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_capesr_scheduled (1 bit, R/W)''')
    msk_mon_capesr_pending = property(fget=lambda self: self._read_bf('msk_mon_capesr_pending'),
                                         fset=lambda self,data: self._write_bf('msk_mon_capesr_pending',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_capesr_pending (1 bit, R/W)''')
    msk_mon_cap_done = property(fget=lambda self: self._read_bf('msk_mon_cap_done'),
                                         fset=lambda self,data: self._write_bf('msk_mon_cap_done',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_cap_done (1 bit, R/W)''')
    msk_mon_esr_done = property(fget=lambda self: self._read_bf('msk_mon_esr_done'),
                                         fset=lambda self,data: self._write_bf('msk_mon_esr_done',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_esr_done (1 bit, R/W)''')
    msk_mon_cap_failed = property(fget=lambda self: self._read_bf('msk_mon_cap_failed'),
                                         fset=lambda self,data: self._write_bf('msk_mon_cap_failed',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_cap_failed (1 bit, R/W)''')
    msk_mon_esr_failed = property(fget=lambda self: self._read_bf('msk_mon_esr_failed'),
                                         fset=lambda self,data: self._write_bf('msk_mon_esr_failed',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_esr_failed (1 bit, R/W)''')
    msk_mon_power_failed = property(fget=lambda self: self._read_bf('msk_mon_power_failed'),
                                         fset=lambda self,data: self._write_bf('msk_mon_power_failed',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_power_failed (1 bit, R/W)''')
    msk_mon_power_returned = property(fget=lambda self: self._read_bf('msk_mon_power_returned'),
                                         fset=lambda self,data: self._write_bf('msk_mon_power_returned',data),
                                         doc='''Set the SMBALERT when there is a rising edge on mon_power_returned (1 bit, R/W)''')
    msk_mon_status = property(fget=lambda self: self._read_bf('msk_mon_status'),
                                         fset=lambda self,data: self._write_bf('msk_mon_status',data),
                                         doc='''Mask Monitor Status Register: Writing a one to any bit in this register enables a rising edge of its respective bit in the mon_status register to trigger an SMBALERT.''')
    cap_esr_per = property(fget=lambda self: self._read_bf('cap_esr_per'),
                                         fset=lambda self,data: self._write_bf('cap_esr_per',data),
                                         doc='''Capacitance and ESR Measurement Period: This register sets the period of repeated capacitance and ESR measurements. Each LSB represents 10 seconds. Capacitance and ESR measurements will not repeat if this register is zero. (16 bits, R/W)
                                                 Format: cap_per''')
    vcapfb_dac = property(fget=lambda self: self._read_bf('vcapfb_dac'),
                                         fset=lambda self,data: self._write_bf('vcapfb_dac',data),
                                         doc='''VCAP Regulation Reference: This register is used to program the capacitor voltage feedback loop's reference voltage. Only bits 3:0 are active. CAPFBREF = 37.5mV * vcapfb_dac + 637.5mV (4 bits, R/W)
                                                 Format: vcapfb_dac_format''')
    vcapfb_dac_CMD = property(fget=lambda self: self._read_bf('vcapfb_dac_CMD'),
                                         fset=lambda self,data: self._write_bf('vcapfb_dac_CMD',data),
                                         doc='''''')
    vshunt = property(fget=lambda self: self._read_bf('vshunt'),
                                         fset=lambda self,data: self._write_bf('vshunt',data),
                                         doc='''Shunt Voltage Register: This register programs the shunt voltage for each capacitor in the stack. The charger will limit current and the active shunts will shunt current to prevent this voltage from being exceeded. As a capacitor voltage nears this level, the charge current will be reduced. This should be programmed higher than the intended final balanced individual capacitor voltage. Setting this register to 0x0000 disables the shunt. 183.5uV per LSB. (16 bits, R/W)
                                                 Format: cell''')
    cap_uv_lvl = property(fget=lambda self: self._read_bf('cap_uv_lvl'),
                                         fset=lambda self,data: self._write_bf('cap_uv_lvl',data),
                                         doc='''Capacitor Undervoltage Level: This is an alarm threshold for each individual capacitor voltage in the stack. If enabled, any capacitor voltage falling below this level will trigger an alarm and an SMBALERT. 183.5uV per LSB. (16 bits, R/W)
                                                 Format: cell''')
    cap_ov_lvl = property(fget=lambda self: self._read_bf('cap_ov_lvl'),
                                         fset=lambda self,data: self._write_bf('cap_ov_lvl',data),
                                         doc='''Capacitor Overvoltage Level: This is an alarm threshold for each individual capacitor in the stack. If enabled, any capacitor voltage rising above this level will trigger an alarm and an SMBALERT. 183.5uV per LSB (16 bits, R/W)
                                                 Format: cell''')
    gpi_uv_lvl = property(fget=lambda self: self._read_bf('gpi_uv_lvl'),
                                         fset=lambda self,data: self._write_bf('gpi_uv_lvl',data),
                                         doc='''General Purpose Input Undervoltage Level: This is an alarm threshold for the GPI pin. If enabled, the voltage falling below this level will trigger an alarm and an SMBALERT. 183.5uV per LSB (16 bits, R/W)
                                                 Format: cell''')
    gpi_ov_lvl = property(fget=lambda self: self._read_bf('gpi_ov_lvl'),
                                         fset=lambda self,data: self._write_bf('gpi_ov_lvl',data),
                                         doc='''General Purpose Input Overvoltage Level: This is an alarm threshold for the GPI pin. If enabled, the voltage rising above this level will trigger an alarm and an SMBALERT. 183.5uV per LSB (16 bits, R/W)
                                                 Format: cell''')
    vin_uv_lvl = property(fget=lambda self: self._read_bf('vin_uv_lvl'),
                                         fset=lambda self,data: self._write_bf('vin_uv_lvl',data),
                                         doc='''VIN Undervoltage Level: This is an alarm threshold for the input voltage. If enabled, the voltage falling below this level will trigger an alarm and an SMBALERT. 2.21mV per LSB (16 bits, R/W)
                                                 Format: vin''')
    vin_ov_lvl = property(fget=lambda self: self._read_bf('vin_ov_lvl'),
                                         fset=lambda self,data: self._write_bf('vin_ov_lvl',data),
                                         doc='''VIN Overvoltage Level: This is an alarm threshold for the input voltage. If enabled, the voltage rising above this level will trigger an alarm and an SMBALERT. 2.21mV per LSB (16 bits, R/W)
                                                 Format: vin''')
    vcap_uv_lvl = property(fget=lambda self: self._read_bf('vcap_uv_lvl'),
                                         fset=lambda self,data: self._write_bf('vcap_uv_lvl',data),
                                         doc='''VCAP Undervoltage Level: This is an alarm threshold for the capacitor stack voltage. If enabled, the voltage falling below this level will trigger an alarm and an SMBALERT. 1.476mV per LSB (16 bits, R/W)
                                                 Format: vcap''')
    vcap_ov_lvl = property(fget=lambda self: self._read_bf('vcap_ov_lvl'),
                                         fset=lambda self,data: self._write_bf('vcap_ov_lvl',data),
                                         doc='''VCAP Overvoltage Level: This is an alarm threshold for the capacitor stack voltage. If enabled, the voltage rising above this level will trigger an alarm and an SMBALERT. 1.476mV per LSB (16 bits, R/W)
                                                 Format: vcap''')
    vout_uv_lvl = property(fget=lambda self: self._read_bf('vout_uv_lvl'),
                                         fset=lambda self,data: self._write_bf('vout_uv_lvl',data),
                                         doc='''VOUT Undervoltage Level: This is an alarm threshold for the output voltage. If enabled, the voltage falling below this level will trigger an alarm and an SMBALERT. 2.21mV per LSB (16 bits, R/W)
                                                 Format: vout''')
    vout_ov_lvl = property(fget=lambda self: self._read_bf('vout_ov_lvl'),
                                         fset=lambda self,data: self._write_bf('vout_ov_lvl',data),
                                         doc='''VOUT Overvoltage Level: This is an alarm threshold for the output voltage. If enabled, the voltage rising above this level will trigger an alarm and an SMBALERT. 2.21mV per LSB (16 bits, R/W)
                                                 Format: vout''')
    iin_oc_lvl = property(fget=lambda self: self._read_bf('iin_oc_lvl'),
                                         fset=lambda self,data: self._write_bf('iin_oc_lvl',data),
                                         doc='''Input Overcurrent Level: This is an alarm threshold for the input current. If enabled, the current rising above this level will trigger an alarm and an SMBALERT. 1.983uV/RSNSI per LSB (16 bits, R/W)
                                                 Format: iin''')
    ichg_uc_lvl = property(fget=lambda self: self._read_bf('ichg_uc_lvl'),
                                         fset=lambda self,data: self._write_bf('ichg_uc_lvl',data),
                                         doc='''Charge Undercurrent Level: This is an alarm threshold for the charge current. If enabled, the current falling below this level will trigger an alarm and an SMBALERT. 1.983uV/RSNSC per LSB (16 bits, R/W)
                                                 Format: icharge''')
    dtemp_cold_lvl = property(fget=lambda self: self._read_bf('dtemp_cold_lvl'),
                                         fset=lambda self,data: self._write_bf('dtemp_cold_lvl',data),
                                         doc='''Die Temperature Cold Level: This is an alarm threshold for the die temperature. If enabled, the die temperature falling below this level will trigger an alarm and an SMBALERT. Temperature = 0.028C per LSB - 251.4C (16 bits, R/W)
                                                 Format: dtemp''')
    dtemp_hot_lvl = property(fget=lambda self: self._read_bf('dtemp_hot_lvl'),
                                         fset=lambda self,data: self._write_bf('dtemp_hot_lvl',data),
                                         doc='''Die Temperature Hot Level: This is an alarm threshold for the die temperature. If enabled, the die temperature rising above this level will trigger an alarm and an SMBALERT. Temperature = 0.028C per LSB - 251.4C (16 bits, R/W)
                                                 Format: dtemp''')
    esr_hi_lvl = property(fget=lambda self: self._read_bf('esr_hi_lvl'),
                                         fset=lambda self,data: self._write_bf('esr_hi_lvl',data),
                                         doc='''ESR High Level: This is an alarm threshold for the measured stack ESR. If enabled, a measurement of stack ESR exceeding this level will trigger an alarm and an SMBALERT. RSNSC/64 per LSB. (16 bits, R/W)
                                                 Format: esr''')
    cap_lo_lvl = property(fget=lambda self: self._read_bf('cap_lo_lvl'),
                                         fset=lambda self,data: self._write_bf('cap_lo_lvl',data),
                                         doc='''Capacitance Low Level: This is an alarm threshold for the measured stack capacitance. If enabled, if the measured stack capacitance is less than this level it will trigger an alarm and an SMBALERT. When ctl_cap_scale is set to 1, capacitance is 3.36uF * RT/RTST per LSB. When ctl_cap_scale is set to 0 it is 336uF * RT/RTST per LSB. (16 bits, R/W)
                                                 Format: cap''')
    ctl_strt_capesr = property(fget=lambda self: self._read_bf('ctl_strt_capesr'),
                                         fset=lambda self,data: self._write_bf('ctl_strt_capesr',data),
                                         doc='''Begin a capacitance and ESR measurement when possible; this bit clears itself once a cycle begins. (1 bit, R/W)
                                                 Preset 'start': 1''')
    ctl_gpi_buffer_en = property(fget=lambda self: self._read_bf('ctl_gpi_buffer_en'),
                                         fset=lambda self,data: self._write_bf('ctl_gpi_buffer_en',data),
                                         doc='''A one in this bit location enables the input buffer on the GPI pin. With a zero in this location the GPI pin is measured without the buffer. (1 bit, R/W)''')
    ctl_stop_capesr = property(fget=lambda self: self._read_bf('ctl_stop_capesr'),
                                         fset=lambda self,data: self._write_bf('ctl_stop_capesr',data),
                                         doc='''Stops an active capacitance/ESR measurement. (1 bit, R/W)
                                                 Preset 'stop': 1''')
    ctl_cap_scale = property(fget=lambda self: self._read_bf('ctl_cap_scale'),
                                         fset=lambda self,data: self._write_bf('ctl_cap_scale',data),
                                         doc='''Increases capacitor measurement resolution by 100x, this is used when measuring smaller capacitors. (1 bit, R/W)
                                                 Preset 'small': 1
                                                 Preset 'large': 0''')
    ctl_reg = property(fget=lambda self: self._read_bf('ctl_reg'),
                                         fset=lambda self,data: self._write_bf('ctl_reg',data),
                                         doc='''Control Register: Several Control Functions are grouped into this register.''')
    num_caps = property(fget=lambda self: self._read_bf('num_caps'),
                                         fset=lambda self, data: self._error('Write access not allowed to field num_caps'),
                                         doc='''Number of Capacitors. This register shows the state of the CAP_SLCT1, CAP_SLCT0 pins. The value read in this register is the number of capacitors programmed minus one. (2 bits, Read Only)''')
    num_caps_CMD = property(fget=lambda self: self._read_bf('num_caps_CMD'),
                                         fset=lambda self, data: self._error('Write access not allowed to register num_caps_CMD'),
                                         doc='''''')
    chrg_stepdown = property(fget=lambda self: self._read_bf('chrg_stepdown'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_stepdown'),
                                         doc='''The synchronous controller is in step-down mode (charging) (1 bit, Read Only)''')
    chrg_stepup = property(fget=lambda self: self._read_bf('chrg_stepup'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_stepup'),
                                         doc='''The synchronous controller is in step-up mode (backup) (1 bit, Read Only)''')
    chrg_cv = property(fget=lambda self: self._read_bf('chrg_cv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_cv'),
                                         doc='''The charger is in constant voltage mode (1 bit, Read Only)''')
    chrg_uvlo = property(fget=lambda self: self._read_bf('chrg_uvlo'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_uvlo'),
                                         doc='''The charger is in under voltage lockout (1 bit, Read Only)''')
    chrg_input_ilim = property(fget=lambda self: self._read_bf('chrg_input_ilim'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_input_ilim'),
                                         doc='''The charger is in input current limit (1 bit, Read Only)''')
    chrg_cappg = property(fget=lambda self: self._read_bf('chrg_cappg'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_cappg'),
                                         doc='''The capacitor voltage is above power good threshold (1 bit, Read Only)''')
    chrg_shnt = property(fget=lambda self: self._read_bf('chrg_shnt'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_shnt'),
                                         doc='''The capacitor manager is shunting (1 bit, Read Only)''')
    chrg_bal = property(fget=lambda self: self._read_bf('chrg_bal'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_bal'),
                                         doc='''The capacitor manager is balancing (1 bit, Read Only)''')
    chrg_dis = property(fget=lambda self: self._read_bf('chrg_dis'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_dis'),
                                         doc='''The charger is temporarily disabled for capacitance measurement (1 bit, Read Only)''')
    chrg_ci = property(fget=lambda self: self._read_bf('chrg_ci'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_ci'),
                                         doc='''The charger is in constant current mode (1 bit, Read Only)''')
    chrg_pfo = property(fget=lambda self: self._read_bf('chrg_pfo'),
                                         fset=lambda self, data: self._error('Write access not allowed to field chrg_pfo'),
                                         doc='''Input voltage is below PFI threshold (1 bit, Read Only)''')
    chrg_status = property(fget=lambda self: self._read_bf('chrg_status'),
                                         fset=lambda self, data: self._error('Write access not allowed to register chrg_status'),
                                         doc='''Charger Status Register: This register provides real time status information about the state of the charger system. Each bit is active high.''')
    mon_capesr_active = property(fget=lambda self: self._read_bf('mon_capesr_active'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_capesr_active'),
                                         doc='''Capacitance/ESR measurement is in progress (1 bit, Read Only)''')
    mon_capesr_scheduled = property(fget=lambda self: self._read_bf('mon_capesr_scheduled'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_capesr_scheduled'),
                                         doc='''Waiting programmed time to begin a capacitance/ESR measurement (1 bit, Read Only)''')
    mon_capesr_pending = property(fget=lambda self: self._read_bf('mon_capesr_pending'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_capesr_pending'),
                                         doc='''Waiting for satisfactory conditions to begin a capacitance/ESR measurement (1 bit, Read Only)''')
    mon_cap_done = property(fget=lambda self: self._read_bf('mon_cap_done'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_cap_done'),
                                         doc='''Capacitance measurement has completed (1 bit, Read Only)''')
    mon_esr_done = property(fget=lambda self: self._read_bf('mon_esr_done'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_esr_done'),
                                         doc='''ESR Measurement has completed (1 bit, Read Only)''')
    mon_cap_failed = property(fget=lambda self: self._read_bf('mon_cap_failed'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_cap_failed'),
                                         doc='''The last attempted capacitance measurement was unable to complete (1 bit, Read Only)''')
    mon_esr_failed = property(fget=lambda self: self._read_bf('mon_esr_failed'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_esr_failed'),
                                         doc='''The last attempted ESR measurement was unable to complete (1 bit, Read Only)''')
    mon_power_failed = property(fget=lambda self: self._read_bf('mon_power_failed'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_power_failed'),
                                         doc='''This bit is set when VIN falls below the PFI threshold or the charger is unable to charge. It is cleared only when power returns and the charger is able to charge. (1 bit, Read Only)''')
    mon_power_returned = property(fget=lambda self: self._read_bf('mon_power_returned'),
                                         fset=lambda self, data: self._error('Write access not allowed to field mon_power_returned'),
                                         doc='''This bit is set when the input is above the PFI threshold and the charger is able to charge. It is cleared only when mon_power_failed is set. (1 bit, Read Only)''')
    mon_status = property(fget=lambda self: self._read_bf('mon_status'),
                                         fset=lambda self, data: self._error('Write access not allowed to register mon_status'),
                                         doc='''Monitor Status: This register provides real time status information about the state of the monitoring system. Each bit is active high.''')
    alarm_cap_uv = property(fget=lambda self: self._read_bf('alarm_cap_uv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_cap_uv'),
                                         doc='''Capacitor undervoltage alarm (1 bit, Read Only)''')
    alarm_cap_ov = property(fget=lambda self: self._read_bf('alarm_cap_ov'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_cap_ov'),
                                         doc='''Capacitor overvoltage alarm (1 bit, Read Only)''')
    alarm_gpi_uv = property(fget=lambda self: self._read_bf('alarm_gpi_uv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_gpi_uv'),
                                         doc='''GPI undervoltage alarm (1 bit, Read Only)''')
    alarm_gpi_ov = property(fget=lambda self: self._read_bf('alarm_gpi_ov'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_gpi_ov'),
                                         doc='''GPI overvoltage alarm (1 bit, Read Only)''')
    alarm_vin_uv = property(fget=lambda self: self._read_bf('alarm_vin_uv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vin_uv'),
                                         doc='''VIN undervoltage alarm (1 bit, Read Only)''')
    alarm_vin_ov = property(fget=lambda self: self._read_bf('alarm_vin_ov'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vin_ov'),
                                         doc='''VIN overvoltage alarm (1 bit, Read Only)''')
    alarm_vcap_uv = property(fget=lambda self: self._read_bf('alarm_vcap_uv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vcap_uv'),
                                         doc='''VCAP undervoltage alarm (1 bit, Read Only)''')
    alarm_vcap_ov = property(fget=lambda self: self._read_bf('alarm_vcap_ov'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vcap_ov'),
                                         doc='''VCAP overvoltage alarm (1 bit, Read Only)''')
    alarm_vout_uv = property(fget=lambda self: self._read_bf('alarm_vout_uv'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vout_uv'),
                                         doc='''VOUT undervoltage alarm (1 bit, Read Only)''')
    alarm_vout_ov = property(fget=lambda self: self._read_bf('alarm_vout_ov'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_vout_ov'),
                                         doc='''VOUT overvoltage alarm (1 bit, Read Only)''')
    alarm_iin_oc = property(fget=lambda self: self._read_bf('alarm_iin_oc'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_iin_oc'),
                                         doc='''Input overcurrent alarm (1 bit, Read Only)''')
    alarm_ichg_uc = property(fget=lambda self: self._read_bf('alarm_ichg_uc'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_ichg_uc'),
                                         doc='''Charge undercurrent alarm (1 bit, Read Only)''')
    alarm_dtemp_cold = property(fget=lambda self: self._read_bf('alarm_dtemp_cold'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_dtemp_cold'),
                                         doc='''Die temperature cold alarm (1 bit, Read Only)''')
    alarm_dtemp_hot = property(fget=lambda self: self._read_bf('alarm_dtemp_hot'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_dtemp_hot'),
                                         doc='''Die temperature hot alarm (1 bit, Read Only)''')
    alarm_esr_hi = property(fget=lambda self: self._read_bf('alarm_esr_hi'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_esr_hi'),
                                         doc='''ESR high alarm (1 bit, Read Only)''')
    alarm_cap_lo = property(fget=lambda self: self._read_bf('alarm_cap_lo'),
                                         fset=lambda self, data: self._error('Write access not allowed to field alarm_cap_lo'),
                                         doc='''Capacitance low alarm (1 bit, Read Only)''')
    alarm_reg = property(fget=lambda self: self._read_bf('alarm_reg'),
                                         fset=lambda self, data: self._error('Write access not allowed to register alarm_reg'),
                                         doc='''Alarms Register: A one in any bit in the register indicates its respective alarm has triggered. All bits are active high.''')
    meas_cap = property(fget=lambda self: self._read_bf('meas_cap'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_cap'),
                                         doc='''Measured capacitor stack capacitance value.  When ctl_cap_scale is set to 1, capacitance is 3.36uF * RT/RTST per LSB. When ctl_cap_scale is set to 0 it is 336uF * RT/RTST per LSB. (16 bits, Read Only)
                                                 Format: cap
                                                 Format: cap_zs''')
    meas_esr = property(fget=lambda self: self._read_bf('meas_esr'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_esr'),
                                         doc='''Measured capacitor stack equivalent series resistance (ESR) value. RSNSC/64 per LSB (16 bits, Read Only)
                                                 Format: esr''')
    meas_vcap1 = property(fget=lambda self: self._read_bf('meas_vcap1'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vcap1'),
                                         doc='''Measured voltage between the CAP1 and CAPRTN pins. 183.5uV per LSB (16 bits, Read Only)
                                                 Format: cell''')
    meas_vcap2 = property(fget=lambda self: self._read_bf('meas_vcap2'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vcap2'),
                                         doc='''Measured voltage between the CAP2 and CAP1 pins. 183.5uV per LSB (16 bits, Read Only)
                                                 Format: cell''')
    meas_vcap3 = property(fget=lambda self: self._read_bf('meas_vcap3'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vcap3'),
                                         doc='''Measured voltage between the CAP3 and CAP2 pins. 183.5uV per LSB (16 bits, Read Only)
                                                 Format: cell''')
    meas_vcap4 = property(fget=lambda self: self._read_bf('meas_vcap4'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vcap4'),
                                         doc='''Measured voltage between the CAP4 and CAP3 pins. 183.5uV per LSB (16 bits, Read Only)
                                                 Format: cell''')
    meas_gpi = property(fget=lambda self: self._read_bf('meas_gpi'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_gpi'),
                                         doc='''Measurement of GPI pin voltage. 183.5uV per LSB (16 bits, Read Only)
                                                 Format: NTCS0402E3103FLT
                                                 Format: cell''')
    meas_vin = property(fget=lambda self: self._read_bf('meas_vin'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vin'),
                                         doc='''Measured Input Voltage. 2.21mV per LSB (16 bits, Read Only)
                                                 Format: vin
                                                 Format: adc''')
    meas_vcap = property(fget=lambda self: self._read_bf('meas_vcap'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vcap'),
                                         doc='''Measured Capacitor Stack Voltage. 1.476mV per LSB. (16 bits, Read Only)
                                                 Format: vcap
                                                 Format: adc''')
    meas_vout = property(fget=lambda self: self._read_bf('meas_vout'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_vout'),
                                         doc='''Measured Output Voltage. 2.21mV per LSB. (16 bits, Read Only)
                                                 Format: vout
                                                 Format: adc''')
    meas_iin = property(fget=lambda self: self._read_bf('meas_iin'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_iin'),
                                         doc='''Measured Input Current. 1.983uV/RSNSI per LSB (16 bits, Read Only)
                                                 Format: iin
                                                 Format: adc''')
    meas_ichg = property(fget=lambda self: self._read_bf('meas_ichg'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_ichg'),
                                         doc='''Measured Charge Current. 1.983uV/RSNSC per LSB (16 bits, Read Only)
                                                 Format: icharge
                                                 Format: adc''')
    meas_dtemp = property(fget=lambda self: self._read_bf('meas_dtemp'),
                                         fset=lambda self, data: self._error('Write access not allowed to field meas_dtemp'),
                                         doc='''Measured die temperature. Temperature = 0.028C per LSB - 251.4C (16 bits, Read Only)
                                                 Format: dtemp
                                                 Format: adc''')

    _constants = {
        'RSNSI': 0.016,
        'RSNSC': 0.006,
        'RTST': 121,
        'RT': 71.5e3,
        'CTL_CAP_SCALE_VAL': 0,
        'GPI_ADC_SCALE': 183.5e-6,
        'GPI_VREF': 2.5,
        'RNTCBIAS': 10000.0,
        'RNTCSER': 0.0,
        'Rm40': 214063.67,
        'Rm34': 152840.30,
        'Rm28': 110480.73,
        'Rm21': 76798.02,
        'Rm14': 54214.99,
        'Rm6': 37075.65,
        'R4': 23649.71,
        'R33': 7400.97,
        'R44': 5001.22,
        'R53': 3693.55,
        'R62': 2768.21,
        'R70': 2167.17,
        'R78': 1714.08,
        'R86': 1368.87,
        'R94': 1103.18,
        'R102': 896.73,
        'R110': 734.86,
        'R118': 606.86,
        'R126': 504.80,
        'R134': 422.81,
        'R142': 356.45,
        'R150': 302.36,
        }

    _xml_formats = {
        'NTCS0402E3103FLT': {'points': [('(Rm40 + RNTCSER) / (Rm40 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-40'), ('(Rm34 + RNTCSER) / (Rm34 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-34'), ('(Rm28 + RNTCSER) / (Rm28 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-28'), ('(Rm21 + RNTCSER) / (Rm21 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-21'), ('(Rm14 + RNTCSER) / (Rm14 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-14'), ('(Rm6 + RNTCSER) / (Rm6 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '-6'), ('(R4 + RNTCSER) / (R4 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '4'), ('(R33 + RNTCSER) / (R33 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '33'), ('(R44 + RNTCSER) / (R44 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '44'), ('(R53 + RNTCSER) / (R53 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '53'), ('(R62 + RNTCSER) / (R62 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '62'), ('(R70 + RNTCSER) / (R70 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '70'), ('(R78 + RNTCSER) / (R78 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '78'), ('(R86 + RNTCSER) / (R86 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '86'), ('(R94 + RNTCSER) / (R94 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '94'), ('(R102 + RNTCSER) / (R102 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '102'), ('(R110 + RNTCSER) / (R110 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '110'), ('(R118 + RNTCSER) / (R118 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '118'), ('(R126 + RNTCSER) / (R126 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '126'), ('(R134 + RNTCSER) / (R134 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '134'), ('(R142 + RNTCSER) / (R142 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '142'), ('(R150 + RNTCSER) / (R150 + RNTCSER + RNTCBIAS) * GPI_VREF / GPI_ADC_SCALE', '150')], 'description': '''Convert from °C to the thermistor ADC reading.''', 'signed': True},
        'cap': {'points': [('0', '0'), ('1', '(3.36e-6 + 332.64e-6 * (1 - CTL_CAP_SCALE_VAL)) * RT / RTST')], 'description': '''Measured capacitor stack capacitance value. When CTL_CAP_SCALE is set to 1, capacitance is 3.36uF * RT/RTST per LSB. When CTL_CAP_SCALE is set to 0 it is 336uF * RT/RTST per LSB.''', 'signed': False},
        'vcapfb_dac_format': {'points': [('0', '0.6375'), ('1', '0.6375 + 0.0375')], 'description': '''This is used to program the capacitor voltage feedback loop's reference voltage. Only bits 3:0 are active. CAPFBREF = 37.5mV * vcapfb_dac + 637.5mV.''', 'signed': False},
        'cell': {'points': [('0', '0'), ('1', '183.5e-6')], 'description': '''Measured voltage between CAP pins or CAP1 and CAPRTN.''', 'signed': True},
        'vin': {'points': [('0', '0'), ('1', '2.21e-3')], 'description': '''Measured input voltage.''', 'signed': True},
        'vcap': {'points': [('0', '0'), ('1', '1.476e-3')], 'description': '''Measured capacitor stack voltage.''', 'signed': True},
        'vout': {'points': [('0', '0'), ('1', '2.21e-3')], 'description': '''Measured output voltage''', 'signed': True},
        'iin': {'points': [('0', '0'), ('1', '1.983e-6 / RSNSI')], 'description': '''Measured input current.''', 'signed': True},
        'dtemp': {'points': [('0', '-251.4'), ('1', '-251.4 + 0.028')], 'description': '''Measured die temperature.''', 'signed': True},
        'esr': {'points': [('0', '0'), ('1', 'RSNSC / 64.0')], 'description': '''Measured capacitor stack equivalent series resistance (ESR) value.''', 'signed': False},
        'icharge': {'points': [('0', '0'), ('1', '1.983e-6 / RSNSC')], 'description': '''Measured Charge Current.''', 'signed': True},
        'cap_per': {'points': [('0', '0'), ('1', '10.0')], 'description': '''Capacitance and ESR Measurement Period: This register sets the period of repeated capacitance and ESR measurements.''', 'signed': False},
        'adc': {'points': [('0', '0'), ('1', '1 / 24270.0')], 'description': '''Native ADC span.''', 'signed': True},
        'cap_zs': {'points': [('0', '0'), ('1', '1.0')], 'description': '''Unmodified capacitor data''', 'signed': True},
        }

    #Caution! This register map, including active_format settings is shared across all instances of the class.
    _register_map = {
        'clr_cap_lo': {'command_code':0x00, 'size':1, 'offset':15, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_esr_hi': {'command_code':0x00, 'size':1, 'offset':14, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_dtemp_hot': {'command_code':0x00, 'size':1, 'offset':13, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_dtemp_cold': {'command_code':0x00, 'size':1, 'offset':12, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_ichg_uc': {'command_code':0x00, 'size':1, 'offset':11, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_iin_oc': {'command_code':0x00, 'size':1, 'offset':10, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vout_ov': {'command_code':0x00, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vout_uv': {'command_code':0x00, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vcap_ov': {'command_code':0x00, 'size':1, 'offset':7, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vcap_uv': {'command_code':0x00, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vin_ov': {'command_code':0x00, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_vin_uv': {'command_code':0x00, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_gpi_ov': {'command_code':0x00, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_gpi_uv': {'command_code':0x00, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_cap_ov': {'command_code':0x00, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_cap_uv': {'command_code':0x00, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'clr_alarms': {'command_code':0x00, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_cap_uv': {'command_code':0x01, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_cap_ov': {'command_code':0x01, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_gpi_uv': {'command_code':0x01, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_gpi_ov': {'command_code':0x01, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vin_uv': {'command_code':0x01, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vin_ov': {'command_code':0x01, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vcap_uv': {'command_code':0x01, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vcap_ov': {'command_code':0x01, 'size':1, 'offset':7, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vout_uv': {'command_code':0x01, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_vout_ov': {'command_code':0x01, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_iin_oc': {'command_code':0x01, 'size':1, 'offset':10, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_ichg_uc': {'command_code':0x01, 'size':1, 'offset':11, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_dtemp_cold': {'command_code':0x01, 'size':1, 'offset':12, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_dtemp_hot': {'command_code':0x01, 'size':1, 'offset':13, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_esr_hi': {'command_code':0x01, 'size':1, 'offset':14, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_cap_lo': {'command_code':0x01, 'size':1, 'offset':15, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_alarms': {'command_code':0x01, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_capesr_active': {'command_code':0x02, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_capesr_scheduled': {'command_code':0x02, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_capesr_pending': {'command_code':0x02, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_cap_done': {'command_code':0x02, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_esr_done': {'command_code':0x02, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_cap_failed': {'command_code':0x02, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_esr_failed': {'command_code':0x02, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_power_failed': {'command_code':0x02, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_power_returned': {'command_code':0x02, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'msk_mon_status': {'command_code':0x02, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'cap_esr_per': {'command_code':0x04, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cap_per'],'active_format':'cap_per'},
        'vcapfb_dac': {'command_code':0x05, 'size':4, 'offset':0, 'presets':[], 'allowed_formats':['vcapfb_dac_format'],'active_format':'vcapfb_dac_format'},
        'vcapfb_dac_CMD': {'command_code':0x05, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'vshunt': {'command_code':0x06, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'cap_uv_lvl': {'command_code':0x07, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'cap_ov_lvl': {'command_code':0x08, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'gpi_uv_lvl': {'command_code':0x09, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'gpi_ov_lvl': {'command_code':0x0a, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'vin_uv_lvl': {'command_code':0x0b, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vin'],'active_format':'vin'},
        'vin_ov_lvl': {'command_code':0x0c, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vin'],'active_format':'vin'},
        'vcap_uv_lvl': {'command_code':0x0d, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vcap'],'active_format':'vcap'},
        'vcap_ov_lvl': {'command_code':0x0e, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vcap'],'active_format':'vcap'},
        'vout_uv_lvl': {'command_code':0x0f, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vout'],'active_format':'vout'},
        'vout_ov_lvl': {'command_code':0x10, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vout'],'active_format':'vout'},
        'iin_oc_lvl': {'command_code':0x11, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['iin'],'active_format':'iin'},
        'ichg_uc_lvl': {'command_code':0x12, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['icharge'],'active_format':'icharge'},
        'dtemp_cold_lvl': {'command_code':0x13, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['dtemp'],'active_format':'dtemp'},
        'dtemp_hot_lvl': {'command_code':0x14, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['dtemp'],'active_format':'dtemp'},
        'esr_hi_lvl': {'command_code':0x15, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['esr'],'active_format':'esr'},
        'cap_lo_lvl': {'command_code':0x16, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cap'],'active_format':'cap'},
        'ctl_strt_capesr': {'command_code':0x17, 'size':1, 'offset':0, 'presets':[['start',1],], 'allowed_formats':[],'active_format':'None'},
        'ctl_gpi_buffer_en': {'command_code':0x17, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'ctl_stop_capesr': {'command_code':0x17, 'size':1, 'offset':2, 'presets':[['stop',1],], 'allowed_formats':[],'active_format':'None'},
        'ctl_cap_scale': {'command_code':0x17, 'size':1, 'offset':3, 'presets':[['small',1],['large',0],], 'allowed_formats':[],'active_format':'None'},
        'ctl_reg': {'command_code':0x17, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'num_caps': {'command_code':0x1a, 'size':2, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'num_caps_CMD': {'command_code':0x1a, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_stepdown': {'command_code':0x1b, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_stepup': {'command_code':0x1b, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_cv': {'command_code':0x1b, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_uvlo': {'command_code':0x1b, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_input_ilim': {'command_code':0x1b, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_cappg': {'command_code':0x1b, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_shnt': {'command_code':0x1b, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_bal': {'command_code':0x1b, 'size':1, 'offset':7, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_dis': {'command_code':0x1b, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_ci': {'command_code':0x1b, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_pfo': {'command_code':0x1b, 'size':1, 'offset':11, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'chrg_status': {'command_code':0x1b, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_capesr_active': {'command_code':0x1c, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_capesr_scheduled': {'command_code':0x1c, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_capesr_pending': {'command_code':0x1c, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_cap_done': {'command_code':0x1c, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_esr_done': {'command_code':0x1c, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_cap_failed': {'command_code':0x1c, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_esr_failed': {'command_code':0x1c, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_power_failed': {'command_code':0x1c, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_power_returned': {'command_code':0x1c, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'mon_status': {'command_code':0x1c, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_cap_uv': {'command_code':0x1d, 'size':1, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_cap_ov': {'command_code':0x1d, 'size':1, 'offset':1, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_gpi_uv': {'command_code':0x1d, 'size':1, 'offset':2, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_gpi_ov': {'command_code':0x1d, 'size':1, 'offset':3, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vin_uv': {'command_code':0x1d, 'size':1, 'offset':4, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vin_ov': {'command_code':0x1d, 'size':1, 'offset':5, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vcap_uv': {'command_code':0x1d, 'size':1, 'offset':6, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vcap_ov': {'command_code':0x1d, 'size':1, 'offset':7, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vout_uv': {'command_code':0x1d, 'size':1, 'offset':8, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_vout_ov': {'command_code':0x1d, 'size':1, 'offset':9, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_iin_oc': {'command_code':0x1d, 'size':1, 'offset':10, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_ichg_uc': {'command_code':0x1d, 'size':1, 'offset':11, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_dtemp_cold': {'command_code':0x1d, 'size':1, 'offset':12, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_dtemp_hot': {'command_code':0x1d, 'size':1, 'offset':13, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_esr_hi': {'command_code':0x1d, 'size':1, 'offset':14, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_cap_lo': {'command_code':0x1d, 'size':1, 'offset':15, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'alarm_reg': {'command_code':0x1d, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':[],'active_format':'None'},
        'meas_cap': {'command_code':0x1e, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cap', 'cap_zs'],'active_format':'cap'},
        'meas_esr': {'command_code':0x1f, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['esr'],'active_format':'esr'},
        'meas_vcap1': {'command_code':0x20, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'meas_vcap2': {'command_code':0x21, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'meas_vcap3': {'command_code':0x22, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'meas_vcap4': {'command_code':0x23, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['cell'],'active_format':'cell'},
        'meas_gpi': {'command_code':0x24, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['NTCS0402E3103FLT', 'cell'],'active_format':'NTCS0402E3103FLT'},
        'meas_vin': {'command_code':0x25, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vin', 'adc'],'active_format':'vin'},
        'meas_vcap': {'command_code':0x26, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vcap', 'adc'],'active_format':'vcap'},
        'meas_vout': {'command_code':0x27, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['vout', 'adc'],'active_format':'vout'},
        'meas_iin': {'command_code':0x28, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['iin', 'adc'],'active_format':'iin'},
        'meas_ichg': {'command_code':0x29, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['icharge', 'adc'],'active_format':'icharge'},
        'meas_dtemp': {'command_code':0x2a, 'size':16, 'offset':0, 'presets':[], 'allowed_formats':['dtemp', 'adc'],'active_format':'dtemp'},
    }

    _formatters = {}

class LTC3350_APIException(Exception):
    '''Bad data passed to instance of LTC3350 will raise LTC3350_APIException'''
    pass


import serial, time
class ltc_dc590b_interface(object):
    '''Python Communication Interface for DC590B / Linduino.
    Provides the following SMBus protocols with optional PEC (Packet Error Checking) CRC:
        ReadWord
        WriteWord
        ReadByte
        WriteByte
        ReceiveByte
        SendByte
        AlertResponse
    PEC implementation untested.
    Requires PySerial.'''
    def __init__(self, serial_port, PEC=False, **kwargs):
        '''Provide serial port of attached Linear Technology DC590 I2C interface board or DC2026B Linduno board programmed with DC590 emulator sketch.  kwargs will be passed to PySerial init unless serial_port is already configured.'''
        if isinstance(serial_port, serial.Serial):
            self.serial_port = serial_port
        else:
            if 'timeout' not in kwargs:
                kwargs['timeout'] = 1
            if 'baudrate' not in kwargs:
                kwargs['baudrate'] = 115200
            self.serial_port = serial.Serial(serial_port, **kwargs)
        if PEC:
            self.read_word = self._read_word_pec
            self.write_word = self._write_word_pec
            self.read_byte = self._read_byte_pec
            self.write_byte = self._write_byte_pec
            self.receive_byte = self._receive_byte_pec
            self.send_byte = self._send_byte_pec
            self.alert_response = self._alert_response_pec
        else:
            self.read_word = self._read_word_nopec
            self.write_word = self._write_word_nopec
            self.read_byte = self._read_byte_nopec
            self.write_byte = self._write_byte_nopec
            self.receive_byte = self._receive_byte_nopec
            self.send_byte = self._send_byte_nopec
            self.alert_response = self._alert_response_nopec
        self._serial_wait = 0.1
        time.sleep(2.5) #Linduino bootloader delay!
        self.serial_port.write('\n'*10) 
        time.sleep(2.5) #Linduino bootloader delay!
        self.serial_port.write('MI') #Switch to isolated I2C Mode
        self.serial_port.write('O') #Enable isolated power
        time.sleep(self._serial_wait)
        print('DC590B init response: {}'.format(self.serial_port.read(self.serial_port.inWaiting()))) #discard any responses
    def read_word(self, addr_7bit, command_code):
        '''SMBus Read Word Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns 16-bit data from slave.'''
    def write_word(self, addr_7bit, command_code, data16):
        '''SMBus Write Word Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns None.'''
    def read_byte(self, addr_7bit, command_code):
        '''SMBus Read Byte Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
    def write_byte(self, addr_7bit, command_code, data8):
        '''SMBus Write Byte Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns None.'''
    def receive_byte(self, addr_7bit):
        '''SMBus Receive Byte Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
    def send_byte(self, addr_7bit, data8):
        '''SMBus Send Byte Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns None.'''
    def alert_response(self):
        '''SMBus Send Byte Protocol.
        Packet Error Checking controlled by class init.
        Slave device address specified in 7-bit format.
        Returns None if no response to ARA, otherwise the 7-bit device address provided by the slave.
        Returns transmit device address placed in the 7 most significant bits of the byte. The returned LSB will be zero in this implementation (Write address).'''
    def _hex_str(self, integer):
        '''return integer formatted correctly for transmission over DC590 serial link'''
        return hex(integer)[2:].rjust(2,"0").upper()
    def _read_addr(self,addr_7bit):
        '''compute 8-bit read address from 7-bit address'''
        return self._hex_str((addr_7bit << 1) + 1)
    def _write_addr(self, addr_7bit):
        '''compute 8-bit write address from 7-bit address'''
        return self._hex_str(addr_7bit << 1)
    def _word(self, lowByte, highByte):
        '''Return 16-bit value from two SMBus bytes'''
        return ((highByte & 0xFF) << 8) + (lowByte & 0xFF)
    def _lowByte(self, data16):
        return self._hex_str(data16 & 0xFF)
    def _highByte(self, data16):
        return self._hex_str(data16 >> 8)
    def _pec(self,byteList):
        '''byteList is an ordered list of every byte in the transaction including address, command code (subaddr) and data'''
        crc = 0
        poly = 0x07 #x^8 + x^2 + x^1 + 1, discard x^8 term
        for byte in byteList:
            crc ^= byte
            for cycle in range(8):
                crc <<= 1
                if (crc & 0x100): #msb was set before left shift ( & 0x80)
                    #xor with crc if pre-shift msb was 1
                    crc ^= poly
        return int(crc & 0xFF)
    def _read_word_nopec(self, addr_7bit, command_code):
        '''SMBus Read Word Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 16-bit data from slave.'''
        write_str = 'sS{}S{}sS{}QRp'.format(self._write_addr(addr_7bit), self._hex_str(command_code), self._read_addr(addr_7bit))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(4)
        if (len(ret_str) != 4):
            raise Exception('Short response to DC590 read_word command, EEPROM Present?: %s' % ret_str)
        if ('N' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 read_word failed acknowledge: {} then {}'.format(ret_str, ret_extra))
        if ('X' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 EEPROM detection failed, communications disabled: {} then {}'.format(ret_str, ret_extra))
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 read_word command: {} then {}'.format(ret_str, ret_extra))
        return self._word(lowByte=int(ret_str[0:2],16), highByte=int(ret_str[2:4],16))
    def _write_word_nopec(self, addr_7bit, command_code, data16):
        '''SMBus Write Word Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        write_str = 'sS{}S{}S{}S{}p'.format(self._write_addr(addr_7bit), self._hex_str(command_code), self._lowByte(data16), self._highByte(data16))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_word command'.format(ret_str))
    def _read_byte_nopec(self, addr_7bit, command_code):
        '''SMBus Read Byte Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
        write_str = 'sS{}S{}sS{}Rp'.format(self._write_addr(addr_7bit), self._hex_str(command_code), self._read_addr(addr_7bit))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(2)
        if (len(ret_str) != 2):
            raise Exception('Short response to DC590 read_byte command, EEPROM Present?: %s' % ret_str)
        if ('N' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 read_byte failed acknowledge: {} then {}'.format(ret_str, ret_extra))
        if ('X' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 EEPROM detection failed, communications disabled: {} then {}'.format(ret_str, ret_extra))
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 read_byte command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str,16)
    def _write_byte_nopec(self, addr_7bit, command_code, data8):
        '''SMBus Write Byte Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        write_str = 'sS{}S{}S{}p'.format(self._write_addr(addr_7bit), self._hex_str(command_code), self._hex_str(data8))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_byte command'.format(ret_str))
    def _receive_byte_nopec(self, addr_7bit):
        '''SMBus Receive Byte Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
        write_str = 'sS{}Rp'.format(self._read_addr(addr_7bit))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(2)
        if (len(ret_str) != 2):
            raise Exception('Short response to DC590 receive_byte command, EEPROM Present?: %s' % ret_str)
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 receive_byte command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str,16)  
    def _send_byte_nopec(self, addr_7bit, data8):
        '''SMBus Send Byte Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        write_str = 'sS{}S{}p'.format(self._write_addr(addr_7bit), self._hex_str(data8))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_byte command'.format(ret_str))
    def _alert_response_nopec(self):
        '''SMBus Send Byte Protocol without Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None if no response to ARA, otherwise the 7-bit device address provided by the slave
        transmit device is placed in the 7 most significant bits of the byte. The eighth bit can be a zero or one.'''
        write_str = 'sS{}Rp'.format(self._read_addr(0x0C))
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(2)
        if (len(ret_str) != 2):
            raise Exception('Short response to DC590 ARA command')
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 ARA command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str,16) >> 1  
    def _read_word_pec(self, addr_7bit, command_code):
        '''SMBus Read Word Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 16-bit data from slave.'''
        byteList = [self._write_addr(addr_7bit), self._hex_str(command_code), self._read_addr(addr_7bit)]
        write_str = 'sS{}S{}sS{}QQRp'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(6)
        if (len(ret_str) != 6):
            raise Exception('Short response to DC590 read_word_pec command, EEPROM Present?: %s' % ret_str)
        byteList.append(int(ret_str[0:2],16))
        byteList.append(int(ret_str[2:4],16))
        if int(ret_str[4:6],16) != self._pec(byteList):
            raise Exception('DC590 read_word_pec command failed PEC')
        if ('N' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 read_word failed acknowledge: {} then {}'.format(ret_str, ret_extra))
        if ('X' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 EEPROM detection failed, communications disabled: {} then {}'.format(ret_str, ret_extra))
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 read_word command: {} then {}'.format(ret_str, ret_extra))
        return self._word(lowByte=int(ret_str[0:2],16), highByte=int(ret_str[2:4],16))
    def _write_word_pec(self, addr_7bit, command_code, data):
        '''SMBus Write Word Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        byteList = [self._write_addr(addr_7bit), self._hex_str(command_code), self._lowByte(data16), self._highByte(data16)]
        byteList.append(self._pec(byteList))
        write_str = 'sS{}S{}S{}S{}S{}p'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_word command'.format(ret_str))
    def _read_byte_pec(self, addr_7bit, command_code):
        '''SMBus Read Byte Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
        byteList = [self._write_addr(addr_7bit), self._hex_str(command_code), self._read_addr(addr_7bit)]
        write_str = 'sS{}S{}sS{}QRp'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(4)
        if (len(ret_str) != 4):
            raise Exception('Short response to DC590 read_byte_pec command, EEPROM Present?: %s' % ret_str)
        byteList.append(int(ret_str[0:2],16))
        if int(ret_str[2:4],16) != self._pec(byteList):
            raise Exception('DC590 read_byte_pec command failed PEC')
        if ('N' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 read_word_pec failed acknowledge: {} then {}'.format(ret_str, ret_extra))
        if ('X' in ret_str):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('DC590 EEPROM detection failed, communications disabled: {} then {}'.format(ret_str, ret_extra))
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 read_byte_pec command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str[0:2],16)
    def _write_byte_pec(self, addr_7bit, command_code, data):
        '''SMBus Write Byte Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        byteList = [self._write_addr(addr_7bit), self._hex_str(command_code), self._hex_str(data8)]
        byteList.append(self._pec(byteList))
        write_str = 'sS{}S{}S{}S{}p'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_byte command'.format(ret_str))
    def _receive_byte_pec(self, addr_7bit):
        '''SMBus Receive Byte Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns 8-bit data from slave.'''
        byteList = [self._read_addr(addr_7bit)]
        write_str = 'sS{}QRp'.format()
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(4)
        if (len(ret_str) != 4):
            raise Exception('Short response to DC590 receive_byte_pec command, EEPROM Present?: %s' % ret_str)
        byteList.append(int(ret_str[0:2],16))
        if int(ret_str[2:4],16) != self._pec(byteList):
            raise Exception('DC590 receive_byte_pec command failed PEC')
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 receive_byte_pec command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str[0:2],16)  
    def _send_byte_pec(self, addr_7bit, data8):
        '''SMBus Send Byte Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None.'''
        byteList = [self._write_addr(addr_7bit), self._hex_str(data8)]
        byteList.append(self._pec(byteList))
        write_str = 'sS{}S{}S{}p'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(self.serial_port.inWaiting())
        if (len(ret_str) != 0):
            raise Exception('Response: {} from DC590 write_byte command'.format(ret_str))
    def _alert_response_pec(self):
        '''SMBus Send Byte Protocol with Packet Error Checking.
        Slave device address specified in 7-bit format.
        Returns None if no response to ARA, otherwise the 7-bit device address provided by the slave
        transmit device is placed in the 7 most significant bits of the byte. The eighth bit can be a zero or one.'''
        byteList = [self._read_addr(0x0C)]
        write_str = 'sS{}QRp'.format(*byteList)
        self.serial_port.write(write_str)
        ret_str = self.serial_port.read(4)
        if (len(ret_str) != 4):
            raise Exception('Short response to DC590 ARA_pec command')
        byteList.append(int(ret_str[0:2],16))
        if int(ret_str[2:4],16) != self._pec(byteList):
            raise Exception('DC590 ARA_pec command failed PEC')
        if (self.serial_port.inWaiting()):
            time.sleep(self._serial_wait)
            ret_extra = self.serial_port.read(self.serial_port.inWaiting())
            raise Exception('Long response to DC590 ARA command: {} then {}'.format(ret_str, ret_extra))
        return int(ret_str[0:2],16) >> 1  

if __name__ == '__main__':
    print('\n\nPlease see the file "example.py"\n\n')
