// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
module usb_tx
(
	input arst,
	
	//write side (usb side)
	input wr_clk,
	input wr_en, 
	input c_to_usb_stop,
	input [15:0] cmd,
	input [15:0] wr_D,
	output reg wr_ready, 
	
	//read side (fpga side)
	input rd_clk,
	input rd_en,
	output reg rd_ready,
	output [15:0] rd_D
);
	
	// write state definations
	parameter
		skip_cmd_word  = 3'b000,
		write_waiting  = 3'b001,
		first_write    = 3'b010,
		writing        = 3'b011,
		reset_counter  = 3'b100,
		write_cmd      = 3'b101,
		write_complete = 3'b110,
		write_default  = 3'bxxx;
		
	// read state definations
	parameter
		read_waiting   = 3'b000,
		first_read     = 3'b001, // remove mem latency
		second_read    = 3'b010, // remove mem latency
		reading        = 3'b011,
		read_complete  = 3'b101,
		read_default   = 3'bxxx;
	
	// state variables
	reg [2:0] write_state;
	reg [2:0] write_next_state;
	reg [2:0] read_state;
	reg [2:0] read_next_state;
	
	// address counter variables
	reg [7:0] wr_A;
	reg [7:0] rd_A;
	
	// state machine outputs
	reg reset_wr_counter;
	reg cmd_wr_en;
	reg wr_complete;
	reg initial_reads;
	reg rd_complete;
	
	wire wr_sync;
	wire rd_sync;
	
	// internal wr en and rd en with protection circuitry
	wire internal_wr_en = ( wr_ready & wr_en ) | cmd_wr_en;
	wire internal_rd_en = ( rd_ready & rd_en ) | initial_reads;
	
	// ram buffer with two separate memory blocks
	usb_double_buffer usbbuffer
	(
		.arst( arst ),
	
		//write side
		.wr_clk( wr_clk ),
		.wr_en( internal_wr_en ),
		.wr_complete( wr_complete ),
		.wr_A( wr_A ),
		.wr_D( cmd_wr_en ? cmd : wr_D ),
		
		.wr_sync( wr_sync ),
		
		//read side
		.rd_clk( rd_clk ),
		.rd_en( internal_rd_en ),
		.rd_complete( rd_complete ),
		.rd_A( rd_A ),
		
		.rd_sync( rd_sync ),
		.rd_D( rd_D )
	);
	
	// write address counter
	always@( posedge wr_clk, posedge arst )
		if( arst )
			wr_A <= 8'b0;
		else if( reset_wr_counter )
			wr_A <= 8'b0;
		else if( internal_wr_en )
			wr_A <= wr_A + 8'b1;
	
	// read address counter
	always@( posedge rd_clk, posedge arst )
		if( arst )
			rd_A <= 8'b0;
		else if( rd_complete )
			rd_A <= 8'b0;
		else if( internal_rd_en )
			rd_A <= rd_A + 8'b1;

//-----------------------------------------------------------------------------
// write state machine
//-----------------------------------------------------------------------------

	// state register
	always@( posedge wr_clk, posedge arst )
		if( arst )
			write_state <= skip_cmd_word;
		else
			write_state <= write_next_state;
	
	// state transition and output logic
	always@( * )
	begin
		wr_ready = 1'b0;
		reset_wr_counter = 1'b0;
		cmd_wr_en = 1'b0;
		wr_complete = 1'b0;
		write_next_state = write_state;
		
		case( write_state )
		
			skip_cmd_word: begin
				cmd_wr_en = 1'b1;
				write_next_state = write_waiting;
			end
		
			write_waiting: begin
				if( wr_sync )
					write_next_state = first_write;
			end
			
			first_write: begin
				wr_ready = 1'b1;
				if( internal_wr_en )
					write_next_state = writing;
			end
			
			writing: begin
				wr_ready = 1'b1;
				if( (internal_wr_en & (wr_A == 8'b11111111)) | c_to_usb_stop )
					write_next_state = reset_counter;
			end
			
			reset_counter: begin
				reset_wr_counter = 1'b1;
				write_next_state = write_cmd;
			end
			
			write_cmd: begin
				cmd_wr_en = 1'b1;
				write_next_state = write_complete;
			end
			
			write_complete: begin
				wr_complete = 1'b1;
				if( ~wr_sync )
					write_next_state = write_waiting;
			end
			
			default: write_next_state = write_default;
		
		endcase
	end

//-----------------------------------------------------------------------------
// read state machine
//-----------------------------------------------------------------------------

	// state register
	always@( posedge rd_clk, posedge arst )
		if( arst )
			read_state <= read_waiting;
		else
			read_state <= read_next_state;
	
	// state transition and output logic
	always@( * )
	begin
		initial_reads = 1'b0;
		rd_ready = 1'b0;
		rd_complete = 1'b0;
		read_next_state = read_state;
		
		case( read_state )
		
			read_waiting: begin
				if( rd_sync )
					read_next_state = first_read;
			end
			
			first_read: begin
				initial_reads = 1'b1;
				read_next_state = second_read;
			end
			
			second_read: begin
				initial_reads = 1'b1;
				read_next_state = reading;
			end
						
			reading: begin
				rd_ready = 1'b1;
				if( internal_rd_en & (rd_A == 8'b00000001) )
					read_next_state = read_complete;
			end
			
			read_complete: begin
				rd_complete = 1'b1;
				if( ~rd_sync )
					read_next_state = read_waiting;
			end
			
			default: read_next_state = read_default;
		
		endcase
	end
		
endmodule
