// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
module usb_rx
(
	input arst,
	
	//write side (usb side)
	input wr_clk,
	input wr_en, 
	input [15:0] wr_D,
	output reg wr_ready,
	
	//read side (fpga side)
	input rd_clk,
	input rd_en,
	input c_to_usb_stop,
	output reg usb_to_c_start,
	output reg rd_ready,
	output reg usb_to_c_stop,
	output reg [15:0] cmd,
	output [15:0] rd_D
);
	
	// write state definations
	parameter
		write_waiting  = 2'b00,
		writing        = 2'b01,
		write_complete = 2'b10,
		write_default  = 2'bxx;
		
	// read state definations
	parameter
		read_waiting        = 4'b0000,
		first_read          = 4'b0001, // remove mem latency
		second_read         = 4'b0010, // remove mem latency
		cmd_read            = 4'b0011,
		check_start         = 4'b0100,
		assert_start        = 4'b0101,
		reading             = 4'b0110,
		assert_end          = 4'b0111,
		read_complete       = 4'b1000,
		read_default        = 4'bxxxx;
	
	// state variables
	reg [1:0] write_state;
	reg [1:0] write_next_state;
	reg [3:0] read_state;
	reg [3:0] read_next_state;
	
	// address counter variables
	reg [7:0] wr_A;
	reg [7:0] rd_A;
	
	// state machine outputs
	reg wr_complete;
	reg initial_reads;
	reg rd_complete;
	
	wire wr_sync;
	wire rd_sync;
	
	// internal wr en and rd en with protection circuitry
	wire internal_wr_en = wr_ready & wr_en;
	wire internal_rd_en = ( rd_ready & rd_en ) | initial_reads;
	
	// ram buffer with two separate memory blocks
	usb_double_buffer usbbuffer
	(
		.arst( arst ),
	
		//write side
		.wr_clk( wr_clk ),
		.wr_en( internal_wr_en ),
		.wr_complete( wr_complete ),
		.wr_A( wr_A ),
		.wr_D( wr_D ),
		
		.wr_sync( wr_sync ),
		
		//read side
		.rd_clk( rd_clk ),
		.rd_en( internal_rd_en ),
		.rd_complete( rd_complete ),
		.rd_A( rd_A ),
		
		.rd_sync( rd_sync ),
		.rd_D( rd_D )
	);
	
	//cmd register
	reg cmd_reg_en;
	always@( posedge rd_clk, posedge arst )
		if( arst )
			cmd <= 16'b0;
		else if( cmd_reg_en )
			cmd <= rd_D;
	
	// write address counter
	always@( posedge wr_clk, posedge arst )
		if( arst )
			wr_A <= 8'b0;
		else if( wr_complete )
			wr_A <= 8'b0;
		else if( internal_wr_en )
			wr_A <= wr_A + 8'b1;
	
	// read address counter
	always@( posedge rd_clk, posedge arst )
		if( arst )
			rd_A <= 8'b0;
		else if( rd_complete )
			rd_A <= 8'b0;
		else if( internal_rd_en )
			rd_A <= rd_A + 8'b1;

//-----------------------------------------------------------------------------
// write state machine
//-----------------------------------------------------------------------------

	// state register
	always@( posedge wr_clk, posedge arst )
		if( arst )
			write_state <= write_waiting;
		else
			write_state <= write_next_state;
	
	// state transition and output logic
	always@( * )
	begin
		wr_ready = 1'b0;
		wr_complete = 1'b0;
		write_next_state = write_state;
		
		case( write_state )
		
			write_waiting: begin
				if( wr_sync )
					write_next_state = writing;
			end
			
			writing: begin
				wr_ready = 1'b1;
				if( internal_wr_en & (wr_A == 8'b11111111) )
					write_next_state = write_complete;
			end
			
			write_complete: begin
				wr_complete = 1'b1;
				if( ~wr_sync )
					write_next_state = write_waiting;
			end
			
			default: write_next_state = write_default;
		
		endcase
	end

//-----------------------------------------------------------------------------
// read state machine
//-----------------------------------------------------------------------------

	// state register
	always@( posedge rd_clk, posedge arst )
		if( arst )
			read_state <= read_waiting;
		else
			read_state <= read_next_state;
	
	// state transition and output logic
	always@( * )
	begin
		initial_reads = 1'b0;
		cmd_reg_en = 1'b0;
		usb_to_c_start = 1'b0;
		rd_ready = 1'b0;
		usb_to_c_stop = 1'b0;
		rd_complete = 1'b0;
		read_next_state = read_state;
		
		case( read_state )
		
			read_waiting: begin
				if( rd_sync )
					read_next_state = first_read;
			end
			
			first_read: begin
				initial_reads = 1'b1;
				read_next_state = second_read;
			end
			
			second_read: begin
				initial_reads = 1'b1;
				read_next_state = cmd_read;
			end
			
			cmd_read: begin
				cmd_reg_en = 1'b1;
				initial_reads = 1'b1;
				read_next_state = check_start;
			end
			
			check_start: begin
				if( cmd[15] )                                //if start bit set
					read_next_state = assert_start;
				else
					read_next_state = reading;
			end
			
			assert_start: begin
				usb_to_c_start = 1'b1;
				read_next_state = reading;
			end
						
			reading: begin
				rd_ready = 1'b1;
				if( (internal_rd_en & (rd_A == 8'b00000001)) | c_to_usb_stop )
				begin
					if( cmd[14] )                              //if end bit set
						read_next_state = assert_end;
					else
						read_next_state = read_complete;
				end
			end
			
			assert_end: begin
				usb_to_c_stop = 1'b1;
				if( c_to_usb_stop )
					read_next_state = read_complete;
			end
			
			read_complete: begin
				rd_complete = 1'b1;
				if( ~rd_sync )
					read_next_state = read_waiting;
			end
			
			default: read_next_state = read_default;
		
		endcase
	end
		
endmodule
