// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
module usb_double_buffer
(
	input arst,
	
	//write side
	input wr_clk,
	input wr_en,
	input wr_complete,
	input [7:0] wr_A,
	input [15:0] wr_D,
	output wr_sync,
	
	//read side
	input rd_clk,
	input rd_en,
	input rd_complete,
	input [7:0] rd_A,
	output rd_sync,
	output [15:0] rd_D
);

	// write state definations
	parameter
		wr_mem0    = 2'b00,
		wr_wait1   = 2'b01,
		wr_mem1    = 2'b10,
		wr_wait0   = 2'b11,
		wr_default = 2'bxx;

	// read state definations
	parameter
		rd_wait0   = 2'b00,
		rd_mem0    = 2'b01,
		rd_wait1   = 2'b10,
		rd_mem1    = 2'b11,
		rd_default = 2'bxx;
	
	// state variables
	reg [1:0] wr_state;
	reg [1:0] wr_next_state; 
	reg [1:0] rd_state;
	reg [1:0] rd_next_state;
	
	//write state machine inputs
	reg rd0wait_wr;
	reg rd0_wr;
	reg rd1wait_wr;
	reg rd1_wr;
	
	//write state machine outputs
	reg wr0;
	reg wr1wait;
	reg wr1;
	reg wr0wait;
	
	//Buses used in synchronising signals between the two clock domains
	reg [3:0] wr_to_rd;
	reg [3:0] rd_to_wr;
	
	//read state machine inputs
	reg wr0_rd;
	reg wr1wait_rd;
	reg wr1_rd;
	reg wr0wait_rd;
	
	//read state machine outputs
	reg rd0wait;
	reg rd0;
	reg rd1wait;
	reg rd1;
	
	// memory outputs
	wire [15:0] rd_D0;
	wire [15:0] rd_D1;
	

	// Two dual port rams
	mem_block mem0
	(
		.wr_clk  ( wr_clk ),
		.wr_en   ( wr_en & wr0 ),
		.wr_A    ( wr_A ),
		.wr_D    ( wr_D ),
		.rd_clk  ( rd_clk ),
		.rd_en   ( rd_en & rd0 ),
		.rd_A    ( rd_A ),
		
		.rd_D    ( rd_D0 )
	);
	
	mem_block mem1
	(
		.wr_clk  ( wr_clk ),
		.wr_en   ( wr_en & wr1 ),
		.wr_A    ( wr_A ),
		.wr_D    ( wr_D ),
		.rd_clk  ( rd_clk ),
		.rd_en   ( rd_en & rd1 ),
		.rd_A    ( rd_A ),
		
		.rd_D    ( rd_D1 )
	);

	// mux memory outputs
	assign rd_D = rd0 ? rd_D0 : rd1 ? rd_D1 : 16'hxxxx;

	// sync signal outputs
	assign wr_sync = wr0 | wr1;
	assign rd_sync = rd0 | rd1;
	
	//Synchronise wr outputs to rd clk !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	always@( posedge rd_clk )
	begin
		wr_to_rd <= { wr0, wr1wait, wr1, wr0wait };
		{ wr0_rd, wr1wait_rd, wr1_rd, wr0wait_rd } <= wr_to_rd;
	end
	
	//Synchronise rd outputs to wr clk !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	always@( posedge wr_clk )
	begin
		rd_to_wr <= { rd0wait, rd0, rd1wait, rd1 };
		{ rd0wait_wr, rd0_wr, rd1wait_wr, rd1_wr } <= rd_to_wr;
	end
		

//-----------------------------------------------------------------------------
// write side sync state machine
//-----------------------------------------------------------------------------
	
	// write state register
	always@( posedge wr_clk, posedge arst )
		if( arst )
			wr_state <= wr_mem0;
		else
			wr_state <= wr_next_state;
	
	// write state transition and output logic
	always@( * )
	begin
		wr0 = 1'b0;
		wr1wait = 1'b0;
		wr1 = 1'b0;
		wr0wait = 1'b0;
		wr_next_state = wr_state;
		
		case( wr_state )
			
			wr_mem0: begin
				wr0 = 1'b1;
				if( ~rd1wait_wr & wr_complete )
					wr_next_state = wr_wait1;
			end
			
			wr_wait1: begin
				wr1wait = 1'b1;
				if( ~rd1_wr )
					wr_next_state = wr_mem1;
			end
			
			wr_mem1: begin
				wr1 = 1'b1;
				if( ~rd0wait_wr & wr_complete )
					wr_next_state = wr_wait0;
			end
			
			wr_wait0: begin
				wr0wait = 1'b1;
				if( ~rd0_wr )
					wr_next_state = wr_mem0;
			end
			
			default: begin
				wr_next_state = wr_default;
			end
			
		endcase
	end
	

//-----------------------------------------------------------------------------
// read side sync state machine
//-----------------------------------------------------------------------------
		
	// read state register
	always@( posedge rd_clk, posedge arst )
		if( arst )
			rd_state <= rd_wait0;
		else
			rd_state <= rd_next_state;
	
	// read state transition and output logic
	always@( * )
	begin
		rd0wait = 1'b0;
		rd0 = 1'b0;
		rd1wait = 1'b0;
		rd1 = 1'b0;
		rd_next_state = rd_state;
		
		case( rd_state )
		
			rd_wait0: begin
				rd0wait = 1'b1;
				if( ~wr0_rd )
					rd_next_state = rd_mem0;
			end
			
			rd_mem0: begin
				rd0 = 1'b1;
				if( ~wr1wait_rd & rd_complete )
					rd_next_state = rd_wait1;
			end
			
			rd_wait1: begin
				rd1wait = 1'b1;
				if( ~wr1_rd )
					rd_next_state = rd_mem1;
			end
			
			rd_mem1: begin
				rd1 = 1'b1;
				if( ~wr0wait_rd & rd_complete )
					rd_next_state = rd_wait0;
			end
			
			default: rd_next_state = rd_default;
			
		endcase
	end
		
endmodule