// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
//-----------------------------------------------------------------------------
//	SPI Interface
//

/****************************************************
			REGISTERS FOR AD7760_62_63
****************************************************/

//  Control Register 1. 
//  Address 0x0001 ; Default Value 0x001A
//
//  --------------------------------------------------
//  Control Register 1 (CR1): {address[5:0],data[9:0]}
//  --------------------------------------------------

	// Frame[9:0] 10 bits for Control Register 1
	// If Frame[11] == 1 => Send Control Register 1

/*---------------------------------------------------------
 D9  D8   D7   D6    D5   D4    D3   D2  D1  D0             
------------------------------------------------	
SYNCFLEN3FLEN2FLEN1FLEN0nBYPF3nBYPF1DEC2DEC1DEC0
---------------------------------------------------------*/

	// The header of the frame is 16'b1111_X0XX_XXXX_XXXX


//  Control Register 2. 
//  Address 0x0002 ; Default Value 0x009B
//
//  --------------------------------------------------
//  Control Register 2 (CR2): {address[5:0],data[9:0]}
//  --------------------------------------------------

	// Frame[5:0] 5 bits for Control Register 2
	// If Frame[11] == 1 => Send Control Register 2

			/*--------------------------
			  D5 D4 D3  D2 D1 D0 
			---------------------
			nCDIV0  PD LPWR 1D1PD
			--------------------------*/

	// The header of the frame is 16'b1111_X000_00X0_XX1X
	
	/*
	THE FRAME[10] INDICATES IF REQUIRED CR1 OR CR2
	FRAME[10] = 0 -> CR2 
	FRAME[10] = 1 -> CR1 
	*/
	

/****************************************************
			REGISTERS FOR AD7764
****************************************************/
// Only one register to SPI

/*

	Dec Rate is encoded D15-D14
	Address is encoded D13-D10:  "0001"  
	PD Mode is D3
	LWR Mode is D2
	Normal Mode is D3 and D2 = 0
	 
	
	
	-------------------------------
	 P. Mode  Dec  SPI Instruction  
	-------------------------------
	  Normal   64   	  0x8400      
	-------------------------------
	  Normal  128   	  0xC400      
	-------------------------------
	  Normal  256   	  0x0400            
	-------------------------------
	  Lower    64   	  0x8404      
	-------------------------------
	  Lower   128   	  0xC404      
	-------------------------------
	  Lower   256   	  0x0404      
	-------------------------------
	  P Down   64   	  0x8408      
	-------------------------------
	  P Down  128   	  0xC408      
	-------------------------------
	  P Down  256   	  0x0408       
	-------------------------------

Sent from LabVIEW the next Frame:

D15-D12: 1111
D11: 	 1
D10: 	 X  (AD7760_62_63)
D9-D8:	 11 (AD7760_62_63) or 00 (AD7764_65)
D7-D6:   Power Mode
D0-D1:   Dec_Rate


*/
/****************************************************
			REGISTERS FOR AD7765
****************************************************/
// Only one register to SPI

/*

	Dec Rate is encoded D15-D14
	Address is encoded D13-D10 "0001"
	PD Mode is D3
	LWR Mode is D2
	Normal Mode is D3 and D2 = 0
	 
	
	
	-------------------------------
	 P. Mode  Dec  SPI Instruction  
	-------------------------------
	  Normal  128   	  0x8400      
	-------------------------------
	  Normal  256   	  0xC400            
	-------------------------------
	  Lower   128   	  0x8404      
	-------------------------------
	  Lower   256   	  0xC404      
	-------------------------------
	  P Down  128   	  0x8408      
	-------------------------------
	  P Down  256   	  0xC408       
	-------------------------------
	
Sent from LabVIEW the next Frame:

D15-D12: 1111
D11: 	 1
D10: 	 X  (AD7760_62_63)
D9-D8:	 11 (AD7760_62_63) or 00 (AD7764_65)
D7-D6:   Power Mode
D0-D1:   Dec_Rate

	
*/

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------


module spi 
(

	input arst,
	
	input spi_clk,
	
	// SPI signals
	
	// Data input from CPLD
	output reg SPI_SEL1,
	
	// Serial Data
	output reg SPI_MOSI,
	
	// Serial Clock
	output SPI_SCK,
	
	// Communication with Parallel block
	input spi_enable,
	input [15:0] spi_register
	
);

	parameter pCyclesW = 	16;
		
		
	reg [15:0] 	frame;
	reg 		enable_clock;
	reg	[4:0]	count;
	reg         valid;
	
	wire		enable;
	
	
	assign SPI_SCK = spi_clk;
	
	
	synchronizer spi_synchronized 
	(
		.clk 		( spi_clk ),	
		.in_signal  ( spi_enable ),
		.out_signal ( enable )
	);
	
	
	always@( posedge spi_clk, posedge arst )
	begin
		
		if ( arst )
		begin
			enable_clock <= 1'b0;
			count <= 5'b0;
			SPI_MOSI <= 1'b1;
			SPI_SEL1 <= 1'b1;
			frame <= 16'b0;
			valid <= 1'b0;
		end
		else 
		begin
			
			if ( enable )					
			begin
				valid <= 1'b1;
				
				// Which ADC ?
				
				if ( spi_register[9:8] == 2'b00 )
				begin
					// AD7764_65
					
					frame[15:14] <= spi_register[1:0];      // Dec Rate
					frame[13:10] <= 4'b0001;                // Address
					frame[9:4] <= 6'b0;
					frame[3:2] <= spi_register[7:6];        // Address
					frame[1:0] <= 2'b0;					
					
				end
				else if ( spi_register[9:8] == 2'b11 )
				begin
					// AD7760_62_63
					
					// Write something... CR1
					// frame = Address(6 bits), c_to_io_D[9:0]
					// Write something... CR2
					// frame = Address(6 bits), 0000, c_to_io_D[5:0]
				
					if ( spi_register[10] )
						frame <= {6'b000001,2'b00,spi_register[7:0]};
					else 
						frame <= {6'b000010,4'b0000,spi_register[5:0]};
				end
			end
				
			if ( valid )
			begin
			
				if ( count == pCyclesW )
				begin
					SPI_MOSI <= 1'b1;
					enable_clock <= 1'b0;
					count <= 5'h0;							
					SPI_SEL1 <= 1'b1;
					valid <= 1'b0;			
				end
				else
				begin
					count <= count + 5'b00001;
					SPI_MOSI <= frame[pCyclesW-1-count];
					SPI_SEL1 <= 1'b0;
					enable_clock <= 1'b1;	
				end			
			
			end
		end
			
	end
		
		
	
endmodule	
