// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
//-----------------------------------------------------------------------------
// code format
//               
//   CED config
//                            usb_cmd[3:0]    usb_to_c_D[15:0]      hex
//      ADT write                     xxx1 0AAA AAAA DDDD DDDD      aa**
//      POT write           VARD      xxx1 1x01 xx00 DDDD DDDD      90**
//                          VARA      xxx1 1x01 xx01 DDDD DDDD      91**
//                          VIO       xxx1 1x01 xx10 DDDD DDDD      92**
//							3_3       xxx1 1x01 xx11 DDDD DDDD      93**
//      Power Supplies en             xxx1 1x10 xxxx DDDD DDDD      A0**
//                          12/15 SEL _______________|||| ||||
//                          -5VA      ________________||| ||||
//                          +-15VA    _________________|| ||||
//                          +5VA      __________________| ||||
//                                                        ||||
//                          +VARA     ____________________||||
//                          +VARD     _____________________|||
//                          +5VD      ______________________||
//                          +3VD      _______________________|
//
//      end                           xxx1 1x00 xxxx xxxx xxxx      8xxx
//                                                                       
//   CED status
//                            usb_cmd[3:0]    usb_to_c_D[15:0]    
//      ADT read                      xxx0 0AAA AAAA xxxx xxxx      aaxx
//      POT read            VARD      xxx0 1x01 xx00 xxxx xxxx      90xx
//                          VARA      xxx0 1x01 xx01 xxxx xxxx      91xx
//                          VIO       xxx0 1x01 xx10 xxxx xxxx      92xx
//							3_3       xxx0 1x01 xx11 xxxx xxxx      93xx
//
//      end                           xxx0 1x00 xxxx xxxx xxxx      8xxx
//
// Note: A is an address bit, D is a data bit and x is don't care
//-----------------------------------------------------------------------------
`define config_n_status       ( usb_cmd )
`define i2c                   ( ~usb_to_c_D[15] | usb_to_c_D[12] )
`define power_en              ( usb_to_c_D[13] )
`define adt7411_n_ad5282      ( ~usb_to_c_D[15] )
`define adt7411_part_address  ( 7'b1001010 )
`define adt7411_reg_address   { 1'b0, usb_to_c_D[14:8] }
`define ad5282_part_address   { 5'b01011, ~usb_to_c_D[9], usb_to_c_D[9] }
`define ad5282_reg_address    { usb_to_c_D[8], 7'b000_0000 }

// I2C controller register definations
`define I2CSTA 2'b00 // Status
`define I2CTO  2'b00 // Time-out
`define I2CDAT 2'b01 // Data
`define I2CADR 2'b10 // Own Address
`define I2CCON 2'b11 // Control

module power_controller
(
	input arst,
	input clk,
	input clk_en_2M5,
	
	// fpga side
	input en,
	input usb_cmd,
	//input usb_to_c_start,
	input usb_to_c_stop,
	input usb_to_c_cts,
	input usb_to_c_rts,
	input [15:0] usb_to_c_D,
	output c_to_usb_busy,
	output reg c_to_usb_stop,
	output c_to_usb_cts,
	output c_to_usb_rts,
	output reg [15:0] c_to_usb_D,
	// board side
	output reg PWR_EN_REG_CLK,
	output nPWR_I2C_CS,
	output reg nPWR_I2C_RD,
	output reg nPWR_I2C_WR,
	output reg [1:0] PWR_I2C_A,
	inout [7:0] PWR_D
);

	// fpga side state definations ( fs is Fpga State )
	parameter
		fs_decision                = 4'b0000,
		fs_usb_stop                = 4'b0001,
		fs_power_en                = 4'b0010,
		fs_i2c_start               = 4'b0011,
		fs_i2c_tx_part_address     = 4'b0100,
		fs_i2c_tx_reg_address      = 4'b0101,
		fs_i2c_wr                  = 4'b0110,
		fs_i2c_rd_start            = 4'b0111,
		fs_i2c_rd_tx_part_address  = 4'b1000,
		fs_i2c_rd                  = 4'b1001,
		fs_i2c_stop                = 4'b1010,
		fs_data_ack                = 4'b1011,
		fs_default                 = 4'bxxxx;
	
	// board side state definations ( bs is Board State )
	parameter
		bs_reset            = 5'b00000,
		bs_init             = 5'b00001,
		bs_init2            = 5'b00010,
		bs_decision         = 5'b00011,
		bs_power_en         = 5'b00100,
		bs_power_en2        = 5'b00101,
		bs_i2c_start        = 5'b00110,
		bs_i2c_start2       = 5'b00111,
		bs_i2c_tx           = 5'b01000,
		bs_i2c_tx2          = 5'b01001,
		bs_i2c_tx3          = 5'b01010,
		bs_i2c_tx4          = 5'b01011,
		bs_i2c_tx5          = 5'b01100,
		bs_i2c_rx           = 5'b01101,
		bs_i2c_rx2          = 5'b01110,
		bs_i2c_rx3          = 5'b01111,
		bs_i2c_rx4          = 5'b10000,
		bs_i2c_rx5          = 5'b10001,
		bs_i2c_stop         = 5'b10010,
		bs_i2c_stop2        = 5'b10011,
		bs_i2c_stop3        = 5'b10100,
		bs_i2c_wait_for_si  = 5'b10101,
		bs_command_compleat = 5'b10110,
		bs_default          = 5'bxxxxx;

	// state variables
	reg [3:0] fpga_state;
	reg [3:0] fpga_next_state;
	reg [4:0] board_state;
	reg [4:0] board_next_state;

	// signals from fpga side state machine to board side state machine
	reg power_supply_en_flipflop;
	reg i2c_start;
	reg i2c_tx;
	reg i2c_rx;
	reg i2c_stop;
	reg [7:0] internal_data;
	
	// signals for board side state machine to fpga side state machine
	reg command_compleat;
	
	// fpga side state machine outputs
	reg c_to_usb_cts_2M5;
	reg c_to_usb_rts_2M5;
	
	// Data 
	reg Dout_en;
	reg [7:0] Dout, Din;
	
	// tri state buffer for bidirectional data bus
	wire [7:0] Din_wire = Dout_en ? 8'bxxxx_xxxx : PWR_D;
	assign PWR_D = Dout_en ? Dout : 8'bzzzz_zzzz;
	
	// sync rx and tx enables with core clk
	assign c_to_usb_cts = clk_en_2M5 & c_to_usb_cts_2M5;
	assign c_to_usb_rts = clk_en_2M5 & c_to_usb_rts_2M5;
	
	// i2c controler chip select
	assign nPWR_I2C_CS = nPWR_I2C_WR & nPWR_I2C_RD;
	
	// c_to_usb_busy not used
	assign c_to_usb_busy = 1'b0;
	
	// data in register to make data synchronous
	always@( posedge clk, posedge arst )
		if( arst )
			Din <= 8'b00000000;
		else
			Din <= Din_wire;
	
//-----------------------------------------------------------------------------
// fpga side state machine
//-----------------------------------------------------------------------------
	
	// fpga state register
	always@( posedge clk, posedge arst )
		if( arst )
			fpga_state <= fs_decision;
		else if( clk_en_2M5 | usb_to_c_stop )
			fpga_state <= fpga_next_state;

	// fpga state transition and output logic
	always@( * )
	begin
	
		c_to_usb_cts_2M5 = 1'b0;
		c_to_usb_stop = 1'b0;
		power_supply_en_flipflop = 1'b0;
		i2c_start = 1'b0;
		i2c_tx = 1'b0;
		i2c_rx = 1'b0;
		i2c_stop = 1'b0;
		internal_data = 8'bxxxx_xxxx;
		
		case( fpga_state )
		
			// decision and usb_stop states -----------------------------------
						
			fs_decision: begin
				if( en & usb_to_c_rts & usb_to_c_cts )
					if( `i2c )
						fpga_next_state = fs_i2c_start;
					else if( `power_en )
						fpga_next_state = fs_power_en;
					else
						fpga_next_state = fs_usb_stop;
				else
					if( usb_to_c_stop )
						fpga_next_state = fs_usb_stop;
					else
						fpga_next_state = fs_decision;	
			end
			
			fs_usb_stop: begin
				c_to_usb_stop = 1'b1;
				if( usb_to_c_stop )
					fpga_next_state = fs_decision;
				else
					fpga_next_state = fs_usb_stop;
			end
			
			// power supplies enbles ------------------------------------------
			fs_power_en: begin
				power_supply_en_flipflop = 1'b1;
				internal_data = usb_to_c_D[7:0];
				if( command_compleat )
					fpga_next_state = fs_data_ack;
				else
					fpga_next_state = fs_power_en;
			end
			
			// i2c start ------------------------------------------------------
			fs_i2c_start: begin
				i2c_start = 1'b1;
				if( command_compleat )
					fpga_next_state = fs_i2c_tx_part_address;
				else
					fpga_next_state = fs_i2c_start;
			end
			
			fs_i2c_tx_part_address: begin
				i2c_tx = 1'b1;
				if( `adt7411_n_ad5282 )
					internal_data = { `adt7411_part_address, 1'b0 };
				else
					internal_data = { `ad5282_part_address, 1'b0 };
					
				if( command_compleat )
					fpga_next_state = fs_i2c_tx_reg_address;
				else
					fpga_next_state = fs_i2c_tx_part_address;
			end
			
			fs_i2c_tx_reg_address: begin
				i2c_tx = 1'b1;
				if( `adt7411_n_ad5282 )
					internal_data = `adt7411_reg_address;
				else
					internal_data = `ad5282_reg_address;
					
				if( command_compleat )
					if( `config_n_status )
						fpga_next_state = fs_i2c_wr;
					else
						fpga_next_state = fs_i2c_rd_start;
				else
					fpga_next_state = fs_i2c_tx_reg_address;
			end
			
			// i2c write ------------------------------------------------------
			fs_i2c_wr: begin
				i2c_tx = 1'b1;
				internal_data = usb_to_c_D[7:0];
				if( command_compleat )
					fpga_next_state = fs_i2c_stop;
				else
					fpga_next_state = fs_i2c_wr;
			end
			
			// i2c read -------------------------------------------------------
			fs_i2c_rd_start: begin
				i2c_start = 1'b1;
				if( command_compleat )
					fpga_next_state = fs_i2c_rd_tx_part_address;
				else
					fpga_next_state = fs_i2c_rd_start;
			end
			
			fs_i2c_rd_tx_part_address: begin
				i2c_tx = 1'b1;
				if( `adt7411_n_ad5282 )
					internal_data = { `adt7411_part_address, 1'b1 };
				else
					internal_data = { `ad5282_part_address, 1'b1 };
					
				if( command_compleat )
					fpga_next_state = fs_i2c_rd;
				else
					fpga_next_state = fs_i2c_rd_tx_part_address;
			end
			
			fs_i2c_rd: begin
				i2c_rx = 1'b1;
				internal_data = usb_to_c_D[15:8];
				if( command_compleat )
					fpga_next_state = fs_i2c_stop;
				else
					fpga_next_state = fs_i2c_rd;
			end
			
			// i2c stop -------------------------------------------------------
			fs_i2c_stop: begin
				i2c_stop = 1'b1;
				if( command_compleat )
					fpga_next_state = fs_data_ack;
				else
					fpga_next_state = fs_i2c_stop;
			end
			
			// data acknolage -------------------------------------------------
			fs_data_ack: begin
				c_to_usb_cts_2M5 = 1'b1;
				fpga_next_state = fs_decision;
			end
			
			//-----------------------------------------------------------------
			
			default: fpga_next_state = fs_default;
			
		endcase	
	end
	
//-----------------------------------------------------------------------------	
// board side state machine
//-----------------------------------------------------------------------------

// I2CCON   _______________________________________________
//         | AA | ENSIO | STA | STO | SI | CR2 | CR1 | CR0 |
	
	// board state register
	always@( posedge clk, posedge arst )
		if( arst )
			board_state <= bs_reset;
		else if( clk_en_2M5 )
			board_state <= board_next_state;
	
	// board state transition and output logic
	always@( * )
	begin
	
		c_to_usb_D = 16'bxxxx_xxxx_xxxx_xxxx;
		c_to_usb_rts_2M5 = 1'b0;
		PWR_EN_REG_CLK = 1'b1;
		nPWR_I2C_RD = 1'b1;
		nPWR_I2C_WR = 1'b1;
		PWR_I2C_A = `I2CCON;  //note I2CCON is default address
		Dout_en = 1'b0;
		Dout = 8'bxxxx_xxxx;
		
		case( board_state )
			
			// init i2c controler ---------------------------------------------
			
			bs_reset: begin
				command_compleat = 1'b0;
				board_next_state = bs_init;
			end
			
			// set ENSIO bit in I2CCON reg to enable internal oscillator
			bs_init: begin
				command_compleat = 1'b0;
				nPWR_I2C_WR = 0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_init2;
			end
			
			bs_init2: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_decision;
			end
			// might need to add 500us delay after init ?!
			
			// decision state -------------------------------------------------
			
			bs_decision: begin
				command_compleat = 1'b0;
				if( power_supply_en_flipflop )
					board_next_state = bs_power_en;
				else if( i2c_start )
					board_next_state = bs_i2c_start;
				else if( i2c_tx )
					board_next_state = bs_i2c_tx;
				else if( i2c_rx )
					board_next_state = bs_i2c_rx;
				else if( i2c_stop )
					board_next_state = bs_i2c_stop;
				else
					board_next_state = bs_decision;
			end
			
			// power supply enables flipflop write ----------------------------
			
			// output data and bring flipflop clk low
			bs_power_en: begin
				command_compleat = 1'b0;
				PWR_EN_REG_CLK = 1'b0;
				Dout_en = 1'b1;
				Dout = internal_data;
				board_next_state = bs_power_en2;
			end
			
			// keep data valid and bring flipflop clk high to clk in data
			bs_power_en2: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = internal_data;
				board_next_state = bs_command_compleat;
			end
			
			// i2c_start ------------------------------------------------------
			
			// set STA bit in I2CCON reg
			bs_i2c_start: begin
				command_compleat = 1'b0;
				nPWR_I2C_WR = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h60;
				board_next_state = bs_i2c_start2;
			end
			
			bs_i2c_start2: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h60;
				board_next_state = bs_i2c_wait_for_si;
			end
								
			// i2c_tx ---------------------------------------------------------
			
			// write data to I2CDAT reg
			bs_i2c_tx: begin
				command_compleat = 1'b0;
				PWR_I2C_A = `I2CDAT;
				board_next_state = bs_i2c_tx2;
			end
			
			bs_i2c_tx2: begin
				command_compleat = 1'b0;
				PWR_I2C_A = `I2CDAT;
				nPWR_I2C_WR = 1'b0;
				Dout_en = 1'b1;
				Dout = internal_data;
				board_next_state = bs_i2c_tx3;
			end
			
			bs_i2c_tx3: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = internal_data;
				board_next_state = bs_i2c_tx4;
			end
			
			// clear SI bit in I2CCON reg
			bs_i2c_tx4: begin
				command_compleat = 1'b0;
				nPWR_I2C_WR = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_i2c_tx5;
			end
			
			bs_i2c_tx5: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_i2c_wait_for_si;
			end
			
			// i2c_rx ---------------------------------------------------------
			
			// clear SI bit in I2CCON reg leave AA low (give on ack)
			bs_i2c_rx: begin
				command_compleat = 1'b0;
				nPWR_I2C_WR = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_i2c_rx2;
			end
			
			bs_i2c_rx2: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h40;
				board_next_state = bs_i2c_rx3;
			end			
			
			// wait until SI flag in I2CCON reg is set (wait for data)
			bs_i2c_rx3: begin
				command_compleat = 1'b0;
				nPWR_I2C_RD = 1'b0;
				if( Din[3] )
					board_next_state = bs_i2c_rx4;
				else
					board_next_state = bs_i2c_rx3;
			end
			
			// read data reg to c_to_usb_D
			bs_i2c_rx4: begin
				command_compleat = 1'b0;
				PWR_I2C_A = `I2CDAT;
				board_next_state = bs_i2c_rx5;
			end
			
			bs_i2c_rx5: begin
				command_compleat = 1'b0;
				PWR_I2C_A = `I2CDAT;
				nPWR_I2C_RD = 1'b0;
				c_to_usb_rts_2M5 = 1'b1;
				c_to_usb_D = { internal_data, Din };
				board_next_state = bs_command_compleat;
			end
			
			// i2c_stop -------------------------------------------------------
			
			// set STO bit in I2CCON reg
			bs_i2c_stop: begin
				command_compleat = 1'b0;
				nPWR_I2C_WR = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h50;
				board_next_state = bs_i2c_stop2;
			end
			
			bs_i2c_stop2: begin
				command_compleat = 1'b0;
				Dout_en = 1'b1;
				Dout = 8'h50;
				board_next_state = bs_i2c_stop3;
			end
			
			bs_i2c_stop3: begin
				command_compleat = 1'b0;
				nPWR_I2C_RD = 1'b0;
				if( ~Din[4] )
					board_next_state = bs_command_compleat;
				else
					board_next_state = bs_i2c_stop3;
			end
			
			// wait for SI flag and command compleat --------------------------
			
			// wait until SI flag in I2CCON reg is set
			bs_i2c_wait_for_si: begin
				command_compleat = 1'b0;
				nPWR_I2C_RD = 1'b0;
				if( Din[3] )
					board_next_state = bs_command_compleat;
				else
					board_next_state = bs_i2c_wait_for_si;
			end
			
			// command compleat
			bs_command_compleat: begin
				command_compleat = 1'b1;
				board_next_state = bs_decision;
			end
			
			//-----------------------------------------------------------------
			
			default: begin
				command_compleat = 1'b0;
				board_next_state = bs_default;
			end
			
		endcase
	end
	
endmodule		