// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
//-----------------------------------------------------------------------------
//	Parallel Interface
//
//  Status register when reading DATA
//
	/*--------------------------------------------------------
	   D7     D6    D5     D4     D3     D2   D1  D0    
	---------------------------------------------------
	 DVALID  OVR   UFILT  LPWR  FILTOK  DLOK   0   0 
	--------------------------------------------------------*/

// Different options from the LabVIEW application
/*
	In function of the value of c_to_io_D:	
	
	16'hEFFE	RESTART
	16'hFEEF	MODULATOR
	16'hF0XX	USER FILTER
	16'hFFXX	WRITE CR1	
	16'hFBXX	WRITE CR2
	16'hFC03	READ OVERRANGE
	16'hFC05	READ GAIN
	16'hFC07	READ OFFSET
	16'hFA03	WRITE OVERRANGE
	16'hFA05	WRITE GAIN
	16'hFA07	WRITE OFFSET
	
	
*/		
//-----------------------------------------------------------------------------
module parallel_new
 (
	input arst,
	input clk,
	
	input [7:0] usb_cmd,
	
	input c_to_io_start,
	input c_to_io_stop,
	input c_to_io_cts,
	input c_to_io_rts,
	input [15:0] c_to_io_D,
		
	output io_to_c_busy,
	output reg io_to_c_stop,
	output io_to_c_cts,
	output reg io_to_c_rts,
	output reg [15:0] io_to_c_D,
	
	input mclk,
	output reg nRESET,
	output reg nCS,
	output reg nRD_WR,
	output reg nSYNC,
	input nDRDY,
	
	inout [15:0] PARD,
		
	input [21:0] n_samples
	
);


	parameter pDataW = 	16;
	
	parameter filter_coeffs = 48;
	
	parameter
		wait_for_start           = 3'b000,
		wait_for_end_of_config   = 3'b001,
		wait_for_sdata_valid     = 3'b010,
		send_a                   = 3'b011,
		stop                     = 3'b100;


	parameter
		Receive_CMDS		= 4'b0000,
		Wait_For_Reset		= 4'b0001,
		Wait_After_Reset	= 4'b0010,
		Write_Register      = 4'b0011,
		Wait_6_Cycles		= 4'b0100,
		Wait_For_Sync       = 4'b0101,
		User_Filter			= 4'b0110,
		Wait_After_Filter	= 4'b0111,
		Modulator_Interface	= 4'b1000,
		Normal_Interface	= 4'b1001,
		Calculate_Timing	= 4'b1010,
		Read_Register		= 4'b1011;

		
	reg                    sdata_valid;
	reg                    count_clr;
	reg [pDataW-1:0]       data_a;
	reg [21:0]             count;
	reg [2:0]              state;
	reg [2:0]              next_state;
	reg [31:0]         	   data;		
	reg [31:0]         	   data_read;	
	reg [2:0] 			   state_dato;
	reg [3:0]			   process;
	reg [3:0]			   last_process;
	reg [15:0]			   write_register_CR1;	
	reg [15:0]			   write_register_CR2;
	reg                    data_enable;
	reg [7:0]			   aux_count;
	reg 				   receive;	
	reg					   output_enable;
	reg [15:0]			   data_out;	
	reg					   downloading_filter;
	reg [7:0]			   length;	
	reg [7:0]			   count_send;
	reg  				   filter_c;
	reg					   count_clr_last;
	reg 				   power_up;
	wire				   data_enable_out;
	reg					   modulator;   
	reg 				   nDRDY_last;
	reg					   read_regs;	
	reg [15:0]			   address; 
	reg [15:0] 			   value;		
	reg 				   value_flag;
	reg					   write_regs;
	reg 				   close1;	
	reg [15:0]			   data_valid;
	reg 				   contar;
	reg					   calculate;
	reg [21:0]             samples;
	
	wire receive_out;	
	
	parameter
		send_dato1        = 3'b000,
		send_nodato1      = 3'b001,
		send_wait_dato2   = 3'b010,
		send_dato2        = 3'b011,
		send_nodato2      = 3'b100,
		state_init        = 3'b101;
	
	

	assign io_to_c_cts = c_to_io_rts;
	
	assign PARD = output_enable ? data_out : 16'hzzzz;
	
	
	synchronizer sync
	(
		.clk		( clk ),
		.in_signal	( data_enable ),
		.out_signal	( data_enable_out )
	);

	synchronizer sync2
	(
		.clk		( mclk ),
		.in_signal	( receive ),
		.out_signal	( receive_out )
	);
	
	always@( posedge mclk, posedge arst )
	begin
	
		if ( arst )
		begin
			// Output signals
			nRESET <= 1'b0;
			nCS <= 1'b1;
			nRD_WR <= 1'b1;
			nSYNC <= 1'b1;
			
			// State
			process <= Wait_For_Reset;
			last_process <= Wait_For_Reset;
			
			// Control registers
			power_up <= 1'b1;
			downloading_filter <= 1'b0;
			modulator <= 1'b0;   			
			read_regs <= 1'b0;
			write_regs <= 1'b0;
			close1 <= 1'b0;				
							
			data_enable <= 1'b0;
			aux_count <= 8'd0;
			output_enable <= 1'b0;
			data_out <= 16'h0;	
			length <= 8'b0;	
			count_send <= 8'b0;
			filter_c <= 1'b0;
			nDRDY_last <= 1'b1;
			
			address <= 16'h0000; 
			data_valid <= 16'h0000;  	// Only enable for a timing issue
	    	contar <= 1'b0;				// Only enable for a timing issue
			calculate <= 1'b0;			// Only enable for a timing issue
			data <= 32'h00000000;		
			data_read <= 32'h00000000;	
			write_register_CR1 <= 16'h001A;	
			write_register_CR2 <= 16'h0002;
			
			samples <= 22'b0;
				
		end
		else
		begin
			
			nDRDY_last <= nDRDY;	
			
			if ( contar )
				data_valid <= data_valid + 16'b1;
			
		
			case ( process )
			
				/**************************************
							RECEIVE COMMANDS
				**************************************/
				Receive_CMDS:begin
				
					if ( (c_to_io_D == 16'hAFFA) )			
					begin
					
							last_process <= Receive_CMDS;
						
							nRESET <= 1'b0;
							nCS <= 1'b1;
							nRD_WR <= 1'b1;
							nSYNC <= 1'b1;
													
							aux_count <= 8'd0;
							
							process <= Wait_For_Reset;
							last_process <= Wait_For_Reset;
							
							aux_count <= 8'b0;
																
							data <= 32'h0;
							
							
							output_enable <= 1'b0;	
							data_out <= 16'h0000;	
														
							downloading_filter <= 1'b0;
							length <= 4'b0;
							
							count_send <= 8'b0;
							
							filter_c <= 1'b0;
							modulator <= 1'b0;
							read_regs <= 1'b0;
							
							address <= 16'h0000;
							close1 <= 1'b0;
							data_valid <= 16'h0;
							
							contar <= 1'b0;
							power_up <= 1'b1;	
							calculate <= 1'b1;
					end
					else if ( (c_to_io_D == 16'hEFFE) )			// RESTART
					begin
						process <= Wait_For_Reset;
						last_process <= Wait_For_Reset;
						
						write_register_CR1 <= 16'h1A;
						write_register_CR2 <= 16'h02;
						
						nRESET <= 1'b0;
						nCS <= 1'b1;
						nRD_WR <= 1'b1;
						nSYNC <= 1'b1;
			
						// Control registers
						power_up <= 1'b1;
						downloading_filter <= 1'b0;
						modulator <= 1'b0;   			
						read_regs <= 1'b0;
						write_regs <= 1'b0;
						close1 <= 1'b0;				
													
						data_enable <= 1'b0;
						aux_count <= 8'd0;
						output_enable <= 1'b0;
						data_out <= 16'h0;	
						length <= 8'b0;	
						count_send <= 8'b0;
						filter_c <= 1'b0;
						nDRDY_last <= 1'b1;
						
						address <= 16'h0000; 
						data_valid <= 16'h0000;  	// Only enable for a timing issue
						contar <= 1'b0;				// Only enable for a timing issue
						calculate <= 1'b0;			// Only enable for a timing issue
						data <= 32'h00000000;		
						
					end
					else if ( c_to_io_D == 16'hFEEF )			// MODULATOR
					begin
						nRD_WR <= 1'b1;						
						nCS <= 1'b1;
						modulator <= 1'b1;						
						power_up <= 1'b1;				
						write_register_CR1 <= 16'h0000;			// write register 1					
						process <= Wait_For_Reset;				
						last_process <= Receive_CMDS;
					end
					else if ( c_to_io_D[15:8] == 8'hFA )		// WRITE REGISTER 3,4,5
					begin
						last_process <= Receive_CMDS;
						process <= Write_Register;
						write_regs <= 1'b1;
												
						if ( c_to_io_D[7:0] == 8'h03 )			// WR_OVR
							address <= 16'h0005;						
						else if ( c_to_io_D[7:0] == 8'h05 )		// WR_GAIN
							address <= 16'h0004;						
						else if ( c_to_io_D[7:0] == 8'h07 )		// WR_OFF
							address <= 16'h0003;					
						
					end
					else if ( c_to_io_D[15:8] == 8'hFC )		// READ REGISTER 3,4,5
					begin
						last_process <= Receive_CMDS;
						read_regs <= 1'b1;
						process <= Write_Register;
						address <= 16'h0001;                    
																	
						if ( c_to_io_D[7:0] == 8'h03 )			// RD_OVR
							write_register_CR1 <= {write_register_CR1[15],5'b10000,write_register_CR1[9:0]};
						else if ( c_to_io_D[7:0] == 8'h05 )		// RD_GAIN
							write_register_CR1 <= {write_register_CR1[15],5'b01000,write_register_CR1[9:0]};
						else if ( c_to_io_D[7:0] == 8'h07 )		// RD_OFF
							write_register_CR1 <= {write_register_CR1[15],5'b00100,write_register_CR1[9:0]};
						
					end
					else if ( c_to_io_D[15:8] == 8'hFF )		// WRITE REGISTER 1
					begin
						write_register_CR1 <= {11'b0,c_to_io_D[4:0]};
						process <= Write_Register;
						last_process <= Receive_CMDS;
						address <= 16'h0001;	
					end
					else if ( c_to_io_D[15:8] == 8'hFB )		// WRITE REGISTER 2
					begin
						write_register_CR2 <= {10'b0,c_to_io_D[5:0]};
						process <= Write_Register;	
						address <= 16'h0002;
						last_process <= Receive_CMDS;
					end
					else if ( c_to_io_D[15:8] == 8'hF0 )		// FILTER
					begin
											
						data_enable <= 1'b0;
																		
						downloading_filter <= 1'b1;
						write_register_CR1 <= {1'b1,6'b0,c_to_io_D[3:0],1'b1,c_to_io_D[7:4]}; 
						process <= Write_Register;
						last_process <= Receive_CMDS;
						address <= 16'h0001;
												
						case ( c_to_io_D[3:0] )
					
							4'd1:begin
								length <= 8'd12;   
							end
							4'd3:begin
								length <= 8'd24;   
							end
							4'd5:begin
								length <= 8'd36;
							end
							4'd7:begin
								length <= 8'd48;
							end
							4'd9:begin
								length <= 8'd60;
							end
							4'd11:begin
								length <= 8'd72;
							end
							4'd13:begin
								length <= 8'd84;
							end
							4'd15:begin
								length <= 8'd96;
							end
							
							
						endcase		
					
					end
					else if ( receive_out )						// GET SAMPLES
					begin
						close1 <= 1'b1;
						last_process <= Receive_CMDS;
						
						samples <= 22'b0;
						
						if ( last_process == Normal_Interface )
							process <= Normal_Interface;
						else if ( last_process == Modulator_Interface )
							process <= Modulator_Interface;
						else if ( last_process == Write_Register )
							process <= Normal_Interface;						
						else if ( (last_process == Wait_For_Sync) )
						begin
														
							if ( modulator )
								process <= Modulator_Interface;
							else
								process <= Normal_Interface;
							
						end
						
					
					end
				
				end
				
				/**************************************
							WAIT FOR RESET
				**************************************/
				Wait_For_Reset:begin
					// Reset for one cycle (min)
					if ( aux_count < 8'd3 ) 
					begin
						aux_count <= aux_count + 8'd1;
						nRESET <= 1'b0;
					end
					else
					begin
						process <= Wait_After_Reset;
						nRESET <= 1'b1;
						aux_count <= 8'b0;
					end	
				end
				
				/**************************************
							WAIT AFTER RESET
				**************************************/
				Wait_After_Reset:begin
					// Wait for 2 cycles
					if ( aux_count < 8'd2 )   
						aux_count <= aux_count + 8'd1;
					else
					begin
						process <= Write_Register;
						last_process <= Wait_After_Reset;
						address <= 16'h0002;
						aux_count <= 8'b0;
					end				
				end
					
				/**************************************
							WRITE REGISTER
				**************************************/
				Write_Register:begin
				
					case ( aux_count )			
				
						6'd0:begin
							
							if ( write_regs )
							begin
								if ( value_flag )
								begin	
									nCS <= 1'b0;
									nRD_WR <= 1'b1;
									output_enable <= 1'b1;
									data_out <=	address;     							
									aux_count <= aux_count + 8'd1;
								end
							end
							else
							begin
								nCS <= 1'b0;
								nRD_WR <= 1'b1;
								output_enable <= 1'b1;
								data_out <=	address;     							
								aux_count <= aux_count + 8'd1;
							end
							
						end
						6'd8:begin
							nCS <= 1'b1;
							aux_count <= aux_count + 8'd1;
							output_enable <= 1'b0;
						end
						6'd16:begin
						
							nCS <= 1'b0;
							output_enable <= 1'b1;
							aux_count <= aux_count + 8'd1;
							
							if ( write_regs )
								data_out <= value;
							else if ( address == 16'h0002 ) 
								data_out <= write_register_CR2;
							else	
								data_out <= write_register_CR1;	
														
						end
						6'd24:begin
							nCS <= 1'b1;
							output_enable <= 1'b0;
							aux_count <= 8'b0;
							
							last_process <= Write_Register;
							
							if ( write_regs )				
							begin
								process <= Receive_CMDS;								
								write_regs <= 1'b0;	
							end
							else if ( downloading_filter )
							begin
								count_send <= 7'b0;
								if ( last_process == Wait_After_Filter )
								begin
									process <= Receive_CMDS;
									downloading_filter <= 1'b0;
								end
								else
									process <= User_Filter;
							
							end
							else if ( power_up && (last_process == Write_Register) )  
							begin
								process <= Wait_6_Cycles;									
							end							
							else if ( power_up && (last_process == Wait_After_Reset) )
							begin
								process <= Write_Register;
								address <= 16'h0001;
							end
							else
								process <= Receive_CMDS;								
												
							
						end						
						
						default:begin
							aux_count <= aux_count + 8'd1;
						end
						
				
					endcase	
				
				end
				
				/**************************************
							WAIT 6 CYCLES
				**************************************/
				Wait_6_Cycles:begin
				
					if ( aux_count < 8'd6 )  
						aux_count <= aux_count + 8'd1;
					else
					begin
						process <= Wait_For_Sync;
						aux_count <= 8'b0;
					end
				
				end
				
				/**************************************
							WAIT FOR SYNC
				**************************************/
				Wait_For_Sync:begin
				
					power_up <= 1'b0;
					
					if ( aux_count < 8'd4 )
					begin
						aux_count <= aux_count + 8'd1;
						nSYNC <= 1'b0;
					end
					else
					begin
						aux_count <= 8'b0;
						nSYNC <= 1'b1;
						
						if ( calculate )
						begin
							process <= Calculate_Timing;
							contar <= 1'b1;
							data_valid <= 16'h0;
						end
						else
							process <= Receive_CMDS;
						
						last_process <= Wait_For_Sync;					
					end
				
				end
						
				/**************************************
							  FILTER
				**************************************/
				User_Filter:begin								
					// Send the Length/2 coef
					
					case ( aux_count )
												//0 - 16
						8'd0:begin
							nCS <= 1'b1;
							output_enable <= 1'b0;
							aux_count <= aux_count + 8'd1;
						end
						8'd8:begin
							nCS <= 1'b0;
							output_enable <= 1'b1;
							aux_count <= aux_count + 8'd1;
							count_send <= count_send + 8'b1;
							filter_c <= 1'b1;
														
							data_out <= c_to_io_D;   // first half 	
														
						end
						8'd16:begin
							nCS <= 1'b1;
							output_enable <= 1'b0;
										
							aux_count <= aux_count + 8'd1;							
							
						end
						8'd24:begin
						
							if ( count_send == length + 1 )
							begin
								last_process <= User_Filter;
								process <= Wait_After_Filter; 								
								write_register_CR1 <= {4'b0,1'b1,write_register_CR1[10:0]};								   count_send <= 8'b0;	
								aux_count <= 8'd0;						
											
								
							end
							else
							begin
								nCS <= 1'b0;
								output_enable <= 1'b1;
								aux_count <= aux_count + 8'd1;
							
								filter_c <= 1'b1;
								count_send <= count_send + 8'b1;
								
								data_out <= c_to_io_D;   // second half	
							end
					
							
						end
						8'd32:begin
							nCS <= 1'b1;
							output_enable <= 1'b0;			
							aux_count <= 8'd0;							
						end						
						default:begin
							aux_count <= aux_count + 8'd1;
							filter_c <= 1'b0;
						end
						
					endcase				
				
				end
				
				/**************************************
							WAIT FOR FILTER
				**************************************/
				Wait_After_Filter:begin								
					// Wait 0.5 x Ticlk x number of unused coefficients
					// Number of not unused coefficients mclk
					
					if ( aux_count == (filter_coeffs - {1'b0,length[7:1]}) )
					begin
						aux_count <= 8'b0;
						process <= Write_Register;
						last_process <= Wait_After_Filter;
					end
					else
						aux_count <= aux_count + 8'b1;					
					
				end
				
				/**************************************
						MODULATOR INTERFACE
				**************************************/
				Modulator_Interface:begin
					
					if ( close1 )
					begin
						modulator <= 1'b1;
						
						// n is even
						if ( nDRDY && ~nDRDY_last && (aux_count == 8'b0) )
						begin
							nCS <= 1'b0;
							nRD_WR <= 1'b0;
							aux_count <= 8'b1;	
							output_enable <= 1'b0;	
							samples <= 22'b0; 				
						end
						else if ( nDRDY && (aux_count == 8'b1) )
						begin
							aux_count <= 8'd2;							
						end
						else if ( nDRDY && (aux_count == 8'd2) )
						begin
							data[15:0] <= PARD;
							data_enable <= 1'b1;
							
							samples <= samples + 22'b1;
							
						end
						
						if ( ~nDRDY )
						begin
							data_enable <= 1'b0;							
							
							if ( samples == n_samples )
								close1 <= 1'b0;
						end
							
												
							
					end
					else
					begin
					
						nCS <= 1'b1;
						nRD_WR <= 1'b1;
						aux_count <= 1'b0;
						data_enable <= 1'b0;
						output_enable <= 1'b1;
					
						if ( ~receive_out )
						begin
							modulator <= 1'b0;	
							last_process <= Modulator_Interface;
							process <= Receive_CMDS;												
						end
						
					end
				
				end
				
				/**************************************
						NORMAL INTERFACE
				**************************************/
				Normal_Interface:begin
				
																			
					if ( close1 )
					begin
					
						if ( ~nDRDY )  
						begin
							aux_count <= 8'b1;	
							output_enable <= 1'b0;											
						end
						
						if ( aux_count == 8'd1 )
						begin
							nRD_WR <= 1'b0;
							aux_count <= 8'd2;
						end
						else if ( aux_count == 8'd2 )
						begin
							nCS <= 1'b0;
							aux_count <= 8'd3;							
						end
						else if ( aux_count == 8'd3 )
						begin
							aux_count <= 8'd4;													
						end
						else if ( aux_count == 8'd4 )
						begin
							aux_count <= 8'd5;	
							
							data_enable <= 1'b1;
							samples <= samples + 22'b1;
							
													
							if ( calculate )
								data[15:0] <= 16'h0;	
							else
								data[15:0] <= PARD;	
																					
															
						end
						else if ( aux_count == 8'd5 )
						begin
							data_enable <= 1'b0;
							nCS <= 1'b1;
							aux_count <= 8'd6;		
						end
						else if ( aux_count == 8'd6 )
						begin
							nRD_WR <= 1'b1;	
							aux_count <= 8'd7;								
						end
						else if ( aux_count == 8'd7 )
						begin
							aux_count <= 8'd8;
						end
						else if ( aux_count == 8'd8 )
						begin
							nRD_WR <= 1'b0;
							aux_count <= 8'd9;
						end
						else if ( aux_count == 8'd9 )
						begin
							nCS <= 1'b0;
							aux_count <= 8'd10;									
						end
						else if ( aux_count == 8'd10 )
						begin
							aux_count <= 8'd11;														
						end
						else if ( aux_count == 8'd11 )
						begin
							samples <= samples + 22'b1;
							
							if ( calculate )
								data[15:0] <= data_valid; 								
							else if ( ~read_regs )
								data[15:0] <= PARD;
						
							data_enable <= 1'b1;								
							aux_count <= 8'd12;							
						end	
						else if ( aux_count == 8'd12 )
						begin		
							data_enable <= 1'b0;														
							nCS <= 1'b1;
							aux_count <= 8'd13;										
						end
						else if ( aux_count == 8'd13 )
						begin
							nRD_WR <= 1'b1;
							aux_count <= 8'd0;
							output_enable <= 1'b1;
							
							if ( samples == n_samples )					
								close1 <= 1'b0;
												
						
							if ( read_regs || calculate )
							begin
								read_regs <= 1'b0;
								close1 <= 1'b0;			
								calculate <= 1'b0;					
							end
						end
												
					end	
					else
					begin	// ~close1 
						
						if ( ~receive_out )
						begin
							data_enable <= 1'b0;
							nRD_WR <= 1'b1;
							nCS <= 1'b1;
							aux_count <= 8'd0;											
							output_enable <= 1'b0;
							data_enable <= 1'b0;
								
							process <= Receive_CMDS;
							last_process <= Normal_Interface;	
								
							samples <= 22'b0;
						end
																
					end
				
				end
				
				/**************************************
						CALCULATE TIMING
				**************************************/
				Calculate_Timing:begin							
					if ( ~nDRDY && nDRDY_last ) 
					begin
						aux_count <= 8'b1;	
						close1 <= 1'b1;										
					end
					
					if ( aux_count == 8'd1 )
					begin
						nRD_WR <= 1'b0;
						aux_count <= 8'd2;
					end
					else if ( aux_count == 8'd2 )
					begin
						nCS <= 1'b0;
						aux_count <= 8'd3;														
					end
					else if ( aux_count == 8'd3 )
					begin
						aux_count <= 8'd4;
						output_enable <= 1'b0;							
					end
					else if ( aux_count == 8'd4 )
					begin
						data[15:0] <= 16'h0;
						aux_count <= 8'd5;							
							
					end
					else if ( aux_count == 8'd5 )
					begin
						output_enable <= 1'b1;
						nCS <= 1'b1;
						aux_count <= 8'd6;						
					end
					else if ( aux_count == 8'd6 )
					begin
						nRD_WR <= 1'b1;	
						aux_count <= 8'd8;								
					end
					else if ( aux_count == 8'd8 )
					begin
						nRD_WR <= 1'b0;
						aux_count <= 8'd9;
					end
					else if ( aux_count == 8'd9 )
					begin
						nCS <= 1'b0;
						aux_count <= 8'd10;							
					end
					else if ( aux_count == 8'd10 )
					begin
						aux_count <= 8'd11;
						output_enable <= 1'b0;																	
					end
					else if ( aux_count == 8'd11 )
					begin
						aux_count <= 8'd12;
						contar <= 1'b0;			
						
					end	
					else if ( aux_count == 8'd12 )
					begin		
						data_enable <= 1'b0;							
						output_enable <= 1'b0;
						nCS <= 1'b1;
						nRD_WR <= 1'b1;	
						aux_count <= 8'd0;	
						
						process <= Receive_CMDS;
						last_process <= Wait_For_Sync;
															
					end
				end
				
			endcase
	
		end
	end
	
	
		
	always@( posedge clk, posedge arst )
	begin
		if ( arst )
		begin
			
			state <= wait_for_start;
			sdata_valid <= 1'b0;
			count <= 22'b0;						
			state_dato <= send_dato1;				
				
			receive <= 1'b0;	
			count_clr_last <= 1'b1;		
			value_flag <= 1'b0;	
			value <= 16'h0;			
		end
		else 
		begin
		
			if ( ~nRESET && (state == wait_for_start) && power_up )
			begin
				sdata_valid <= 1'b0;
				count <= 22'b0;						
				state_dato <= send_dato1;				
				receive <= 1'b0;	
				count_clr_last <= 1'b1;	
			end
			
			if ( write_regs && ~value_flag)
			begin
				if ( (usb_cmd == 8'h50) )
				begin
					if ( c_to_io_D[15:8] == 8'hF1 )
						value[15:8] <= c_to_io_D[7:0];
					else if ( c_to_io_D[15:8] == 8'hF2 )
					begin
						value[7:0] <= c_to_io_D[7:0];
						value_flag <= 1'b1;
					end
						
				end				
			end
			else if ( ~write_regs )
				value_flag <= 1'b0;
			
			
			count_clr_last <= count_clr;
			state <= next_state;
		
						
			if ( data_enable_out && (count < n_samples) ) 
			begin
				sdata_valid <= 1'b1;
				data_a <= data[15:0];
				count <= count + 22'b1;
				
			end
			else
				sdata_valid <= 1'b0;
			
						
			if ( count_clr )
				receive <= 1'b0;
			
					
			if ( ~count_clr && count_clr_last && (usb_cmd == 8'h10) )
			begin
				receive <= 1'b1;
				count <= 22'b0;											
			end
				
				
		end
	end
	
	
	always@( * )
	begin
	
		count_clr = 1'b0;
		
		io_to_c_D = 16'hxxxx;
		io_to_c_stop = 1'b0;
		next_state = state;
		
		if ( filter_c )
			io_to_c_rts = 1'b1;
		else 
			io_to_c_rts = 1'b0;
		
	
		case( state )
		
			wait_for_start: begin
				count_clr = 1'b1;
				if( c_to_io_start && ( usb_cmd != 8'h40 ) && ~power_up )
					next_state = wait_for_end_of_config;
			end
			
			wait_for_end_of_config: begin
				count_clr = 1'b1;
				if( ~c_to_io_rts )
					next_state = wait_for_sdata_valid;
			end
		
			wait_for_sdata_valid: begin
				if( c_to_io_stop & ( count == n_samples ) )
					next_state = stop;
				else if( ~c_to_io_stop & sdata_valid )
					next_state = send_a;
					
			end
			
			send_a: begin
				io_to_c_rts = 1'b1;
				io_to_c_D = data_a;
				if( c_to_io_cts )
				begin
					next_state = wait_for_sdata_valid;
				end
			end
			
			
			stop: begin
				count_clr = 1'b1;
				next_state = wait_for_start;
				io_to_c_stop = 1'b1;
				
			end
			
		endcase
	end

	
endmodule
