// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
//-----------------------------------------------------------------------------
//	Parallel Interface
//
//  Status register when reading DATA
//
	/*--------------------------------------------------------
	   D7     D6    D5     D4     D3     D2   D1  D0    
	---------------------------------------------------
	 DVALID  OVR   UFILT  LPWR  FILTOK  DLOK   0   0 
	--------------------------------------------------------*/
	
//-----------------------------------------------------------------------------

module parallel
(
	input arst,
	input clk,
	
	input c_to_io_start,
	input c_to_io_stop,
	input c_to_io_cts,
	input c_to_io_rts,
	input [15:0] c_to_io_D,
		
	output io_to_c_busy,
	output reg io_to_c_stop,
	output io_to_c_cts,
	output reg io_to_c_rts,
	output reg [15:0] io_to_c_D,
	
	input ppi_data_valid,
	input [31:0] ppi_data,
	
	// communication with SPI block
	output reg spi_enable,	
	output reg [15:0] spi_register,
	output reg rx_enable,
	
	input [21:0] n_samples
	
);


	parameter pDataW = 	16;
	
	parameter
		wait_for_start           = 3'b000,
		wait_for_end_of_config   = 3'b001,
		wait_for_sdata_valid     = 3'b010,
		send_a                   = 3'b011,
		stop                     = 3'b100;
		
	reg                    sdata_valid;
	reg                    count_clr;
	reg [pDataW-1:0]       data_a;
	reg [21:0]             count;
	reg [2:0]              state;
	reg [2:0]              next_state;
	reg [31:0]         	   data;
	reg 				   ready_data;	
	reg [2:0] 			   state_dato;
	
	reg [21:0]             samples;
		
	parameter
		send_dato1        = 3'b000,
		send_nodato1      = 3'b001,
		send_wait_dato2   = 3'b010,
		send_dato2        = 3'b011,
		send_nodato2      = 3'b100,
		state_init        = 3'b101;
	
	

	assign io_to_c_cts = c_to_io_rts;
	
	
	always@( posedge clk, posedge arst )
	begin
		if ( arst )
		begin
			
			state <= wait_for_start;
			sdata_valid <= 1'b0;
			count <= 22'b0;						
			state_dato <= send_dato1;				
			spi_enable <= 1'b0;	
			spi_register <= 16'b0;	
			ready_data <= 1'b0;
			
			data <= 32'h0;
			
			rx_enable <= 1'b0;
				
		end
		else 
		begin
		
			if ( c_to_io_D[15:12] == 4'hF && c_to_io_D[11] )				 
			begin
				spi_enable <= 1'b1;
				spi_register <= c_to_io_D;
			end
			else 
				spi_enable <= 1'b0;
			
							
			state <= next_state;
			
			if ( ppi_data_valid && ~count_clr )
			begin
				ready_data <= 1'b1; 
				data <= ppi_data;				
			end		
			
			if ( ready_data )
			begin
				if ( state_dato == send_dato1 )
				begin
					
					sdata_valid <= 1'b1;
					data_a <= data[31:16];
					state_dato <= send_nodato1;	
					count <= count + 22'b1;
									
				end
				else if ( state_dato == send_nodato1 )
				begin
					sdata_valid <= 1'b0;
					state_dato <= send_wait_dato2;					
				end
				else if ( state_dato == send_wait_dato2 )
				begin
					state_dato <= send_dato2;
				end
				else if ( state_dato == send_dato2 )
				begin

					data_a <= data[15:0];
					sdata_valid <= 1'b1;
					count <= count + 22'b1;
					state_dato <= send_nodato2;					
				end
				else
				begin

					sdata_valid <= 1'b0;					
					state_dato <= send_dato1;
					ready_data <= 1'b0;
				end			
			end
			
					
			if( c_to_io_stop && (count_clr | ( count == samples )) )
			begin
				count <= 17'b0;			
				rx_enable <= 1'b0;
			end
									
			if( c_to_io_rts )
			begin
				count <= 22'b0;		
				rx_enable <= 1'b1;	
				samples <= n_samples;	
			end
				
		end
	end
	
	
	always@( * )
	begin
	
		count_clr = 1'b0;
		io_to_c_rts = 1'b0;
		io_to_c_D = 16'hxxxx;
		io_to_c_stop = 1'b0;
		next_state = state;
	
		case( state )
		
			wait_for_start: begin
				count_clr = 1'b1;
				if( c_to_io_start )
					next_state = wait_for_end_of_config;
			end
			
			wait_for_end_of_config: begin
				count_clr = 1'b1;
				if( ~c_to_io_rts )
					next_state = wait_for_sdata_valid;
			end
		
			wait_for_sdata_valid: begin
				if( c_to_io_stop & ( count == samples ) )
					next_state = stop;
				else if( ~c_to_io_stop & sdata_valid )
					next_state = send_a;
			end
			
			send_a: begin
				io_to_c_rts = 1'b1;
				io_to_c_D = data_a;
				if( c_to_io_cts )
				begin
					next_state = wait_for_sdata_valid;
				end
			end
			
			
			stop: begin
				count_clr = 1'b1;
				io_to_c_stop = 1'b1;
				next_state = wait_for_start;
			end
			
		endcase
	end

	
endmodule
