// Copyright (c) 2007 Analog Devices, Inc. All rights reserved.
//-----------------------------------------------------------------------------
//	GPIO controler
//
//	send command hC200
//	followed by one or more of:
//		output en        0001 0100 dddd dddd
//		load             0001 0000 dddd dddd
//		clr              0001 0001 dddd dddd
//		set              0001 0010 dddd dddd
//		tgl              0001 0011 dddd dddd
//      read inputs      0010 xxxx xxxx xxxx
//	followed by h0000
//	
//	where dddd dddd is used for the data / gpio pin select.( GPIO 7 down to 0 )
//
//  With the AD7266 are used the GPIO pins 2,3,4 to select the input channels.
//
//-----------------------------------------------------------------------------
module gpio_controller
(
	input arst,
	input clk,
		
	// usb signals
	input en,
	input usb_to_c_start,
	input usb_to_c_stop,
	input usb_to_c_cts,
	input usb_to_c_rts,
	input [15:0] usb_to_c_D,
	
	output c_to_usb_busy,
	output reg c_to_usb_stop,
	output reg c_to_usb_cts,
	output reg c_to_usb_rts,
	output [15:0] c_to_usb_D
	
	
);

	parameter 
		wait_for_start      = 3'b000,
		decision            = 3'b001,
		usb_stop            = 3'b010,
		gpio_update_state   = 3'b011,
		gpio_read_state     = 3'b100;
	
	reg [2:0] CLKOUT_reg;
	reg [7:0] gpio_output_en;
	reg [7:0] GPIO_out;
	reg [7:0] GPIO_in;
	reg gpio_update;
	
	reg [2:0] state;
	reg [2:0] next_state;
	
	assign c_to_usb_D = 16'b0;
	assign c_to_usb_busy = 1'b0;
	
	
	always@( posedge clk, posedge arst )
	begin
		if( arst )
		begin
			
			state <= wait_for_start;
			CLKOUT_reg <= 3'b0;
			gpio_output_en <= 8'b0;
			GPIO_out <= 8'b0;
			GPIO_in <= 8'b0;
			
		end
		else
		begin
			
			state <= next_state;
			
			CLKOUT_reg <= CLKOUT_reg + 3'b1;
						
			
			if( gpio_update )
			begin
				case( usb_to_c_D[11:8] )
					4'b0100: gpio_output_en <= usb_to_c_D[7:0];       //output en
					4'b0000: GPIO_out <= usb_to_c_D[7:0];             //load
					4'b0001: GPIO_out <= ~usb_to_c_D[7:0] & GPIO_out; //clr
					4'b0010: GPIO_out <= usb_to_c_D[7:0] | GPIO_out;  //set
					4'b0011: GPIO_out <= usb_to_c_D[7:0] ^ GPIO_out;  //tgl
					default: ;
				endcase
			end
			
		end
	end
	
	always@( * )
	begin
		
		c_to_usb_stop = 1'b0;
		c_to_usb_cts = 1'b0;
		c_to_usb_rts = 1'b0;
		gpio_update = 1'b0;
		next_state = state;
		
		case( state )
			
			wait_for_start: begin
				if( en & usb_to_c_start )
					next_state = decision;
			end
			
			decision: begin
				if( usb_to_c_stop )
					next_state = usb_stop;
				else if( usb_to_c_rts & usb_to_c_cts )
					case( usb_to_c_D[15:12] )
						4'b0001: next_state = gpio_update_state;
						4'b0010: next_state = gpio_read_state;
						default: next_state = usb_stop;
					endcase
				else
					next_state = decision;
			end
			
			usb_stop: begin
				c_to_usb_stop = 1'b1;
				if( usb_to_c_stop )
					next_state = wait_for_start;
				else
					next_state = usb_stop;
			end
			
			gpio_update_state: begin
				c_to_usb_cts = 1'b1;
				gpio_update = 1'b1;
				next_state = decision;
			end
			
			gpio_read_state: begin
				c_to_usb_cts = 1'b1;
				c_to_usb_rts = 1'b1;
				next_state  = decision;
			end
			
		endcase
	end

endmodule
