// Copyright (c) 2008 Analog Devices, Inc. All rights reserved.
module controller0
(
	input arst,
	input clk,
	
	// usb signals
	input en,
	input [7:0] usb_cmd,
	input usb_to_c_start,
	input usb_to_c_stop,
	input usb_to_c_cts,
	input usb_to_c_rts,
	input [15:0] usb_to_c_D,
	
	output reg c_to_usb_busy,
	output reg c_to_usb_stop,
	output reg c_to_usb_cts,
	output reg c_to_usb_rts,
	output reg [15:0] c_to_usb_D,
	
	// SRAM signals
	output reg nSRAM_CE1,
	output nSRAM_WE,
	output reg nSRAM_OE,
	output reg [20:0] SRAM_A,
	inout [15:0] SRAM_D,
	
	// io signals
	input io_to_c_busy,
	input io_to_c_stop,
	input io_to_c_cts,
	input io_to_c_rts,
	input [15:0] io_to_c_D,
	
	output reg c_to_io_start,
	output reg c_to_io_stop,
	output reg c_to_io_cts,
	output reg c_to_io_rts,
	output [15:0] c_to_io_D,
	
	output reg [21:0] n_samples
);

	// state definitions
	parameter
		wait_for_cmd                = 6'b000000,
		setup                       = 6'b000001,
		setup2                      = 6'b000010,
		decision                    = 6'b000011,
		usb_stop                    = 6'b000100,
		sram_first_wr               = 6'b000101,
		sram_wr_wait_for_data       = 6'b000110,
		sram_wr                     = 6'b000111,
		sram_wr2                    = 6'b001000,
		sram_test_initial_rd        = 6'b001001,
		sram_rd_setup               = 6'b001010,
		sram_rd_setup_wait_for_cts  = 6'b001011,
		sram_rd                     = 6'b001100,
		sram_rd_rts                 = 6'b001101,
		sram_rd_wait_for_cts        = 6'b001110,
		sram_test_first_wr          = 6'b001111,
		sram_test_wr                = 6'b010000,
		sram_test_wr2               = 6'b010001,
		sram_test_rd_wait_for_cts   = 6'b010010,
		sram_test_rd                = 6'b010011,
		sram_test_rd_check          = 6'b010100,
		sram_test_pass              = 6'b010101,
		sram_test_fail              = 6'b010110,
		sram_test1                  = 6'b010111,
		sram_test2                  = 6'b011000,
		sram_test3                  = 6'b011001,
		sram_test4                  = 6'b011010,
		sram_test5                  = 6'b011011,
		capture                     = 6'b011100,
		capture2                    = 6'b011101,
		capture3                    = 6'b011110,
		capture4                    = 6'b011111,
		capture_first_wr            = 6'b100000,
		capture_wr                  = 6'b100001,
		capture_wr2                 = 6'b100010,
		capture_stop                = 6'b100011,
		linearity                   = 6'b100100,
		lin_zero_ram                = 6'b100101,
		lin_zero_ram2               = 6'b100110,
		lin_stop_val_load           = 6'b100111,
		lin_max_bin_high_load       = 6'b101000,
		lin_max_bin_low_load        = 6'b101001,
		lin_min_bin_high_load       = 6'b101010,
		lin_min_bin_low_load        = 6'b101011,
		lin_io_config               = 6'b101100,
		lin_io_config2              = 6'b101101,
		lin_stop_or_wait_for_io     = 6'b101110,
		lin_A_load                  = 6'b101111,
		lin_rd                      = 6'b110000,
		lin_rd2                     = 6'b110001,
		lin_wr                      = 6'b110010,
		lin_stop                    = 6'b110011,
		lin_max_high                = 6'b110100,
		lin_max_low                 = 6'b110101,
		lin_min_high                = 6'b110110,
		lin_min_low                 = 6'b110111,
		send_filter                 = 6'b111000,
		send_filter_length          = 6'b111001,
		send_filter_word            = 6'b111010,
		send_filter_stop            = 6'b111011;
		
	reg [5:0] state;
	reg [5:0] next_state;
	
	reg [21:0] countdown_load_reg;
	reg [21:0] countdown;
	
	reg [7:0] n_coef;
	reg [7:0] length;
	reg filter;
	reg io_to_c_rts_last;
		
	reg wr;
	reg A_counter_en;
	reg SRAM_D_in_en;
	reg A_counter_clr;
	reg A_counter_io_load;
	reg countdown_en;
	reg countdown_high_load;
	reg countdown_low_load;
	reg countdown_load;
	reg wr1;
	reg wr1_delayed_half_clock;
	
	reg linearity_stop_val_en;
	reg [15:0] linearity_stop_val;
	
	reg max_bin_high_en;
	reg max_bin_low_en;
	reg [20:0] max_bin_val;
	
	reg min_bin_high_en;
	reg min_bin_low_en;
	reg [20:0] min_bin_val;
	
	reg max_min_bin_count_en;
	reg max_min_bin_count_clr;
	reg [47:0] max_bin_count;
	reg [47:0] min_bin_count;
	
	reg [15:0] SRAM_D_out;
	reg [15:0] SRAM_D_in;
	reg [15:0] SRAM_D_tri_out;
	wire [15:0] SRAM_D_tri_in;
	
	wire max_bin = ( max_bin_val == SRAM_A );
	wire min_bin = ( min_bin_val == SRAM_A );
	wire linearity_stop = ( SRAM_D_tri_out == linearity_stop_val );
	
	// SRAM_D bidirectional data bus tristate buffer
	assign SRAM_D = nSRAM_OE ? SRAM_D_tri_out : 16'hzzzz;
	assign SRAM_D_tri_in = nSRAM_OE ? 16'hxxxx : SRAM_D;
	
	assign c_to_io_D = usb_to_c_D;
	
	
	always@( posedge clk, posedge arst )
	begin
	
		if ( arst )
			length <= 8'b0;
		else
			if ( usb_cmd == 8'h40 && usb_to_c_start )
			begin
				case ( usb_to_c_D[3:0] )
					
					4'b0001:begin
						length <= 8'd12;
					end
					4'b0011:begin
						length <= 8'd24;
					end
					4'b0101:begin
						length <= 8'd36;
					end
					4'b0111:begin
						length <= 8'd48;
					end
					4'b1001:begin
						length <= 8'd60;
					end
					4'b1011:begin
						length <= 8'd72;
					end
					4'b1101:begin
						length <= 8'd84;
					end
					4'b1111:begin
						length <= 8'd96;
					end
					
				endcase	
			end
	end

	
	// wr and SRAM_D registers
	always@( posedge clk )
	begin
		wr1 <= wr;
		if( SRAM_D_in_en )
			SRAM_D_in <= SRAM_D_tri_in;
		if( wr )
			SRAM_D_tri_out <= SRAM_D_out;
	end
	
	// generate wr1_delayed_half_clock
	always@( negedge clk )
		wr1_delayed_half_clock <= wr1;
		
	assign nSRAM_WE = ~( wr1 | wr1_delayed_half_clock );
	
	always@( posedge clk )
		nSRAM_OE <= wr | wr1;

	// SRAM address counter
	always@( posedge clk, posedge arst )
		if( arst )
			SRAM_A <= 21'b0;
		else
			if( A_counter_en )
			begin
				if( A_counter_clr )
					SRAM_A <= 21'b0;
				else if( A_counter_io_load )
					SRAM_A <= { 5'b0, io_to_c_D };
				else
					SRAM_A <= SRAM_A + 21'b1;
			end
			
	// countdown load reg
	always@( posedge clk, posedge arst )
		if( arst )
			countdown_load_reg <= 22'b0;
		else if( countdown_high_load )
			countdown_load_reg[21:16] <= usb_to_c_D[5:0];
		else if( countdown_low_load )
			countdown_load_reg[15:0] <= usb_to_c_D;
	
	// countdown counter
	always@( posedge clk, posedge arst )
		if( arst )
		begin
			countdown <= 22'b0;
			n_samples <= 22'b0;
		end
		else
			if( countdown_en )
			begin
				if( countdown_load )
				begin
					countdown <= {countdown_load_reg[20:0],1'b0};
					n_samples <= {countdown_load_reg[20:0],1'b0};
				end
				else
					countdown <= countdown - 22'b1;
			end
			
	// counter - filter
	always@( posedge clk, posedge arst )
	begin
		if( arst )
			n_coef <= 8'b0;
		else
			if ( (state == send_filter_word) && filter )
				n_coef <= n_coef + 8'b1;
			else if ( state == send_filter_stop )
				n_coef <= 8'b0;
	end
		
	always@( posedge clk, posedge arst )
	begin
		if( arst )
		begin
			filter <= 1'b0;
			io_to_c_rts_last <= 1'b0;
		end
		else
		begin
			io_to_c_rts_last <= io_to_c_rts;
			
			if ( io_to_c_rts && ~io_to_c_rts_last)
				filter <= 1'b1;
			else
				filter <= 1'b0;
				
		end
	end                                            
			
			
	// linearity
	always@( posedge clk )
	begin
		
		if( linearity_stop_val_en )
			linearity_stop_val <= usb_to_c_D;
		
		if( max_bin_high_en )
			max_bin_val[20:16] <= usb_to_c_D[4:0];
		else if( max_bin_low_en )
			max_bin_val[15:0] <= usb_to_c_D;
		
		if( min_bin_high_en )
			min_bin_val[20:16] <= usb_to_c_D[4:0];
		else if( min_bin_low_en )
			min_bin_val[15:0] <= usb_to_c_D;
		
		if( max_min_bin_count_clr )
		begin
			max_bin_count <= 48'b0;
			min_bin_count <= 48'b0;
		end
		else 
			if( max_min_bin_count_en )
			begin
				if( max_bin )
					max_bin_count <= max_bin_count + 48'b1;
				if( min_bin )
					min_bin_count <= min_bin_count + 48'b1;
			end		
	end
					
	// state register
	always@( posedge clk, posedge arst )
		if( arst )
			state <= wait_for_cmd;
		else
			state <= next_state;
			
	// state transition and output logic
	always@( * )
	begin	
		c_to_io_start = 1'b0;
	    c_to_io_stop = 1'b0;
	    c_to_io_cts = 1'b0;
	    c_to_io_rts = 1'b0;
		nSRAM_CE1 = 1'b0;
		wr = 1'b0;
		c_to_usb_busy = 1'b0;
		c_to_usb_stop = 1'b0;
		c_to_usb_cts = 1'b0;
		c_to_usb_rts = 1'b0;
		c_to_usb_D = 16'hxxxx;
		SRAM_D_out = 16'hxxxx;
		A_counter_en = 1'b0;
		SRAM_D_in_en = 1'b0;
		A_counter_clr = 1'b0;
		A_counter_io_load = 1'b0;
		countdown_en = 1'b0;
		countdown_low_load = 1'b0;
		countdown_high_load = 1'b0;
		countdown_load = 1'b0;
		
		linearity_stop_val_en = 1'b0;
		max_bin_high_en = 1'b0;
		max_bin_low_en = 1'b0;
		min_bin_high_en = 1'b0;
		min_bin_low_en = 1'b0;
		max_min_bin_count_en = 1'b0;
		max_min_bin_count_clr = 1'b0;
				
		next_state = state;
		
		case( state )
		
			// setup and stop states ------------------------------------------
			
			wait_for_cmd: begin
				nSRAM_CE1 = 1'b1;
				if( en & usb_to_c_rts )
					next_state = setup;
			end
			
			setup: begin
				A_counter_en = 1'b1;
				A_counter_clr = 1'b1;
				countdown_high_load = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = setup2;
			end
			
			setup2: begin
				countdown_low_load = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = decision;
			end
			
			decision: begin
				countdown_en = 1'b1;
				countdown_load = 1'b1;
				case( usb_cmd )
					8'h01: next_state = sram_first_wr;
					8'h02: next_state = sram_rd_setup;
					8'h03: next_state = sram_test_first_wr;
					8'h10: next_state = capture;
					8'h20: next_state = linearity;
					8'h40: next_state = send_filter;					
					default: next_state = usb_stop;
				endcase
			end
						
			usb_stop: begin
				c_to_usb_stop = 1'b1;
				if( usb_to_c_stop )
					next_state = wait_for_cmd;
			end
			
			// sram write -----------------------------------------------------
			
			sram_first_wr: begin
				wr = 1'b1;
				SRAM_D_out = usb_to_c_D;
				c_to_usb_cts = 1'b1;
				next_state = sram_wr2;
			end
			
			sram_wr_wait_for_data: begin
				if( usb_to_c_rts )
					next_state = sram_wr;
			end
			
			sram_wr: begin
				wr = 1'b1;
				A_counter_en = 1'b1;
				countdown_en = 1'b1;
				SRAM_D_out = usb_to_c_D;
				c_to_usb_cts = 1'b1;
				next_state = sram_wr2;
			end
			
			sram_wr2: begin
				if( countdown == 22'b1 )
					next_state = usb_stop;
				else
					if( usb_to_c_rts )
						next_state = sram_wr;
					else
						next_state = sram_wr_wait_for_data;
			end
			
			// sram read ------------------------------------------------------
				
			sram_rd_setup: begin
				countdown_en = 1'b1;
				countdown_load = 1'b1;
				A_counter_en = 1'b1;
				A_counter_clr = 1'b1;
				next_state = sram_rd_setup_wait_for_cts;
			end
			
			sram_rd_setup_wait_for_cts: begin
				if( usb_to_c_cts )
					next_state = sram_rd;
			end
			
			sram_rd: begin
				A_counter_en = 1'b1;
				SRAM_D_in_en = 1'b1;
				next_state = sram_rd_rts;
			end
			
			sram_rd_rts: begin
				c_to_usb_D = SRAM_D_in;
				c_to_usb_rts = 1'b1;
				countdown_en = 1'b1;
				if( usb_to_c_cts )
				begin
					if( countdown == 22'b1 )
						next_state = usb_stop;
					else
						next_state = sram_rd;
				end
				else
					next_state = sram_rd_wait_for_cts;
			end
			
			sram_rd_wait_for_cts: begin
				c_to_usb_D = SRAM_D_in;
				c_to_usb_rts = 1'b1;
				if( usb_to_c_cts )
				begin
					if( countdown == 22'b0 )
						next_state = usb_stop;
					else
						next_state = sram_rd;
				end
			end
			
			// sram test ------------------------------------------------------
			sram_test_first_wr: begin
				wr = 1'b1;
				SRAM_D_out = countdown[15:0];
				countdown_en = 1'b1;
				next_state = sram_test_wr2;	
			end
						
			sram_test_wr: begin
				wr = 1'b1;
				A_counter_en = 1'b1;
				countdown_en = 1'b1;
				SRAM_D_out = countdown[15:0];
				next_state = sram_test_wr2;
			end
			
			sram_test_wr2: begin
				if( SRAM_A == 21'h1FFFFF )
					next_state = sram_test_initial_rd;
				else
					next_state = sram_test_wr;
			end
			
			sram_test_initial_rd: begin
				A_counter_en = 1'b1;
				SRAM_D_in_en = 1'b1;
				next_state = sram_test_rd_wait_for_cts;
			end
			
			sram_test_rd_wait_for_cts: begin
				if( usb_to_c_cts )
					next_state = sram_test_rd;
			end
			
			sram_test_rd: begin
				A_counter_en = 1'b1;
				SRAM_D_in_en = 1'b1;
				next_state = sram_test_rd_check;
			end
			
			sram_test_rd_check: begin
				countdown_en = 1'b1;
				if( countdown[15:0] !== SRAM_D_in )
					next_state = sram_test_fail;
				else
					if( SRAM_A == 21'h0 )
						next_state = sram_test_pass;
					else
						next_state = sram_test_rd;
			end
			
			sram_test_pass: begin
				c_to_usb_D = 16'h0012;
				c_to_usb_rts = 1'b1;
				next_state = sram_test1;
			end
			
			sram_test_fail: begin
				c_to_usb_D = 16'h00FA;
				c_to_usb_rts = 1'b1;
				next_state = sram_test1;
			end
			
			sram_test1: begin
				c_to_usb_D = SRAM_A[20:16];
				c_to_usb_rts = 1'b1;
				next_state = sram_test2;
			end
			
			sram_test2: begin
				c_to_usb_D = SRAM_A[15:0];
				c_to_usb_rts = 1'b1;
				next_state = sram_test3;
			end
			
			sram_test3: begin
				c_to_usb_D = countdown[21:16];
				c_to_usb_rts = 1'b1;
				next_state = sram_test4;
			end
			
			sram_test4: begin
				c_to_usb_D = countdown[15:0];
				c_to_usb_rts = 1'b1;
				next_state = sram_test5;
			end
			
			sram_test5: begin
				c_to_usb_D = SRAM_D_in;
				c_to_usb_rts = 1'b1;
				next_state = usb_stop;
			end
			
			// capture --------------------------------------------------------
			
			capture: begin
				c_to_io_start = 1'b1;
				next_state = capture2;
			end
			
			capture2: begin
				c_to_usb_cts = 1'b1;
				c_to_io_rts = 1'b1;
				next_state = capture3;
			end
			
			capture3: begin
				c_to_usb_cts = 1'b1;
				c_to_io_rts = 1'b1;
				next_state = capture4;
			end
			
			capture4: begin
				if( io_to_c_rts )
					next_state = capture_first_wr;
			end
			
			capture_first_wr: begin
				wr = 1'b1;
				SRAM_D_out = io_to_c_D;
				c_to_io_cts = 1'b1;
				next_state = capture_wr2;
				
			end
			
			capture_wr: begin
				wr = 1'b1;
				A_counter_en = 1'b1;
				countdown_en = 1'b1;
				SRAM_D_out = io_to_c_D;
				c_to_io_cts = 1'b1;
				next_state = capture_wr2;
			end 
			
			capture_wr2: begin
				if( countdown == 22'b1 )
					next_state = capture_stop;
				else
					if( io_to_c_rts )
						next_state = capture_wr;
					else
						next_state = capture_wr2;
			end
			
			capture_stop: begin
				c_to_io_stop = 1'b1;
				if( io_to_c_stop )
					next_state = sram_rd_setup;
			end
			
			// linearity ------------------------------------------------------
			
			linearity: begin
				max_min_bin_count_clr = 1'b1;
				wr = 1'b1;
				SRAM_D_out = 16'b0;
				next_state = lin_zero_ram2;
			end
			
			lin_zero_ram: begin
				wr = 1'b1;
				A_counter_en = 1'b1;
				SRAM_D_out = 16'b0;
				next_state = lin_zero_ram2;
			end
			
			lin_zero_ram2: begin
				if( SRAM_A == 21'h1FFFFF )
					next_state = lin_stop_val_load;
				else
					next_state = lin_zero_ram;
			end
			
			lin_stop_val_load: begin
				linearity_stop_val_en = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = lin_max_bin_high_load;
			end
			
			lin_max_bin_high_load: begin
				max_bin_high_en = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = lin_max_bin_low_load;			
			end
			
			lin_max_bin_low_load: begin
				max_bin_low_en = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = lin_min_bin_high_load;	
			end
						
			lin_min_bin_high_load: begin
				min_bin_high_en = 1'b1;
				c_to_usb_cts = 1'b1;
				next_state = lin_min_bin_low_load;			
			end
			
			lin_min_bin_low_load: begin
				min_bin_low_en = 1'b1;
				c_to_usb_cts = 1'b1;
				c_to_io_start = 1'b1;
				next_state = lin_io_config;	
			end
			
			lin_io_config: begin
				c_to_usb_cts = 1'b1;
				c_to_io_rts = 1'b1;
				next_state = lin_io_config2;
			end
			
			lin_io_config2: begin
				c_to_usb_cts = 1'b1;
				c_to_io_rts = 1'b1;
				next_state = lin_stop_or_wait_for_io;
			end
			
			lin_stop_or_wait_for_io: begin
				if( linearity_stop & ~max_bin & ~min_bin )
					next_state = lin_stop;
				else if( io_to_c_rts )
					next_state = lin_A_load;
			end
			
			lin_A_load: begin
				A_counter_en = 1'b1;
				A_counter_io_load = 1'b1;
				c_to_io_cts = 1'b1;
				next_state = lin_rd;
			end
			
			lin_rd: begin
				next_state = lin_rd2;
			end
			
			lin_rd2: begin
				SRAM_D_in_en = 1'b1;
				next_state = lin_wr;
			end
			
			lin_wr: begin
				wr = 1'b1;
				SRAM_D_out = SRAM_D_in + 16'b1;
				max_min_bin_count_en = 1'b1;
				next_state = lin_stop_or_wait_for_io;
			end
		
			lin_stop: begin
				c_to_io_stop = 1'b1;
				if( io_to_c_stop )
					next_state = lin_max_high;
			end
			
			lin_max_high: begin
				c_to_usb_D = max_bin_count[47:32];
				c_to_usb_rts = 1'b1;
				if( usb_to_c_cts )
					next_state = lin_max_low;
			end
			
			lin_max_low: begin
				c_to_usb_D = max_bin_count[31:16];
				c_to_usb_rts = 1'b1;
				if( usb_to_c_cts )
					next_state = lin_min_high;
			end
			
			lin_min_high: begin
				c_to_usb_D = min_bin_count[47:32];
				c_to_usb_rts = 1'b1;
				if( usb_to_c_cts )
					next_state = lin_min_low;
			end
			
			lin_min_low: begin
				c_to_usb_D = min_bin_count[31:16];
				c_to_usb_rts = 1'b1;
				if( usb_to_c_cts )
					next_state = sram_rd_setup;
			end
			
			// send filter --------------------------------------------------------
			
			send_filter: begin
				c_to_io_start = 1'b1;
				next_state = send_filter_length;				
			end
			
			send_filter_length: begin
				next_state = send_filter_word;
			end
			
			send_filter_word: begin
				
				if ( filter )
				begin				
					c_to_usb_cts = 1'b1;
					c_to_io_rts = 1'b1;
										
					if ( n_coef == length )
						next_state = send_filter_stop;
					else
						next_state = send_filter_word;						
					
				end
				
			end
			
			send_filter_stop: begin
				c_to_io_stop = 1'b1;
				next_state = usb_stop;
			end
			
						
		endcase
	end
	
endmodule
