/*!
 LTC2945: 12-Bit Wide Range Power Monitor

@verbatim

The LTC2945 is a rail-to-rail system monitor that measures current, voltage, and
power. It features an operating range of 2.7V to 80V and includes a shunt
regulator for supplies above 80V to allow flexibility in the selection of input
supply. The current measurement range of 0V to 80V is independent of the input
supply. An onboard 0.75% accurate 12-bit ADC measures load current, input
voltage and an auxiliary external voltage. A 24-bit power value is generated by
digitally multiplying the measured 12-bit load current and input voltage data.
Minimum and maximum values are stored and an overrange alert with programmable
thresholds minimizes the need for software polling. Data is reported via a
standard I2C interface. Shutdown mode reduces power consumption to 20uA.
standard I2C interface. Shutdown mode reduces power consumption to 20uA.

I2C DATA FORMAT (MSB FIRST):

Data Out:
Byte #1                                    Byte #2                     Byte #3

START  SA6 SA5 SA4 SA3 SA2 SA1 SA0 W SACK  X  X C5 C4 C3 C2 C1 C0 SACK D7 D6 D5 D4 D3 D2 D1 D0 SACK  STOP

Data In:
Byte #1                                    Byte #2                                    Byte #3

START  SA6 SA5 SA4 SA3 SA2 SA1 SA0 W SACK  X  X  C5 C4 C3 C2 C1 C0 SACK  Repeat Start SA6 SA5 SA4 SA3 SA2 SA1 SA0 R SACK

Byte #4                                   Byte #5
MSB                                       LSB
D15 D14  D13  D12  D11  D10  D9 D8 MACK   D7 D6 D5 D4 D3  D2  D1  D0  MNACK  STOP

START       : I2C Start
Repeat Start: I2c Repeat Start
STOP        : I2C Stop
SAx         : I2C Address
SACK        : I2C Slave Generated Acknowledge (Active Low)
MACK        : I2C Master Generated Acknowledge (Active Low)
MNACK       : I2c Master Generated Not Acknowledge
W           : I2C Write (0)
R           : I2C Read  (1)
Cx          : Command Code
Dx          : Data Bits
X           : Don't care



Example Code:

Read power, current, and voltage.

    adc_command = LTC2945_SENSE_MONITOR | LTC2945_CONTINUOUS_MODE; // Builds commands to set LTC2945 to continuous mode
    ack |= LTC2945_write(LTC2945_I2C_ADDRESS, LTC2945_CONTROL_REG, adc_command);   // Sets the LTC2945 to continuous mode

    resistor = .02; // Resistor Value On Demo Board

    ack |= LTC2945_read_24_bits(LTC2945_I2C_ADDRESS, LTC2945_POWER_MSB2_REG, &power_code);  // Reads the ADC registers that contains V^2
    power = LTC2945_code_to_power(power_code, resistor, LTC2945_Power_lsb); // Calculates power from power code, resistor value and power lsb

    ack |= LTC2945_read_12_bits(LTC2945_I2C_ADDRESS, LTC2945_DELTA_SENSE_MSB_REG, &current_code); // Reads the voltage code across sense resistor
    current = LTC2945_code_to_current(current_code, resistor, LTC2945_DELTA_SENSE_lsb); // Calculates current from current code, resistor value and current lsb

    ack |= LTC2945_read_12_bits(LTC2945_I2C_ADDRESS, LTC2945_VIN_MSB_REG, &VIN_code);   // Reads VIN voltage code
    VIN = LTC2945_VIN_code_to_voltage(VIN_code, LTC2945_VIN_lsb);  // Calculates VIN voltage from VIN code and lsb


@endverbatim

http://www.linear.com/product/LTC2945

http://www.linear.com/product/ltc2945#demoboards


Copyright 2018(c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
 - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
 - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in
   the documentation and/or other materials provided with the
   distribution.
 - Neither the name of Analog Devices, Inc. nor the names of its
   contributors may be used to endorse or promote products derived
   from this software without specific prior written permission.
 - The use of this software may or may not infringe the patent rights
   of one or more patent holders.  This license does not release you
   from the requirement that you obtain separate licenses from these
   patent holders to use this software.
 - Use of the software either in source or binary form, must be run
   on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*! @file
    @ingroup LTC2945
    Header for LTC2945: 12-bit Wide Range Power Monitor
*/

#ifndef LTC2945_H
#define LTC2945_H

#include <Wire.h>

//! Use table to select address
/*!
| LTC2945 I2C Address Assignment    | Value |   AD1    |   AD2    |
| :-------------------------------- | :---: | :------: | :------: |
| LTC2945_I2C_ADDRESS               | 0x67  |   High   |   Low    |
| LTC2945_I2C_ADDRESS               | 0x68  |   Float  |   High   |
| LTC2945_I2C_ADDRESS               | 0x69  |   High   |   High   |
| LTC2945_I2C_ADDRESS               | 0x6A  | Float    |   Float  |
| LTC2945_I2C_ADDRESS               | 0x6B  | Float    |   Low    |
| LTC2945_I2C_ADDRESS               | 0x6C  | Low      |   High   |
| LTC2945_I2C_ADDRESS               | 0x6D  | High     |   Float  |
| LTC2945_I2C_ADDRESS               | 0x6E  | Low      |   Float  |
| LTC2945_I2C_ADDRESS               | 0x6F  | Low      |   Low    |
|                                   |       |          |          |
| LTC2945_I2C_MASS_WRITE            | 0xCC  | X        | X        |
| LTC2945_I2C_ALERT_RESPONSE        | 0x19  | X        | X        |
*/
/*! @name LTC2945 I2C Address Assignments
@{ */

// Address Choices:
// To choose an address, comment out all options except the
// configuration on the demo board.

//  Address assignment
// LTC2945 I2C Address                 //  AD1       AD0
// #define LTC2945_I2C_ADDRESS 0x67    //  High      Low
// #define LTC2945_I2C_ADDRESS 0x68    //  Float     High
// #define LTC2945_I2C_ADDRESS 0x69    //  High      High
// #define LTC2945_I2C_ADDRESS 0x6A    //  Float     Float
// #define LTC2945_I2C_ADDRESS 0x6B    //  Float     Low
// #define LTC2945_I2C_ADDRESS 0x6C    //  Low       High
// #define LTC2945_I2C_ADDRESS 0x6D    //  High      Float
// #define LTC2945_I2C_ADDRESS 0x6E    //  Low       Float
#define LTC2945_I2C_ADDRESS 0x6F       //  Low       Low

#define LTC2945_I2C_MASS_WRITE      0xCC
#define LTC2945_I2C_ALERT_RESPONSE  0x19
//! @}


/*!
| Name                                              | Value |
| :------------------------------------------------ | :---: |
| LTC2945_CONTROL_REG                               | 0x00  |
| LTC2945_ALERT_REG                                 | 0x01  |
| LTC2945_STATUS_REG                                | 0x02  |
| LTC2945_FAULT_REG                                 | 0x03  |
| LTC2945_FAULT_CoR_REG                             | 0x04  |
| LTC2945_POWER_MSB2_REG                            | 0x05  |
| LTC2945_POWER_MSB1_REG                            | 0x06  |
| LTC2945_POWER_LSB_REG                             | 0x07  |
| LTC2945_MAX_POWER_MSB2_REG                        | 0x08  |
| LTC2945_MAX_POWER_MSB1_REG                        | 0x09  |
| LTC2945_MAX_POWER_LSB_REG                         | 0x0A  |
| LTC2945_MIN_POWER_MSB2_REG                        | 0x0B  |
| LTC2945_MIN_POWER_MSB1_REG                        | 0x0C  |
| LTC2945_MIN_POWER_LSB_REG                         | 0x0D  |
| LTC2945_MAX_POWER_THRESHOLD_MSB2_REG              | 0x0E  |
| LTC2945_MAX_POWER_THRESHOLD_MSB1_REG              | 0x0F  |
| LTC2945_MAX_POWER_THRESHOLD_LSB_REG               | 0x10  |
| LTC2945_MIN_POWER_THRESHOLD_MSB2_REG              | 0x11  |
| LTC2945_MIN_POWER_THRESHOLD_MSB1_REG              | 0x12  |
| LTC2945_MIN_POWER_THRESHOLD_LSB_REG               | 0x13  |
| LTC2945_DELTA_SENSE_MSB_REG                       | 0x14  |
| LTC2945_DELTA_SENSE_LSB_REG                       | 0x15  |
| LTC2945_MAX_DELTA_SENSE_MSB_REG                   | 0x16  |
| LTC2945_MAX_DELTA_SENSE_LSB_REG                   | 0x17  |
| LTC2945_MIN_DELTA_SENSE_MSB_REG                   | 0x18  |
| LTC2945_MIN_DELTA_SENSE_LSB_REG                   | 0x19  |
| LTC2945_MAX_DELTA_SENSE_THRESHOLD_MSB_REG         | 0x1A  |
| LTC2945_MAX_DELTA_SENSE_THRESHOLD_LSB_REG         | 0x1B  |
| LTC2945_MIN_DELTA_SENSE_THRESHOLD_MSB_REG         | 0x1C  |
| LTC2945_MIN_DELTA_SENSE_THRESHOLD_LSB_REG         | 0x1D  |
| LTC2945_VIN_MSB_REG                               | 0x1E  |
| LTC2945_VIN_LSB_REG                               | 0x1F  |
| LTC2945_MAX_VIN_MSB_REG                           | 0x20  |
| LTC2945_MAX_VIN_LSB_REG                           | 0x21  |
| LTC2945_MIN_VIN_MSB_REG                           | 0x22  |
| LTC2945_MIN_VIN_LSB_REG                           | 0x23  |
| LTC2945_MAX_VIN_THRESHOLD_MSB_REG                 | 0x24  |
| LTC2945_MAX_VIN_THRESHOLD_LSB_REG                 | 0x25  |
| LTC2945_MIN_VIN_THRESHOLD_MSB_REG                 | 0x26  |
| LTC2945_MIN_VIN_THRESHOLD_LSB_REG                 | 0x27  |
| LTC2945_ADIN_MSB_REG                              | 0x28  |
| LTC2945_ADIN_LSB_REG_REG                          | 0x29  |
| LTC2945_MAX_ADIN_MSB_REG                          | 0x2A  |
| LTC2945_MAX_ADIN_LSB_REG                          | 0x2B  |
| LTC2945_MIN_ADIN_MSB_REG                          | 0x2C  |
| LTC2945_MIN_ADIN_LSB_REG                          | 0x2D  |
| LTC2945_MAX_ADIN_THRESHOLD_MSB_REG                | 0x2E  |
| LTC2945_MAX_ADIN_THRESHOLD_LSB_REG                | 0x2F  |
| LTC2945_MIN_ADIN_THRESHOLD_MSB_REG                | 0x30  |
| LTC2945_MIN_ADIN_THRESHOLD_LSB_REG                | 0x31  |
*/



/*! @name Registers
@{ */
// Registers

#define LTC2945_CONTROL_REG                         0x00
#define LTC2945_ALERT_REG                           0x01
#define LTC2945_STATUS_REG                          0x02
#define LTC2945_FAULT_REG                           0x03
#define LTC2945_FAULT_CoR_REG                       0x04

#define LTC2945_POWER_MSB2_REG                      0x05
#define LTC2945_POWER_MSB1_REG                      0x06
#define LTC2945_POWER_LSB_REG                       0x07
#define LTC2945_MAX_POWER_MSB2_REG                  0x08
#define LTC2945_MAX_POWER_MSB1_REG                  0x09
#define LTC2945_MAX_POWER_LSB_REG                   0x0A
#define LTC2945_MIN_POWER_MSB2_REG                  0x0B
#define LTC2945_MIN_POWER_MSB1_REG                  0x0C
#define LTC2945_MIN_POWER_LSB_REG                   0x0D
#define LTC2945_MAX_POWER_THRESHOLD_MSB2_REG        0x0E
#define LTC2945_MAX_POWER_THRESHOLD_MSB1_REG        0x0F
#define LTC2945_MAX_POWER_THRESHOLD_LSB_REG         0x10
#define LTC2945_MIN_POWER_THRESHOLD_MSB2_REG        0x11
#define LTC2945_MIN_POWER_THRESHOLD_MSB1_REG        0x12
#define LTC2945_MIN_POWER_THRESHOLD_LSB_REG         0x13

#define LTC2945_DELTA_SENSE_MSB_REG                 0x14
#define LTC2945_DELTA_SENSE_LSB_REG                 0x15
#define LTC2945_MAX_DELTA_SENSE_MSB_REG             0x16
#define LTC2945_MAX_DELTA_SENSE_LSB_REG             0x17
#define LTC2945_MIN_DELTA_SENSE_MSB_REG             0x18
#define LTC2945_MIN_DELTA_SENSE_LSB_REG             0x19
#define LTC2945_MAX_DELTA_SENSE_THRESHOLD_MSB_REG   0x1A
#define LTC2945_MAX_DELTA_SENSE_THRESHOLD_LSB_REG   0x1B
#define LTC2945_MIN_DELTA_SENSE_THRESHOLD_MSB_REG   0x1C
#define LTC2945_MIN_DELTA_SENSE_THRESHOLD_LSB_REG   0x1D

#define LTC2945_VIN_MSB_REG                         0x1E
#define LTC2945_VIN_LSB_REG                         0x1F
#define LTC2945_MAX_VIN_MSB_REG                     0x20
#define LTC2945_MAX_VIN_LSB_REG                     0x21
#define LTC2945_MIN_VIN_MSB_REG                     0x22
#define LTC2945_MIN_VIN_LSB_REG                     0x23
#define LTC2945_MAX_VIN_THRESHOLD_MSB_REG           0x24
#define LTC2945_MAX_VIN_THRESHOLD_LSB_REG           0x25
#define LTC2945_MIN_VIN_THRESHOLD_MSB_REG           0x26
#define LTC2945_MIN_VIN_THRESHOLD_LSB_REG           0x27

#define LTC2945_ADIN_MSB_REG                        0x28
#define LTC2945_ADIN_LSB_REG_REG                    0x29
#define LTC2945_MAX_ADIN_MSB_REG                    0x2A
#define LTC2945_MAX_ADIN_LSB_REG                    0x2B
#define LTC2945_MIN_ADIN_MSB_REG                    0x2C
#define LTC2945_MIN_ADIN_LSB_REG                    0x2D
#define LTC2945_MAX_ADIN_THRESHOLD_MSB_REG          0x2E
#define LTC2945_MAX_ADIN_THRESHOLD_LSB_REG          0x2F
#define LTC2945_MIN_ADIN_THRESHOLD_MSB_REG          0x30
#define LTC2945_MIN_ADIN_THRESHOLD_LSB_REG          0x31
//! @}

/*!
| Snapshot Selection Command                    | Value |
| :-------------------------------------------- | :---: |
| LTC2945_DELTA_SENSE_SNAPSHOT                  | 0x80  |
| LTC2945_VIN_SNAPSHOT                          | 0xA0  |
| LTC2945_ADIN_SNAPSHOT                         | 0xC0  |
*/

/*! @name Selection Command
@{ */
// Snapshot Selection Command
#define LTC2945_DELTA_SENSE_SNAPSHOT            0x80
#define LTC2945_VIN_SNAPSHOT                    0xA0
#define LTC2945_ADIN_SNAPSHOT                   0xC0
//! @}

/*!
| Command Codes                                 | Value     |
| :-------------------------------------------- | :-------: |
| LTC2945_CONTINUOUS_MODE                       | 0x00      |
| LTC2945_SHUTDOWN_MODE                         | 0x02      |
| LTC2945_TEST_MODE                             | 0x10      |
| LTC2945_RESET_ALL_FAULTS                      | 0x00      |
| LTC2945_SENSE_MULTIPLIER                      | 0x01      |
| LTC2945_SENSE_MONITOR                         | 0x04      |
| LTC2945_MAX_POWER_MSB2_RESET                  | 0x00      |
| LTC2945_MIN_POWER_MSB2_RESET                  | 0xFF      |
| LTC2945_MAX_DELTA_SENSE_MSB_RESET             | 0x00      |
| LTC2945_MIN_DELTA_SENSE_MSB_RESET             | 0xFF      |
| LTC2945_MAX_VIN_MSB_RESET                     | 0x00      |
| LTC2945_MIN_VIN_MSB_RESET                     | 0xFF      |
| LTC2945_MAX_ADIN_MSB_RESET                    | 0x00      |
| LTC2945_MIN_ADIN_MSB_RESET                    | 0xFF      |
| LTC2945_ENABLE_MAX_POWER_ALERT                | 0x80      |
| LTC2945_ENABLE_MIN_POWER_ALERT                | 0x40      |
| LTC2945_DISABLE_MAX_POWER_ALERT               | 0x7F      |
| LTC2945_DISABLE_MIN_POWER_ALERT               | 0xBF      |
| LTC2945_ENABLE_MAX_DELTA_SENSE_ALERT          | 0x20      |
| LTC2945_ENABLE_MIN_DELTA_SENSE_ALERT          | 0x10      |
| LTC2945_DISABLE_MAX_DELTA_SENSE_ALERT         | 0xDF      |
| LTC2945_DISABLE_MIN_DELTA_SENSE_ALERT         | 0xEF      |
| LTC2945_ENABLE_MAX_VIN_ALERT                  | 0x08      |
| LTC2945_ENABLE_MIN_VIN_ALERT                  | 0x04      |
| LTC2945_DISABLE_MAX_VIN_ALERT                 | 0xF7      |
| LTC2945_DISABLE_MIN_VIN_ALERT                 | 0xFB      |
| LTC2945_ENABLE_MAX_ADIN_ALERT                 | 0x02      |
| LTC2945_ENABLE_MIN_ADIN_ALERT                 | 0x01      |
| LTC2945_DISABLE_MAX_ADIN_ALERT                | 0xFD      |
| LTC2945_DISABLE_MIN_ADIN_ALERT                | 0xFE      |
*/
/*! @name Command Codes
@{ */
// Command Codes
#define LTC2945_CONTINUOUS_MODE                 0x00
#define LTC2945_SHUTDOWN_MODE                   0x02
#define LTC2945_TEST_MODE                       0x10
#define LTC2945_RESET_ALL_FAULTS                0x00
#define LTC2945_SENSE_MULTIPLIER                0x01
#define LTC2945_SENSE_MONITOR                   0x04

#define LTC2945_MAX_POWER_MSB2_RESET            0x00
#define LTC2945_MIN_POWER_MSB2_RESET            0xFF
#define LTC2945_MAX_DELTA_SENSE_MSB_RESET       0x00
#define LTC2945_MIN_DELTA_SENSE_MSB_RESET       0xFF
#define LTC2945_MAX_VIN_MSB_RESET               0x00
#define LTC2945_MIN_VIN_MSB_RESET               0xFF
#define LTC2945_MAX_ADIN_MSB_RESET              0x00
#define LTC2945_MIN_ADIN_MSB_RESET              0xFF

#define LTC2945_ENABLE_MAX_POWER_ALERT          0x80
#define LTC2945_ENABLE_MIN_POWER_ALERT          0x40
#define LTC2945_DISABLE_MAX_POWER_ALERT         0x7F
#define LTC2945_DISABLE_MIN_POWER_ALERT         0xBF

#define LTC2945_ENABLE_MAX_DELTA_SENSE_ALERT    0x20
#define LTC2945_ENABLE_MIN_DELTA_SENSE_ALERT    0x10
#define LTC2945_DISABLE_MAX_DELTA_SENSE_ALERT   0xDF
#define LTC2945_DISABLE_MIN_DELTA_SENSE_ALERT   0xEF

#define LTC2945_ENABLE_MAX_VIN_ALERT            0x08
#define LTC2945_ENABLE_MIN_VIN_ALERT            0x04
#define LTC2945_DISABLE_MAX_VIN_ALERT           0xF7
#define LTC2945_DISABLE_MIN_VIN_ALERT           0xFB

#define LTC2945_ENABLE_MAX_ADIN_ALERT           0x02
#define LTC2945_ENABLE_MIN_ADIN_ALERT           0x01
#define LTC2945_DISABLE_MAX_ADIN_ALERT          0xFD
#define LTC2945_DISABLE_MIN_ADIN_ALERT          0xFE
//! @}

//! Write an 8-bit code to the LTC2945.
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_write(uint8_t i2c_address, //!< Register address for the LTC2945
                     uint8_t adc_command, //!< The "command byte" for the LTC2945
                     uint8_t code         //!< Value that will be written to the register.
                    );
//! Write a 16-bit code to the LTC2945.
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_write_16_bits(uint8_t i2c_address, //!< Register address for the LTC2945
                             uint8_t adc_command, //!< The "command byte" for the LTC2945
                             uint16_t code        //!< Value that will be written to the register.
                            );

//! Write a 24-bit code to the LTC2945.
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_write_24_bits(uint8_t i2c_address, //!< Register address for the LTC2945
                             uint8_t adc_command, //!< The "command byte" for the LTC2945
                             int32_t code         //!< Value that will be written to the register.
                            );
//! Reads an 8-bit adc_code from LTC2945
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_read(uint8_t i2c_address, //!< Register address for the LTC2945
                    uint8_t adc_command, //!< The "command byte" for the LTC2945
                    uint8_t *adc_code    //!< Value that will be read from the register.
                   );
//! Reads a 12-bit adc_code from LTC2945
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_read_12_bits(uint8_t i2c_address, //!< Register address for the LTC2945
                            uint8_t adc_command, //!< The "command byte" for the LTC2945
                            uint16_t *adc_code   //!< Value that will be read from the register.
                           );
//! Reads a 16-bit adc_code from LTC2945
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_read_16_bits(uint8_t i2c_address, //!< Register address for the LTC2945
                            uint8_t adc_command, //!< The "command byte" for the LTC2945
                            uint16_t *adc_code   //!< Value that will be read from the register.
                           );
//! Reads a 24-bit adc_code from LTC2945
//! @return The function returns the state of the acknowledge bit after the I2C address write. 0=acknowledge, 1=no acknowledge.
int8_t LTC2945_read_24_bits(uint8_t i2c_address, //!< Register address for the LTC2945
                            uint8_t adc_command, //!< The "command byte" for the LTC2945
                            int32_t *adc_code    //!< Value that will be read from the register.
                           );
//! Calculate the LTC2945 VIN voltage
//! @return Returns the VIN Voltage in Volts
float LTC2945_VIN_code_to_voltage(uint16_t adc_code,        //!< The ADC value
                                  float LTC2945_VIN_lsb     //!< VIN lsb weight
                                 );
//! Calculate the LTC2945 ADIN voltage
//! @return Returns the ADIN Voltage in Volts
float LTC2945_ADIN_code_to_voltage(uint16_t adc_code,       //!< The ADC value
                                   float LTC2945_ADIN_lsb   //!< ADIN lsb weight
                                  );
//! Calculate the LTC2945 current with a sense resistor
//! @return The LTC2945 current
float LTC2945_code_to_current(uint16_t adc_code,                //!< The ADC value
                              float resistor,                   //!< The resistor value
                              float LTC2945_DELTA_SENSE_lsb     //!< Delta sense lsb weight
                             );
//! Calculate the LTC2945 power
//! @return The LTC2945 power
float LTC2945_code_to_power(int32_t adc_code,           //!< The ADC value
                            float resistor,             //!< The resistor value
                            float LTC2945_Power_lsb     //!< Power lsb weight
                           );
//! Calculate the LTC2945 power with the ADIN
//! @return The LTC2945 power with the ADIN
float LTC2945_code_to_ADIN_power(int32_t adc_code,                      //!< The ADC value
                                 float resistor,                        //!< The resistor value
                                 float LTC2945_ADIN_DELTA_SENSE_lsb     //!< Power lsb weight
                                );
#endif  // LTC2945_H
