classdef SysObj_AD7403 < matlab.System & ...
                         matlab.system.mixin.Propagates & ...
                         matlab.system.mixin.CustomIcon
    % System Object behavioral model for AD7403
          
    properties (Nontunable,Constant)
        % Model Path
        ModelPath = 'MOTIF/AD7403.pmf';
    end
    
    properties (Nontunable)
        % Sampling Clock Frequency
        Mode = 'Modulator';
        Fclk = 20e6;
    end
    
    properties(Constant, Hidden)
        ModeSet = matlab.system.StringSet({'Modulator', 'Average'});
    end
    
    properties (Access = private)
        pKey = 0;   % key of model instance
        pIO;        % IO Object
    end
    
    properties (DiscreteState)
    end
    
    methods
        % Constructor
        function obj = SysObj_AD7403(varargin)
            % Support name-value pair arguments when constructing the
            % object.
            
            % Add MOTIF path
            modelPath = get_param(gcs,'FileName');
            modelFolder = fileparts(modelPath);
            resourcesFolder = fullfile(modelFolder, 'MOTIF');
            addpath(resourcesFolder);
            
            setProperties(obj,nargin,varargin{:});
        end
    end

    methods (Static, Access = protected)
        function header = getHeaderImpl
            header = matlab.system.display.Header(mfilename('class'),...                
                'Title','System Object for AD7403',...
                'Text','This is a behavioral model of the AD7403 modulator.  The output is a single bit stream.',...
                'ShowSourceLink',false);
        end
        
%         function groups = getPropertyGroupsImpl
%             valueGroup = matlab.system.display.Section(...
%             thresholdGroup = matlab.system.display.Section(...
%             mainGroup = matlab.system.display.SectionGroup(...
%             initGroup = matlab.system.display.SectionGroup(...
%             groups = [mainGroup,initGroup];
%         end       
    end
    
    methods (Access = protected)
        %% Common functions
        function setupImpl(obj)
            % Implement tasks that need to be performed only once,
            % such as pre-computed constants.
            obj.pKey = ImportPMF(obj.ModelPath);
            if (obj.pKey == 0)
                % TODO: Throw error
                return
            end
            
            % Fetch parameters from parent block mask
            if strcmp(obj.Mode, 'Modulator')
                modeID = 'mode1';
            else
                modeID = 'mode2';
            end
            SetMode(modeID, obj.pKey);
            
            SetProperty('GLOBAL', 'fclk', num2str(obj.Fclk), obj.pKey);
                        
            obj.pIO = QueryInterface(obj.pKey);
        end
        
        function y = stepImpl(obj,u,clk)
            % Implement algorithm. Calculate y as a function of
            % input u and discrete states.
            y = RunSamples(u,obj.pIO,obj.pKey);
        end
        
%         function resetImpl(obj)
%             % Initialize discrete-state properties.
%             coder.extrinsic('Reset');
%             Reset(obj.pKey);
%         end
        
        function releaseImpl(obj)
            % Initialize discrete-state properties.
            Destroy(obj.pKey);
            %DestroyAll();
        end
        
        function icon = getIconImpl(~)
            icon = sprintf('AD7403');
        end
        
        function processTunedPropertiesImpl(obj)
            % Generate a lookup table of note frequencies
            %SetProperty('GLOBAL','fclk',obj.Fclk,obj.pKey);
        end       
        
%         %% Backup/restore functions
%         function s = saveObjectImpl(obj)
%             % Save private, protected, or state properties in a
%             % structure s.
%         end
        
%         function loadObjectImpl(obj,s,wasLocked)
%             % Read private, protected, or state properties from
%             % the structure s and assign it to the object obj.
%         end
        
        %% Advanced functions
%         function validateInputsImpl(obj,u)
%             % Validate inputs to the step method at initialization.
%             
%         end
        
%         function z = getDiscreteStateImpl(obj)
%             % Return structure of states with field names as
%             % DiscreteState properties.
%             z = struct([]);
%         end
        
%         function processTunedPropertiesImpl(obj)
%             % Define actions to perform when one or more tunable property
%             % values change.
%         end
        
%         function flag = isInputSizeLockedImpl(obj,index)
%             % Set true when the input size is allowed to change while the
%             % system is running.
%             flag = false;
%         end
     
%         function flag = isInactivePropertyImpl(obj,propertyName)
%             % Implement logic for making public properties invisible based on
%             % object configuration, for the command line and block dialog.
%             if strcmp(propertyName,'ModelPath')
%                 flag = true;
%             else
%                 flag = false;
%             end            
%         end
     
        function dataout = getOutputDataTypeImpl(~)
            dataout = 'double';
        end

        function sizeout = getOutputSizeImpl(~)
            sizeout = [1 1];
        end

        function cplxout = isOutputComplexImpl(~)
            cplxout = false;
        end

        function fixedout = isOutputFixedSizeImpl(~)
            fixedout = true;
        end
        
        function num = getNumInputsImpl(obj)
           num = 2; 
        end
        
        function varargout = getInputNamesImpl(obj)
            numInputs = getNumInputs(obj);
            varargout = cell(1,numInputs);
            varargout{1} = 'in (V)';
            if numInputs > 1
                varargout{2} = 'MCLKIN';
            end
            %varargout = cell(1,1);
            %varargout{1} = 'in (V)';
            %varargout{2} = 'MCLKIN';
        end        
        
        function varargout = getOutputNamesImpl(~)
            varargout = cell(1,1);
            varargout{1} = 'out (Bit)';
        end    
    end
end
