/***********************************************************************


 Author        : Johnson Jiao            Johnson.Jiao@analog.com

 Date          : 1st April. 2008

 File          : main.c

 Hardware      : Applicable to ADuC702x rev H or I silicon
                 Currently targetting ADuC7026.

 Description   : It is used as IAP boot function.
 				 After power on, program wait for upgrading command
				 from host-computer software for 5 seconds
				 If receiving this flag from UART, it will execute
				 upgrade steps. If not, jump 0x00081000 user application
				 area to execute application.


  				 1.	full package
				 |count|package|
				 count: 0xFF
				 package: 256 bytes

				 2. no-full package
				 |count|package|
				 count: lower than 0xFF
				 package: lower than 256 bytes

				 Host-Computer Software ----> UART------> SRAM
				 ----> Flash(0x00081000 ------ 0x0008F7FF)


************************************************************************/
#include <ADuC7026.h>
#include "Flash.h"
#include "IRQ_FIQ.h"
#include "UART.h"
#include "main.h"


#define RX_Flag_UPGRADE 0xA		// if receive 0xA, then send 0xBB, and start upgrade

unsigned int UserAppPTR = 0x00081040;
unsigned short AppStartAddr = 0x1000;				// equal to 0x00081000

unsigned char RX_Flag = 0x0; 	 // RX_Flag = 0x0 means no receive rx flag
								 // RX_Flag = 0xA means receive rx flag

unsigned char RX_End=0x0;			//	RX_end=0: continue to receive
									// 	RX_end=1: end

unsigned char timeout = 0x0;			// 0 means no timeout of 5 sec
									// 0xAA mean timeout 

unsigned char FullPackage = 0x2;		// FullPackage = 0; receive a full package of 256 bytes
									// FullPackage = 1; receive non-full package which lower than 256 bytes
									// FullPackage = 2; wait for a full package to arrive
unsigned int Content_start = 0x1;
unsigned char Package[257]={0};		// receive package content
         

void InitTimer(void);				// init timer for 5 sec
void Build_INT_Vector_Table(void);
void (*UserAPP)(void);
void UpgradeApp(void);
                   

void main(void)
{
	// configures GPIO to flash LED P4.2
	GP4DAT = 0x04000000;			// P4.2 configured as an output. LED is turned on	
	InitUART();						// Baudrate of 115200
	InitTimer();					// Init timer1 for 5 second timeout
	FIQEN = UART_BIT + GP_TIMER_BIT;  // Enable UART and Timer1 interrupt in FIQ mode		 
	FEEMOD = 0x8;				// bit 3 should be set to allow erase/write command
	
	// wait for RX flag to upgrade or timeout to jump app
    while(1)
	{
		if(timeout == 0x00)
		{
			if(RX_Flag==RX_Flag_UPGRADE)
			{
				FIQEN = 0x0;			// disable all IRQs                          	
				Build_INT_Vector_Table();                    
				REMAP = 0x1;           
				UpgradeApp();			// send a byte of 0x89 to let host-computer to start
										// After upgrade, use software reset to reset ADuC702x
			}
		}
		else
		{
		// Jumpe to user application code area 0x00081000
			FIQEN = 0x0;					// disable all IRQs
			UserAPP = (void (*)(void))UserAppPTR;
			UserAPP();
		}
	}                                      

}

/****************************************************************************/
//
//	 Some functions as below
//
/****************************************************************************/
void InitTimer()
{
	//	40960000 / 32768 * 0x186A =5sec
	T1LD = 0x186A;
	T1CON = 0xCF;						// Enable, Periodic, Binary and CLK/32768
}

void Build_INT_Vector_Table()
{
	volatile unsigned long *Des_VEC, *SOU_VEC;
	int i;
  // Interrupt Vector Table in SRAM
	Des_VEC	= (unsigned long * )0x10000;
  // Interrupt Vector Table in Flash, please refer to Startup.s
  //     AREA   STARTUPCODE, CODE, AT 0x00080000
	SOU_VEC	= (unsigned long * )0x80000;

  // Copy the interrupt Vector Table from Flash to SRAM
  // for 16 instructions
	for (i = 0; i < 16; i++)
	{
		*Des_VEC = *SOU_VEC;
		*Des_VEC++;
		*SOU_VEC++;
	}
}


void UpgradeApp() __ram
{
	unsigned int i;
	unsigned short temp;
	// erase flash from 0x00081000 to 0x0008F7FF
	for(i = 0x1000; i < 0xF800; i += 0x200)
	{
		erase_page(i);
	}
	IRQEN = UART_BIT;
	senddata(0xB);							// send 0xB to host-computer to  handshake
	
	while(RX_End!=0x1)
	{
		if(FullPackage==0x0)				// receive a 256 bytes content
		{
			for(i=0;i<256;i+=2)
			{
				temp = Package[i+1] + (Package[i+2]<<8);
	 			save(AppStartAddr,temp);
	 			AppStartAddr += 2;
			}
			Package[0] = 0x0;				// clear counter value
			senddata(0xB);					// send handshake signal
			FullPackage=0x2;
		}
		else if(FullPackage==0x1)	 		// receive content lower than 256 bytes
											// then end receiving
		{
			for(i=0;i<Package[0];i+=2)
			{
				temp = Package[i+1] + (Package[i+2]<<8);
	 			save(AppStartAddr,temp);
	 			AppStartAddr += 2;
			}
			senddata(0xA);					// send handshake signal to indicate finish!!!
			RX_End = 0x1;
			FullPackage=0x2;
		}
		else
		{}
	}
	GP4DAT = 0x04040000;			// LED is turned off, indicate Upgrade Finish!!!
	i=0x1000;
	while(i>0)i--;					// short delay!
	RSTSTA = 0x04;					// software reset
	RSTSTA = 0x04;					// software reset
	while(1);		
}

