/********************************************************************************
 Author : CAC (China Applications Support Team) 

 Date :   May, 2014

 File name :  ADXL362.c 

 Description :	 ADXL362 SPI communication driver                

 Hardware plateform : 	EVAL-ADuCM360MKZ and EVAL-ADXL362Z 
 
 Connection:
                 EVAL-ADuCM360MKZ       EVAL-ADXL362Z 
                               
                 P1.4:MISO,             MISO             
                 P1.5:SCLK,             SCLK                                                             
                 P1.6:MOSI,             MOSI
                 P1.7:GPIO as CS,       CS  
********************************************************************************/

#include <ADuCM360.h>

#include "UrtLib.h"
#include "ClkLib.h"
#include "WdtLib.h"
#include "IntLib.h"
#include "DioLib.h"
#include "SpiLib.h"

#include "ADXL362.h"
#include "ADuCM360Driver.h"

/*******************************************************************
  @brief unsigned char ADXL362RegisterRead(unsigned char Address)
         Read a register value from ADXL362         
  @param
         unsigned char Address:       Register address
  @return   
         unsigned int  ReceiveValue:  read register value from ADXL362
*******************************************************************/
unsigned char ADXL362RegisterRead(unsigned char Address)
{   
    unsigned char SendTemp[3];
    unsigned char ReceiveTemp[3];
    unsigned char ReceiveValue;
 
    DioClr(pADI_GP1,0x80);              //CS down  
    SendTemp[0] = 0x0B;                 //0x0B: read register command
    SendTemp[1] = Address;              //address byte
    SendTemp[2] = 0;  
    SpiFunction(SendTemp, ReceiveTemp, 3);
    ReceiveValue = ReceiveTemp[2];   
    DioSet(pADI_GP1,0x80);
    return(ReceiveValue);               //CS up
}

/*******************************************************************
  @brief void ADXL362RegisterWrite(unsigned char Address, unsigned char SendValue)
         send SPI command to ADXL362
  @param
         unsigned char Address:       Register address
         unsigned char SendValue:     Value written to ADXL362 register
  @return   
         none
*******************************************************************/
void ADXL362RegisterWrite(unsigned char Address, unsigned char SendValue)
{    
    unsigned char SendTemp[3];
    unsigned char ReceiveTemp[3];
    
    DioClr(pADI_GP1,0x80);              //CS down
    SendTemp[0] = 0x0A;                 //0x0A: write register
    SendTemp[1] = Address;              //address byte
    SendTemp[2] = SendValue;
    
    SpiFunction(SendTemp, ReceiveTemp, 3);
    DioSet(pADI_GP1,0x80);              //CS up
}

/*******************************************************************
  @brief void ADXL362BurstRead(unsigned char Address, unsigned char NumberofRegisters, unsigned char *RegisterData)
         Multibyte read from ADXL362
  @param
         unsigned char Address:           Register address
         unsigned char NumberofRegisters: Register numbers to be read
         unsigned char *RegisterData:     Buffer save the read value
  @return   
         none  
*******************************************************************/
void ADXL362BurstRead(unsigned char Address, unsigned char NumberofRegisters, unsigned char *RegisterData)
{    
    unsigned char SendTemp[2];
    unsigned char ReceiveTemp[2];
    unsigned char RegisterIndex;
  
    DioClr(pADI_GP1,0x80);         //CS down
    SendTemp[0] = 0x0B;            //0x0B: read register  
    SendTemp[1] = Address;         //address byte  
    SpiFunction(SendTemp, ReceiveTemp, 2);
    SpiFifoFlush(pADI_SPI0,SPICON_TFLUSH_EN,SPICON_RFLUSH_EN);
  
    SendTemp[0] = 0x00;
    for (RegisterIndex=0; RegisterIndex<NumberofRegisters; RegisterIndex++)
    {    
        SpiFifoFlush(pADI_SPI0,SPICON_TFLUSH_EN,SPICON_RFLUSH_EN);
        SpiFunction(SendTemp, ReceiveTemp, 1);
        *(RegisterData + RegisterIndex) = ReceiveTemp[0];        
    }
    DioSet(pADI_GP1,0x80);         //CS up
}

/*******************************************************************
  @brief void ADXL362BurstWrite(unsigned char Address, unsigned char NumberofRegisters, unsigned char *RegisterData)
         Multibyte write to ADXL362
  @param
         unsigned char Address:           Register address
         unsigned char NumberofRegisters: Register numbers to be written
         unsigned char *RegisterData:     Buffer save the written value
  @return   
         none 
*******************************************************************/
void ADXL362BurstWrite(unsigned char Address, unsigned char NumberofRegisters, unsigned char *RegisterData)
{ 
    unsigned char SendTemp[2];
    unsigned char ReceiveTemp[2];
    unsigned char RegisterIndex;
  
    DioClr(pADI_GP1,0x80);              //CS down
    SendTemp[0] = 0x0A;                 //0x0A: write register
    SendTemp[1] = Address;              //address byte  
    SpiFunction(SendTemp, ReceiveTemp, 2);

    for (RegisterIndex=0; RegisterIndex<NumberofRegisters; RegisterIndex++)
    {    
        SendTemp[0] = *(RegisterData + RegisterIndex);
        SpiFunction(SendTemp, ReceiveTemp, 1);       
    }    
    DioSet(pADI_GP1,0x80);              //CS up
}

