/***************************************************************************

 Author        : ADI - Apps                    www.analog.com/MicroConverter

 Date          : January 2007
 
 Revision		0.1
 Revision		0.2	February 2007

 File          : I2C_Master7020.c

 Hardware      : Applicable to ADuC702x rev H or I silicon
                 Currently targetting ADuC7020.

 Description   : I2C master	to demonstrate with I2C_Slave.c
				 
				 Using the command window based executable, I2CSerial.exe, data bytes
are sent serially to the Master via UART0. Depending on whether	the application sent
a write or read command, the Master sends the appropriate command to the Slave via 
I2C channel 1 (I2C1). If a slave read is requested, the Master will request the data
from the I2Cserial.exe specified address and return the data to the PC.


Revision history
	Revision		0.2	February 2007 - Added support for 128 byte reads
	Revision		0.3	October 2011 - Corrected IRQ handler
		
		
***************************************************************************/

#include<ADuC7020.h>				   

void delay(int);
//void IRQ_Handler() __irq;
void Uart_Packet_Handler( volatile unsigned char value, volatile unsigned char Byte_Number);
unsigned char I2CWritePacket( unsigned char length);
unsigned char I2CReadPacket( unsigned char length);

#define count 0x4;					// Number of bytes to be received - 1
int i = 0;
volatile unsigned char rxDat[256];	  // 256 byte buffer-array for I2C receive data
volatile unsigned char ucRxCount = 0; // rxDat array control variable.
volatile unsigned char ucComRx = 0;   // Variable used to read UART Rx buffer contents into 
volatile unsigned char ucMsgType = 0; // Indicates I2CSerial.exe application requested I2C read or write
volatile unsigned char ucLength = 0;  // Indicates I2CSerial.exe "number of bytes" for I2C read or write
volatile unsigned char ucStartAddress = 0;	// Indicates I2CSerial.exe "Start address" for I2C read or write
volatile unsigned char ucSendI2CPacket = 0;	// Used in main loop to indicate I2C message is ready for transmission
volatile unsigned char ucTransfer[256];	 // Buffer used during I2C "writes to Slave"	
volatile unsigned char ucRdCnt = 0;	 // Counter for UART receive messages
volatile unsigned char ucDataCnt = 0; // Message Parser counter for UART messages
volatile unsigned char ucCOMSTA0 = 0; // Variable used to store COMSTA0, UART status register
volatile unsigned char ucCOMIID0 = 0; // Variable used to store COMIID0, UART Interrupt status register	 
unsigned int uiIRQSTA = 0;			  // Variable used to store IRQSTA, UART interrupt status register
volatile unsigned char ucTxBufferEmpty = 0;	// Indicates UART Tx buffer empty
volatile unsigned char ucTransmitInProgress =0; // Indicates I2C Tx FIFO ready for data
volatile unsigned char ucI2CWrCnt = 0; // Counter for controlling I2C writes
volatile unsigned char ucI2CNT = 0;	   // The 4 following variables handle I2C reads of greater than 8 bytes
volatile unsigned char ucI2CNTReload = 0;
volatile unsigned char ucNumReloads =0;	
volatile unsigned char ucRemainder =0;	
volatile unsigned char ucValidUARTPacket = 0;  // Indicates a valid UART packet has been received.
volatile unsigned char ucRdLength = 0;	   // Stores number of bytes to be read from Slave via I2C
unsigned char ucBytesToRD = 0;			// Stores number of bytes to be read from Slave via I2C
volatile unsigned char ucToggleCnt = 0;
volatile unsigned char ucStringCnt = 0;
volatile unsigned char ucLoadVal = 0;
volatile unsigned char ucLoadCnt = 0;
volatile unsigned char szI2C1CNT[] = {5,7,6,5,
									  4,3,2,7,
									  6,5,4,3,
									  2,7,6,5,
									  4,3,2,7,
									  6,5,4,3,
									  2,7,6,5,
									  4,3,2,7,};
volatile unsigned char szI2C1VAL[] = {0xE,0x13,0x17,0x1E,
									  0x22,0x24,0x26,0x2A, 
									  0x2F,0x34,0x35,0x39,
									  0x3b,0x3d,0x3F,0x46,
									  0x4A,0x4c,0x4e,0x52,
									  0x57,0x5c,0x5d,0x61,
									  0x63,0x65,0x67,0x6e,
									  0x72,0x74,0x77,0x7a,
									  0x7F};
int main()
{
	ucValidUARTPacket = 0; 	
	GP4DAT = 0x04000000;			// P4.2 configured as an output. LED is turned on	
	GP4DAT ^= 0x00040000;		// Complement P4.2
	POWKEY1 = 0x01;				// Set Core clock frequency to 41.78Mhz
	POWCON = 0x00;
	POWKEY2 = 0xF4;	 
	delay(100);
 	GP1CON = 0x2211;	//UART on P1.0/P1.1, I2C on P1.3 and P1.2
	
	// Start setting up UART at 9600bps
	COMCON0 = 0x080;			// Setting DLAB
	COMDIV0 = 0x088;			// Setting DIV0 and DIV1 to DL calculated
	COMDIV1 = 0x000;
	COMCON0 = 0x007;			// Clearing DLAB
	COMIEN0 = 0x3;				// Enable interrupt on byte received and transmit buffer empty
	IRQEN = 0x4000;

	//Configure I2C Master
	I2C1CFG = 0x82;		  			// Master Enable & Enable Generation of Master Clock
	GP4DAT = 0x1C000000;
	I2C1CCNT = 0xFF;
	// I2C-Master setup
 	I2C1DIV = 0xCFCF;				// 0x283C = 400kHz
//	I2C1DIV = 0x283C;								// 0xCFCF = 100kHz	
	IRQEN |= 0x800;					// I2C1 Master Interupt

	while (1)
	{
	
		if (ucSendI2CPacket == 1)
		{
			if (ucMsgType == 0)
			{
			 	I2CWritePacket(ucLength);
				ucSendI2CPacket = 0;
				ucDataCnt = 0;
				ucRdCnt = 0;
			}
			else
			{
				I2CReadPacket(ucLength);
				ucSendI2CPacket = 0;
				ucDataCnt = 0;
				ucRdCnt = 0;
				ucLoadCnt = 0;
			}
		}	
	};
 	return 0;
}




void delay (int length)
{
	while (length >0)
    	length--;
}



void Uart_Packet_Handler( volatile unsigned char value, volatile unsigned char Byte_Number)
{
  switch(Byte_Number)
  {
	   case 1:	//Read/Write indicator
	   	{
			ucMsgType = value;
			break;
		} 
	   case 2:  //	Length of data in packet
	   {
	   		ucLength = value;
			break;
	   }	   
	   case 6 : // Memory Start Address
	   {
	   		ucStartAddress = value;
			break;
	   }
	   case 9 : // Memory Start Address
	   {
	   		if (ucMsgType == 1)	// if msg type is a read - reset buffer
			{
				ucDataCnt = 0;	// Reset message parse register
				ucSendI2CPacket = 1;  //Send Packet to the slave
				ucRdCnt = 0;
				ucValidUARTPacket = 0;
			}
			break;
	   }
		default :
	   break;
	}
}

unsigned char I2CWritePacket(unsigned char length )
{

	// Transmit
	ucTransmitInProgress = 1;
	I2C1ADR = 	0xA0;	 			// set i2c address	 (LSB = 0, Master Write)
	I2C1MTX =   ucStartAddress;				// send i2c byte - register Address 0x7

	ucI2CWrCnt = 0;
	while (ucTransmitInProgress == 1)
	{}
	
	return 1;
}

unsigned char I2CReadPacket( unsigned char length)
{
	unsigned char n, ucRdChars = 0;

	ucRdLength = ucLength;
	ucBytesToRD =  ucLength;
	if (length > 8)	// do we need to reload I2C1CNT register?
	{
		ucRdChars = 7;
		ucI2CNTReload = 1;
	}
	else
	{
	   ucRdChars = (length -1);
	   ucI2CNTReload = 0;
	}
	 
	// Transmit
	ucTransmitInProgress = 1;

	ucLength = 0;
	I2C1ADR = 	0xA0;	 			// set i2c address	 (LSB = 0, Master Write)
	I2C1MTX =   ucStartAddress;				// send i2c byte - register Address 0x7

	while ((I2C1FSTA & 0x30) != 0x00)
	{} 	
	// Receive
	
	I2C1CNT = 	ucRdChars;				// Number of bytes to be read from slave
	I2C1ADR = 	0xA1;	 			// set i2c address	 (LSB = 1, Master Read)	
	ucTransmitInProgress = 1;

	while (ucTransmitInProgress == 1) {}
	 
	delay (0x1200000);
    I2C1CFG &= 0xFFFD;    // Switch I2C Master off/on to flush FIFO buffers
    I2C1CFG |= 0x0002;
	ucRdLength = 0;
	ucDataCnt = 0;
	ucRdCnt = 0;
	ucTxBufferEmpty = 0;
	COMTX = rxDat[0];
	ucSendI2CPacket = 0;
	for (n = 1; n < ucBytesToRD +1 ; n++)
	{
		while (ucTxBufferEmpty != 1){}
		ucTxBufferEmpty = 0;
		COMTX = rxDat[n];
	}
	return 1;
}
/*************************************************/
/*************************************************/
/************	IRQ Service Routine  *************/
/*************************************************/
/*************************************************/

void IRQ_Handler() __irq
{
   static unsigned int uiI2C1MSTA = 0;
   static unsigned int uiTemp = 0;
   

	uiI2C1MSTA = I2C1MSTA;
	uiIRQSTA = IRQSTA;
	if ((uiIRQSTA & 0x800) == 0x800) // Check for I2C1 Master interrupt
	{
		// Transmit
		if((uiI2C1MSTA & 0x4) == 0x4)  	// Master Transmit IRQ	
		{											    
			if (ucI2CWrCnt < ucLength)
				I2C1MTX = ucTransfer[ucI2CWrCnt++];	// Tx FIFO ready for another byte
			else
			{
				ucTransmitInProgress = 0;
				ucRxCount = 0; 
				ucI2CNT = 0;
			}
		}			
		// Receive
		if((uiI2C1MSTA & 0x8) == 0x8)				    // Master Recieve IRQ
		{
			rxDat[ucRxCount++] = I2C1MRX;
			ucI2CNT++;					
			if (ucRxCount == 7 ) 
			{
				I2C1CNT = 6;
				ucI2CNT = 2;
				ucLoadVal = ucRxCount + I2C1CNT;
				ucToggleCnt = 1;
				ucLoadVal = szI2C1VAL[0];							
			}
			if (ucRxCount > 9) 
			{	
				if (ucLoadVal == ucRxCount)
				{	
					I2C1CNT	= szI2C1CNT[ucLoadCnt++];
					ucLoadVal = szI2C1VAL[ucLoadCnt];
				}
			}

			if (ucRdLength <= ucRxCount)
			{
			   ucTransmitInProgress = 0;
			   I2C1CNT = 0;
			}
		}		
	}
	if ((uiIRQSTA & 0x4000) == 0x4000)  // Check UART bit
	{
	 	ucCOMSTA0 = COMSTA0;
		ucCOMIID0 = COMIID0;
		if ((ucCOMIID0 & 0x2) == 0x2)	  // Transmit buffer empty
		{
		  ucTxBufferEmpty = 1;
		}
		
		if ((ucCOMIID0 & 0x4) == 0x4)	  // Receive byte
		{
			ucComRx	= COMRX;
			if 	((ucRdCnt == 0) && (ucComRx == 8))	 // Received Start character 0x08
				ucValidUARTPacket = 1;
			if ((ucRdCnt < 10) && (ucSendI2CPacket == 0) && (ucValidUARTPacket == 1))
			{					
				Uart_Packet_Handler( ucComRx, ucRdCnt);
				ucRdCnt++;
			}
			else if ((ucSendI2CPacket == 0) && (ucValidUARTPacket == 1))
			{
				ucTransfer[ucDataCnt++] = COMRX;				 
				if (ucDataCnt == ucLength)
				{
			  		ucDataCnt = 0;	// Reset message parse register
					ucSendI2CPacket = 1;  //Send Packet to the slave
					ucValidUARTPacket = 0;
				}
			}
		}
	   
	}		
}
