//**************************************************************
// File         : coefRTD.cpp
// Author       : Grayson King, Analog Devices
// Last Revised : 6 August 2004
// Compiler     : Borland "free C++Builder" (freeware)
// Hardware     : DOS/Windows PC
// Description  : Generates coefficient lookup table for
//                piecewise linearization of platinum RTDs.
// More Info    : Details in application note AN-709, available
//                at....  http://www.analog.com/MicroConverter
//**************************************************************

// include files....
#include<iostream.h>
#include<fstream.h>
#include<math.h>

// definitions....
#define A (3.9083E-3)   // "A" term of RTD transfer function
#define B (-5.775E-7)   // "B" term of RTD transfer function
#define C (-4.183E-12)  // "C" term of RTD transfer function

// global variables....
int NSEG;     // number of linear sections in table
double TMIN;  // minimum temperature
double TMAX;  // maximum temperature
double RMIN;  // minimum resistance
double RMAX;  // maximum resistance
double R0;    // resistance at 0degC
double RSEG;  // resistance of each linear segment
int i;        // general purpose indexing variable
float C_rtd[256];  // table of coefficients
float C_temp[256]; // temporary storage for table of coefficients
float AA,BB;  // coefficients for single linear approximation

// _____________________________________________________________
// Resistance of RTD function                              R_rtd

double R_rtd (double t) {
  double r;
  // if t < 0, use negative RTD transfer function
  if (t<0) r = R0*(1+A*t+B*pow(t,2)+C*(t-100)*pow(t,3));
  // if t >= 0, use positive RTD transfer function
  else     r = R0*(1+A*t+B*pow(t,2));
  return (r);
}

// _____________________________________________________________
// Temperature of RTD function                             T_rtd

double T_rtd (double r) {
  double t;
  double rtst;
  double tindex;
  // solve for temperature iteratively, to 9 decimal places precision
  tindex = (TMAX-TMIN)/4;
  t = (TMAX+TMIN)/2;
  rtst = R_rtd(t);
  while (0.999999999*r>rtst || rtst>1.000000001*r) {
    if (rtst<r)  t+=tindex;
    if (rtst>r)  t-=tindex;
    tindex/=2;
    rtst = R_rtd(t);
  }
  return (t);
}

// _____________________________________________________________
// Main Program =========================================== MAIN

int main () {
  char answer;
  int i_test;
  double t_test, r_test, t_result, t_error;
  double minerr, maxerr;

  // get input parameters from user

  cout << "\n";
  cout << "_______________________________________\n";
  cout << "COEFFICIENT GENERATOR FOR PLATINUM RTDs\n";
  cout << "September 2003, Analog Devices Inc.\n\n";
  do {
    cout << "enter nominal resistance of RTD at 0degC 'R0' in ohms: ";
    cin >> R0;
  } while (R0<=0);
  do {
    cout << "enter minimum temperature 'TMIN' in degC (-200..+850): ";
    cin >> TMIN;
  } while (TMIN<-200||TMIN>850);
  do {
    cout << "enter maximum temperature 'TMAX' in degC (-200..+850): ";
    cin >> TMAX;
  } while (TMAX<=TMIN||TMAX>850);
  do {
    cout << "enter table size (i.e. # of linear sections) (1..255): ";
    cin >> NSEG;
  } while (NSEG<1||NSEG>255);

  RMIN = R_rtd(TMIN);
  RMAX = R_rtd(TMAX);
  RSEG = (RMAX-RMIN)/NSEG;

  // generate lookup table of idealized points

  cout << "\n";
  cout << "generating lookup table..";
  for (i=0;i<=NSEG;i++) {
    C_rtd[i]=T_rtd(RMIN+RSEG*i);
  }
  cout << ".done.\n";

  // optimize lookup table to make errors more symetrical about zero

  cout << "optimizing lookup table..";
  for (i=0;i<=NSEG;i++) {
    minerr = maxerr = 0;
    for (r_test=RMIN+RSEG*(i==0 ? 0 : i-0.5);r_test<=RMIN+RSEG*(i==NSEG ? i : i+0.5);r_test+=RSEG/1000) {
      t_test = T_rtd(r_test);
      i_test = (r_test-RMIN)/RSEG;
      t_result = C_rtd[i_test]+(C_rtd[i_test+1]-C_rtd[i_test])*(r_test-(RMIN+RSEG*i_test))/RSEG;
      t_error = t_result-t_test;
      if (t_error<minerr)  minerr=t_error;
      if (t_error>maxerr)  maxerr=t_error;
    }
    C_temp[i]=C_rtd[i]-(minerr+maxerr)/2;
  }
  for (i=0;i<=NSEG;i++) {
    C_rtd[i]=C_temp[i];
  }
  cout << ".done.\n";

  // determine resulting overall error band (for high & low ranges)

  cout << "determining overall resultant error band..";
  minerr = maxerr = 0;
  for (t_test=TMIN;t_test<=TMAX;t_test+=0.1) {
    r_test = R_rtd(t_test);
    i_test = (r_test-RMIN)/RSEG;
    t_result = C_rtd[i_test]+(C_rtd[i_test+1]-C_rtd[i_test])*(r_test-(RMIN+RSEG*i_test))/RSEG;
    t_error = t_result-t_test;
    if (t_error<minerr)  minerr=t_error;
    if (t_error>maxerr)  maxerr=t_error;
  }
  cout << ".done.\n\n";

  // display initial results..

  if (NSEG==1) {  // .. for single linear approximation (table size = 1)
    AA = (C_rtd[1]-C_rtd[0])/(RMAX-RMIN);
    BB = C_rtd[0]-RMIN*AA;
    cout << "single linear approximation.\n";
    cout << "linearization routine error band: " << minerr << "degC .. " << maxerr << "degC\n";
    cout << "overall equation:  t = Ar + B\n";
    cout << "where:  A = " << AA << "degC/ohm\n";
    cout << "        B = " << BB << "degC\n";
    cout << "        r = RTD resistance in ohms\n";
    cout << "        t = RTD temperature in degC\n\n";
  }
  else {          // .. for piecewise linear approximation (table size > 1)
    cout << "piecewise linear approximation.\n";
    cout << "linearization routine error band: " << minerr << "degC .. " << maxerr << "degC\n";
    cout << "lookup table size: \n";
    cout << "  = " << NSEG << " linear sections\n";
    cout << "  = " << NSEG+1 << " coefficients\n";
    cout << "  = " << (NSEG+1)*4 << " bytes (4 bytes per floating point coefficient)\n\n";
  }

  // generate C source code

  if (NSEG==1) {  // .. for single linear approximation (table size = 1)
    cout << "generate C source code 'RTDlin.c' (y/n)?: ";
    cin >> answer;
    if (answer=='y'||answer=='Y') {
      cout << "generating 'RTDlin.c'...";
      ofstream coeffile("RTDlin.c");
      coeffile << "//**************************************************************\n";
      coeffile << "// File         : RTDlin.c\n";
      coeffile << "// Author       : Automatically generated using 'coefRTD.exe'\n";
      coeffile << "// Compiler     : intended for Keil C51\n";
      coeffile << "// Description  : Subroutines for linearization of RTD signals\n";
      coeffile << "//                using single linear approximation method.\n";
      coeffile << "// More Info    : Details in application note AN-709, available\n";
      coeffile << "//                at....  http://www.analog.com/MicroConverter\n";
      coeffile << "//**************************************************************\n";
      coeffile << "\n";
      coeffile << "// definitons....\n";
      coeffile << "#define TMIN (" << TMIN << ")  // = minimum temperature in degC\n";
      coeffile << "#define TMAX (" << TMAX << ")  // = maximum temperature in degC\n";
      coeffile << "#define RMIN (" << RMIN << ")  // = input resistance in ohms at " << TMIN << " degC\n";
      coeffile << "#define RMAX (" << RMAX << ")  // = input resistance in ohms at " << TMAX << " degC\n";
      coeffile << "\n";
      coeffile << "// single linear approximation coefficients....\n";
      coeffile << "#define A (" << AA << ")\n";
      coeffile << "#define B (" << BB << ")\n";
      coeffile << "\n";
      coeffile << "// linearization routine error band:  \n";
      coeffile << "//   = " << minerr << "degC .. " << maxerr << "degC\n";
      coeffile << "// specified over measurement range " << TMIN << "degC .. " << TMAX << "degC\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Temperature of RTD Function                             T_rtd\n";
      coeffile << "// input: r = resistance of RTD\n";
      coeffile << "// output: T_rtd() = corresponding temperature of RTD\n";
      coeffile << "// Calculates temperature of RTD as a function of resistance via\n";
      coeffile << "// a single linear approximation method.\n";
      coeffile << "\n";
      coeffile << "float T_rtd (float r) {\n";
      coeffile << "  float t;\n";
      coeffile << "\n";
      coeffile << "  // compute temperature\n";
      coeffile << "  t = A*r + B;\n";
      coeffile << "\n";
      coeffile << "  return (t);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Resistance of RTD Function                              R_rtd\n";
      coeffile << "// input: t = temperature of RTD\n";
      coeffile << "// output: R_rtd() = corresponding resistance of RTD\n";
      coeffile << "// Calculates resistance of RTD as a function of temperature via\n";
      coeffile << "// a single linear approximation method.\n";
      coeffile << "\n";
      coeffile << "float R_rtd (float t) {\n";
      coeffile << "  float r;\n";
      coeffile << "\n";
      coeffile << "  // compute resistance\n";
      coeffile << "  r = (t-B)/A;\n";
      coeffile << "\n";
      coeffile << "  return (r);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Minimum Temperature Function                         Tmin_rtd\n";
      coeffile << "// Returns minimum temperature specified by lookup table.\n";
      coeffile << "float Tmin_rtd () {\n";
      coeffile << "  return (TMIN);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Maximum Temperature Function                         Tmax_rtd\n";
      coeffile << "// Returns maximum temperature specified by lookup table.\n";
      coeffile << "float Tmax_rtd () {\n";
      coeffile << "  return (TMAX);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Minimum Resistance Function                          Rmin_rtd\n";
      coeffile << "// Returns minimum RTD resistance specified by lookup table.\n";
      coeffile << "float Rmin_rtd () {\n";
      coeffile << "  return (RMIN);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Maximum Resistance Function                          Rmax_rtd\n";
      coeffile << "// Returns maximum RTD resistance specified by lookup table.\n";
      coeffile << "float Rmax_rtd () {\n";
      coeffile << "  return (RMAX);\n";
      coeffile << "}\n";
      coeffile.close();
      cout << "done\n";
    }
  }
  else {          // .. for piecewise linear approximation (table size > 1)
    cout << "generate C source code 'RTDpwl.c' (y/n)?: ";
    cin >> answer;
    if (answer=='y'||answer=='Y') {
      cout << "generating 'RTDpwl.c'...";
      ofstream coeffile("RTDpwl.c");
      coeffile << "//**************************************************************\n";
      coeffile << "// File         : RTDpwl.c\n";
      coeffile << "// Author       : Automatically generated using 'coefRTD.exe'\n";
      coeffile << "// Compiler     : intended for Keil C51\n";
      coeffile << "// Description  : Subroutines for linearization of RTD signals\n";
      coeffile << "//                using piecewise linear approximation method.\n";
      coeffile << "// More Info    : Details in application note AN-709, available\n";
      coeffile << "//                at....  http://www.analog.com/MicroConverter\n";
      coeffile << "//**************************************************************\n";
      coeffile << "\n";
      coeffile << "// definitons....\n";
      coeffile << "#define TMIN (" << TMIN << ")  // = minimum temperature in degC\n";
      coeffile << "#define TMAX (" << TMAX << ")  // = maximum temperature in degC\n";
      coeffile << "#define RMIN (" << RMIN << ")  // = input resistance in ohms at " << TMIN << " degC\n";
      coeffile << "#define RMAX (" << RMAX << ")  // = input resistance in ohms at " << TMAX << " degC\n";
      coeffile << "#define NSEG " << NSEG << "  // = number of sections in table\n";
      coeffile << "#define RSEG " << RSEG << "  // = (RMAX-RMIN)/NSEG = resistance RSEG in ohms of each segment\n";
      coeffile << "\n";
      coeffile << "// lookup table....\n";
      coeffile << "const float code C_rtd[] = {";
      for (i=0;i<NSEG;i++)  coeffile << C_rtd[i] << ",";
      coeffile << C_rtd[i] << "};\n";
      coeffile << "// lookup table size:\n";
      coeffile << "//   = " << NSEG << " linear sections\n";
      coeffile << "//   = " << NSEG+1 << " coefficients\n";
      coeffile << "//   = " << (NSEG+1)*4 << " bytes (4 bytes per floating point coefficient)\n";
      coeffile << "\n";
      coeffile << "// linearization routine error band:  \n";
      coeffile << "//   = " << minerr << "degC .. " << maxerr << "degC\n";
      coeffile << "// specified over measurement range " << TMIN << "degC .. " << TMAX << "degC\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Temperature of RTD Function                             T_rtd\n";
      coeffile << "// input: r = resistance of RTD\n";
      coeffile << "// output: T_rtd() = corresponding temperature of RTD\n";
      coeffile << "// Calculates temperature of RTD as a function of resistance via\n";
      coeffile << "// a piecewise linear approximation method.\n";
      coeffile << "\n";
      coeffile << "float T_rtd (float r) {\n";
      coeffile << "  float t;\n";
      coeffile << "  int i;\n";
      coeffile << "  i=(r-RMIN)/RSEG;       // determine which coefficients to use\n";
      coeffile << "  if (i<0)               // if input is under-range..\n";
      coeffile << "    i=0;                 // ..then use lowest coefficients\n";
      coeffile << "  else if (i>NSEG-1)     // if input is over-range..\n";
      coeffile << "    i=NSEG-1;            // ..then use highest coefficients\n";
      coeffile << "  t = C_rtd[i]+(r-(RMIN+RSEG*i))*(C_rtd[i+1]-C_rtd[i])/RSEG;\n";
      coeffile << "  return (t);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Resistance of RTD Function                              R_rtd\n";
      coeffile << "// input: t = temperature of RTD\n";
      coeffile << "// output: R_rtd() = corresponding resistance of RTD\n";
      coeffile << "// Calculates resistance of RTD as a function of temperature via\n";
      coeffile << "// a piecewise linear approximation method.\n";
      coeffile << "\n";
      coeffile << "float R_rtd (float t) {\n";
      coeffile << "  float r;\n";
      coeffile << "  int i, adder;\n";
      coeffile << "\n";
      coeffile << "  // set up initial values\n";
      coeffile << "  i = NSEG/2;           // starting value for 'i' index\n";
      coeffile << "  adder = (i+1)/2;      // adder value used in do loop\n";
      coeffile << "\n";
      coeffile << "  // determine if input t is within range\n";
      coeffile << "  if (t<C_rtd[0])           // if input is under-range..\n";
      coeffile << "    i=0;                    // ..then use lowest coefficients\n";
      coeffile << "  else if (t>C_rtd[NSEG])   // if input is over-range..\n";
      coeffile << "    i=NSEG-1;               // ..then use highest coefficients\n";
      coeffile << "\n";
      coeffile << "  // if input within range, determine which coefficients to use\n";
      coeffile << "  else do {\n";
      coeffile << "    if (C_rtd[i]>t)   i-=adder; // either decrease i by adder..\n";
      coeffile << "    if (C_rtd[i+1]<t) i+=adder; // ..or increase i by adder\n";
      coeffile << "    if (i<0)       i=0;         // make sure i is >=0..\n";
      coeffile << "    if (i>NSEG-1)  i=NSEG-1;    // ..and <=NSEG-1\n";
      coeffile << "    adder = (adder+1)/2;        // divide adder by two (rounded)\n";
      coeffile << "  } while ((C_rtd[i]>t)||(C_rtd[i+1]<t));   // repeat 'til done\n";
      coeffile << "\n";
      coeffile << "  // compute final result\n";
      coeffile << "  r = RMIN+RSEG*i + (t-C_rtd[i])*RSEG/(C_rtd[i+1]-C_rtd[i]);\n";
      coeffile << "\n";
      coeffile << "  return (r);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Minimum Temperature Function                         Tmin_rtd\n";
      coeffile << "// Returns minimum temperature specified by lookup table.\n";
      coeffile << "float Tmin_rtd () {\n";
      coeffile << "  return (TMIN);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Maximum Temperature Function                         Tmax_rtd\n";
      coeffile << "// Returns maximum temperature specified by lookup table.\n";
      coeffile << "float Tmax_rtd () {\n";
      coeffile << "  return (TMAX);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Minimum Resistance Function                          Rmin_rtd\n";
      coeffile << "// Returns minimum RTD resistance specified by lookup table.\n";
      coeffile << "float Rmin_rtd () {\n";
      coeffile << "  return (RMIN);\n";
      coeffile << "}\n";
      coeffile << "\n";
      coeffile << "// _____________________________________________________________\n";
      coeffile << "// Maximum Resistance Function                          Rmax_rtd\n";
      coeffile << "// Returns maximum RTD resistance specified by lookup table.\n";
      coeffile << "float Rmax_rtd () {\n";
      coeffile << "  return (RMAX);\n";
      coeffile << "}\n";
      coeffile.close();
      cout << "done\n";
    }
  }

  // generate error table file "errRTD.txt";

  cout << "generate error analysis table file 'errRTD.txt' (y/n)?: ";
  cin >> answer;
  if (answer=='y'||answer=='Y') {
    cout << "generating 'errRTD.txt'...";
    ofstream errfile("errRTD.txt");
    errfile << "T_actual\tT_result\tT_error\n";
    for (t_test=TMIN;t_test<=TMAX;t_test+=1) {
      r_test = R_rtd(t_test);
      i_test = (r_test-RMIN)/RSEG;
      t_result = C_rtd[i_test]+(C_rtd[i_test+1]-C_rtd[i_test])*(r_test-(RMIN+RSEG*i_test))/RSEG;
      t_error = t_result-t_test;
      errfile << t_test << "\t" << t_result << "\t" << t_error << "\n";
    }
    errfile.close();
    cout << "done\n";
  }

//  while(1) {}
  return 0;

}
