//**************************************************************
// File         : RTD845.c
// Author       : Grayson King, Analog Devices
// Last Revised : 6 August 2004
// Compiler     : Keil C51
// Hardware     : ADuC845, ADuC847, or ADuC848 with RTD circuit
//                (use eval board or copy RTD circuit on it)
// Description  : Converts platinum RTD signal to digital and
//                performs linearization.  Outputs results as
//                ASCII text via UART at 9600baud.  Accepts user
//                I/O via UART through text menu options for
//                calibration, changing units, etc.
// More Info    : Details in application note AN-709, available
//                at....  http://www.analog.com/MicroConverter
//**************************************************************

// include files....
#include<ADuC845.h>     // ADuC845 special function registers
#include<stdio.h>       // standard i/o functions
#include<intrins.h>     // in-line intrinsic code (for chkfloat)

// definitions....
#define DEFAULTOFFSET0 0     // default offset coefficient (RTD)
#define DEFAULTSCALE0 719.36 // default scale coefficient (RTD)
#define DEFAULTOFFSET1 0     // default offset coefficient (chip)
#define DEFAULTSCALE1 128.0  // default scale coefficient (chip)

// global variable declarations....
float ADC0norm;          // normalized ADC0 conversion result
float ADC1norm;          // normalized ADC1 conversion result
float offset0;           // offset coefficient (RTD)
float scale0;            // scale coefficient (RTD)
float offset1;           // offset coefficient (chip sensor)
float scale1;            // scale coefficient (chip sensor)
float Rprevcal;          // resistance at previous cal point
float ADC0cal;           // normalized ADC0 result for use in calibration
bit fahrenheit;          // 0 for degC, 1 for degF
bit onchip;              // 0 for RTD, 1 for on-chip sensor
bit newADCdata;          // 1 means new ADC data available
unsigned char serial_in; // serial input data

// prototypes of functions local to this source file....
void ADC_isr ();        // ADC interrupt service routine
void UART_isr ();       // UART interrupt service routine
void Temperature ();    // temperature results processing subroutine
void SerialCommand ();  // serial command processing subroutine
void Calibrate (void);  // user calibration subroutine
signed char Cal (float Rcal);  // raw calibration subroutine
void PutFlashFloat (unsigned int addr, float *floatptr); //EEput
void GetFlashFloat (unsigned int addr, float *floatptr); //EEget

// prototypes of functions located in other source files....
float T_rtd (float r);  // RTD transfer function
float R_rtd (float t);  // inverse RTD transfer function
float Tmin_rtd ();      // minimum spec'd rtd temperature
float Tmax_rtd ();      // maximum spec'd rtd temperature
float Rmin_rtd ();      // minimum spec'd rtd resistance
float Rmax_rtd ();      // maximum spec'd rtd resistance
// NOTE: these 6 functions are NOT included in this source file.
// Add to your project an RTD linearization function source file
// (RTDmath.c, RTDlin.c, or RTDpwl.c) which can be generated by
// the coefRTD.exe tool.

// _____________________________________________________________
// Main Program =========================================== MAIN

void main (void) {

  // configure core clock & enable internal XRAM
  PLLCON = 1;           // core clock = 6.291456MHz
  CFG845 = 0x01;        // enable internal XRAM

  // set default values
  offset1 = DEFAULTOFFSET1;
  scale1 = DEFAULTSCALE1;
  fahrenheit = 0;
  onchip = 0;
  serial_in = 0;
  newADCdata = 0;
  ADC0norm = 0;
  ADC1norm = 0;

  // restore cal coefficients from nv flash data memory
  GetFlashFloat (0,&offset0);   // get offset coef from address 0
  GetFlashFloat (1,&scale0);    // get scale coef from address 1
  if (_chkfloat_(offset0)==4 || _chkfloat_(scale0)==4) {
    offset0 = DEFAULTOFFSET0;    // if no cal coefficients exist..
    scale0 = DEFAULTSCALE0;      // ..use default coefficients
  }
  Rprevcal = 0;         // previous cal resistance (no prev.cal)

  // turn on excitation current source
  ICON = 0x0B;          // 400uA out IEXC1 pin (P1.6)
  //ICON = 0x07;          // 400uA out IEXC2 pin (P1.7) [use this pin on eval board when using off-board rtd]

  // configure UART
  T3CON = 0x85;         // configure timer 3...
  T3FD = 0x12;          // ...for 9600 baud
  SCON = 0x52;          // configure UART (using timer 3)

  // configure ADCs & start continuous conversions
  ADC0CON1 = 0x24;      // unipolar, buffered, gain=7.8125
  ADC0CON2 = 0x4A;      // ext.ref, AIN1&2
  ADC1CON = 0x0E;       // int.ref, on-chip temperature sensor
  SF = 0xFF;            // ADC rate = 5.35Hz
  ADCMODE = 0x33;       // continuous conv, both ADCs, chop on

  // enable interrupts, & enter main loop
  IE = 0xD0;            // enable ADC interrupt & UART interrupt
  while(1) {            // main loop
    if (newADCdata) Temperature();   // if new ADC result, display temperature
    if (serial_in)  SerialCommand(); // if serial input received, goto command menu
  }

}

// _____________________________________________________________
// ADC Interrupt Service Routine                        >ADC_isr
// Triggered each time ADC conversion completes.

void ADC_isr () interrupt 6 {
  float ADC0temp, ADC1temp; // temporary locations for normalized ADC results

  // if ADCnorm variables are ready for new data..
  if (newADCdata==0) {
    // ..then put normalized ADC results into ADCtemp variables
    ADC0temp = (ADC0H*0x10000+ADC0M*0x100+ADC0L)/16777216.0;
    ADC1temp = (ADC1H*0x10000+ADC1M*0x100+ADC1L)/8388608.0-1.0;
    // ..and then perform running average with ADC data stored in ADCnorm variables
    ADC0norm = 0.8*ADC0norm + 0.2*ADC0temp;
    ADC1norm = 0.8*ADC1norm + 0.2*ADC1temp;
  }

  // clear ready bits to allow next ADC conversions
  RDY0 = 0;     // primary ADC
  RDY1 = 0;     // auxiliary ADC

  // indicate that ADCnorm variables contain new ADC data
  newADCdata = 1;
}

// _____________________________________________________________
// UART Interrupt Service Routine                      >UART_isr

void UART_isr () interrupt 4 {
  if (RI) {            // if serial byte received..
    RI=0;              // ..then clear RI flag
    serial_in = SBUF;  // ..and put recived byte into the serial_in variable
  }
}

// _____________________________________________________________
// Temperature Results Processing Function          >Temperature
// Triggered each time ADC conversion completes.  Performs
// scaling and calls linearization routine and then outputs
// final temperature results via the chip's UART.

void Temperature () {
  float Rrtd, Trtd;   // resistance & temperature of RTD
  float Tchip;        // temperature of chip

  // determine RTD temperature based on ADC result (& cal coef)
  Rrtd = ADC0norm*scale0+offset0;   // RTD resistance [ohms]
  Trtd = T_rtd(Rrtd);               // RTD temperature [degC]

  // determine chip temperature based on on-chip sensor
  Tchip = ADC1norm*scale1+offset1;  // chip temperature [degC]

  // flag that ADC data has been read out of ADCnorm variables,
  // and that new data can be written into those variables
  newADCdata = 0;

  // output results via UART
  if (onchip) {
    printf ("Chip Temperature = ");
    if (fahrenheit)  printf ("%+8.2fF",(1.8*Tchip+32));
    else             printf ("%+8.2fC",Tchip);
  }
  else {
    printf ("RTD Temperature = ");
    if (fahrenheit)  printf ("%+8.2fF",(1.8*Trtd+32));
    else             printf ("%+8.2fC",Trtd);
    // and indicate error if any
    if (Trtd<Tmin_rtd()) printf (" *ERROR: Under-Range Temperature");
    if (Trtd>Tmax_rtd()) printf (" *ERROR: Over-Range Temperature");
  }
  printf ("\r\n");

  P3 ^= 0x10;   // complement the LED (P3.4 on eval board)

}

// _____________________________________________________________
// Serial Command Processing Function             >SerialCommand

void SerialCommand () {

  // display main menu
  printf("\r\n");
  printf("ADuC845 RTD Interface Program\r\n");
  printf("MAIN MENU....\r\n");
  printf("  R: select RTD sensor\r\n");
  printf("  O: select on-chip sensor\r\n");
  printf("  C: display units in Celcius\r\n");
  printf("  F: display units in Fahrenheit\r\n");
  printf("  X: calibrate\r\n");
  printf("  any other key: resume normal operation\r\n");
  printf("select option: ");

  // get keystroke & perform corresponding action
  IE &= 0xEF;  // first disable serial interrupt
  switch (_getkey()) {
    case 'c' :
    case 'C' : printf ("C");
               fahrenheit = 0;
               break;
    case 'f' :
    case 'F' : printf ("F");
               fahrenheit = 1;
               break;
    case 'o' :
    case 'O' : printf ("O");
               onchip = 1;
               break;
    case 'r' :
    case 'R' : printf ("R");
               onchip = 0;
               break;
    case 'x' :
    case 'X' : printf ("X");
               Calibrate();
               break;
    default  : break;
  }
  IE |= 0x10;  // re-enable serial interrupt
  printf("\r\n\r\n");

  // restore serial_in to zero, ready for next serial interrupt
  serial_in = 0;
}

// _____________________________________________________________
// User Calibration Subroutine                        >Calibrate

void Calibrate () {
  signed char result; // calibration result indicator
  float Rcal, Tcal;   // calibration resistance & temperature
  float Rrtd, Trtd;   // current resistance & temperature of RTD

  // determine RTD temperature based on ADC result & current cal coef
  ADC0cal = ADC0norm;            // capture current ADC0 reading
  Rrtd = ADC0cal*scale0+offset0; // current RTD resistance [ohms]
  Trtd = T_rtd(Rrtd);            // current RTD temperature [degC]

  // display calibration menu
  printf("\r\n\r\n");
  printf("most recent reading: %+8.2fC, %+8.2fohms\r\n", Trtd, Rrtd);
  printf("CALIBRATION MENU....\r\n");
  printf("  T: enter known RTD temperature at most recent reading\r\n");
  printf("  R: enter known RTD resistance at most recent reading\r\n");
  printf("  D: restore default calibration coefficients\r\n");
  printf("  any other key: cancel calibration & resume normal operation\r\n");
  printf("select option: ");

  // get keystroke & perform corresponding action
  IE &= 0xEF;  // first disable serial interrupt
  switch (_getkey()) {
    case 't' :
    case 'T' : printf ("T\r\n");
               printf ("enter RTD temperature [degC]: ");
               scanf ("%f",&Tcal);
               Rcal=R_rtd(Tcal);
               result=Cal(Rcal);
               break;
    case 'r' :
    case 'R' : printf ("R\r\n");
               printf ("enter RTD resistance [ohms]: ");
               scanf ("%f",&Rcal);
               Tcal=T_rtd(Rcal);
               result=Cal(Rcal);
               break;
    case 'd' :
    case 'D' : printf ("D\r\n");
               scale0 = DEFAULTSCALE0;
               offset0 = DEFAULTOFFSET0;
               Rprevcal = 0;
               result = -1;
               break;
    default  : result = -2;
               break;
  }
  IE |= 0x10;  // re-enable serial interrupt

  // if cal coefficients changed, store new ones into flash
  if (result==0||result==-1) {
    PutFlashFloat (0,&offset0);  // offset coef into address 0
    PutFlashFloat (1,&scale0);   // scale coef into address 1
  }

  // indicate results
  switch (result) {
    case -2 :
      break;
    case -1 :
      printf("\r\n");
      printf("RESULT: Default Cal Coefficients Restored");
      break;
    case 0 :
      printf("\r\n");
      printf("RESULT: Calibrated at %fdegC",Tcal);
      break;
    case 1 :
      printf("\r\n");
      printf("RESULT: Out of Range Input, No Cal Performed");
      break;
    case 2 :
      printf("\r\n");
      printf("RESULT: Conflict with Previous Cal, No Cal Performed");
      break;
    default :
      printf("\r\n");
      printf("RESULT: Unknown Calibration Error");
      break;
  }
//  printf("press any key to continue: ");
//  IE &= 0xEF;  // disable serial interrupt
//  _getkey();   // wait for key-press
//  IE |= 0x10;  // re-enable serial interrupt
//  printf("\r\n");
               
}

// _____________________________________________________________
// Raw Calibration Suboutine                                >Cal
// input: Rcal = expected RTD resistance at current temperature
// output: Cal() = error indication (per comments below)
// globals: scale0, offset0, Rprevcal, ADC0cal
// Performs a calibration at current data point while preserving
// cal performed at previous cal point (per "prevcal" globals).

#pragma disable  // disable interrupts during this function

signed char Cal (float Rcal) {
  signed char returnerr;  // to indicate error (0 = no error)
  float ADC0prevcal;      // ADC result at previous cal point

  returnerr=3;  // default error code (indicates program fault)

  // determine ADC value at previous cal point (from Rprevcal)
  ADC0prevcal = (Rprevcal-offset0)/scale0;

  // first check to see if input is out of range or conflicting
  if (Rcal<Rmin_rtd() || Rcal>Rmax_rtd())
    returnerr=1;  // indicate out-of-range Rcal input
  else if (Rcal>=Rprevcal && ADC0cal<=ADC0prevcal)
    returnerr=2;  // indicate conflict with previous cal
  else if (Rcal<=Rprevcal && ADC0cal>=ADC0prevcal)
    returnerr=2;  // indicate conflict with previous cal

  // if everything looks good, go ahead and perform cal
  else {
    // scale0 is essentially "rise-over-run", or deltaR/deltaADC
    scale0=(Rcal-Rprevcal)/(ADC0cal-ADC0prevcal);
    // determine offset using new scale0 value & one known point
    offset0=Rcal-ADC0cal*scale0;
    Rprevcal=Rcal; // we'll need Rprevcal for the next cal point
    returnerr=0;   // indicate succesful calibration
  }

  return (returnerr);
}

// _____________________________________________________________
// Store-Float-into-Flash Subroutine              >PutFlashFloat
// input: addr = page address in flash memory (0 to 0x3FF)
// input: *floatptr = points to floating point value to store
// Stores a floating point value into a 4-byte page of on-chip
// flash data memory.
void PutFlashFloat (unsigned int addr, float *floatptr) {
  // put the 4 bytes of the float value into EDATA SFR bytes
  EDATA1 = *((unsigned char *)floatptr);
  EDATA2 = *((unsigned char *)floatptr+1);
  EDATA3 = *((unsigned char *)floatptr+2);
  EDATA4 = *((unsigned char *)floatptr+3);
  // program EDATA SFR bytes into 4-byte flash data page at addr
  EADRH = *((unsigned char *)&addr);    // address high byte
  EADRL = *((unsigned char *)&addr+1);  // address low byte
  ECON = 5;                             // erase page
  ECON = 2;                             // program page
}

// _____________________________________________________________
// Recal-Float-from-Flash Subroutine              >GetFlashFloat
// input: addr = page address in flash memory (0 to 0x3FF)
// output: *floatptr = points to floating point variable where..
//                     ..fetched result will be stored
// Retrieves a floating point value from a 4-byte page of
// on-chip flash data memory.
void GetFlashFloat (unsigned int addr, float *floatptr) {
  // read EDATA SFR bytes from 4-byte flash data page at addr
  EADRH = *((unsigned char *)&addr);    // address high byte
  EADRL = *((unsigned char *)&addr+1);  // address low byte
  ECON = 1;                             // read page
  // put EDATA SFR bytes into the 4 bytes of the float variable
  *((unsigned char *)floatptr) = EDATA1;
  *((unsigned char *)floatptr+1) = EDATA2;
  *((unsigned char *)floatptr+2) = EDATA3;
  *((unsigned char *)floatptr+3) = EDATA4;
}
